package org.lsst.ccs.utilities.scheduler;

import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Defines behavior of {@link PeriodicTask} under abnormal circumstances.
 * Setters provided by this class can be used to establish policies the periodic task
 * should follow when it throws an exception or runs for an unusually long time.
 * Callback methods {@code onXXX(PeriodicTask task, ...} can be overridden to execute
 * client code under these circumstances. They should be implemented to finish quickly,
 * any time consuming or potentially blocking operations should be offloaded to other
 * threads. Default implementations provided by this class log messages at the level 
 * set through a call to {@code setLevel(...)} or inherited from the {@link Scheduler}
 * where the task runs.
 * <p>
 * <b>Sample Usage.</b> The following code sketch creates a fixed rate task that will
 * skip execution if the previous one has not finished, log a message if an execution
 * is taking longer than the task period, and quit after 2 failures.
 *
 *  <pre> {@code
 * Scheduler scheduler = new Scheduler("tester", 1);
 * ...
 * PeriodicTaskExceptionHandler exceptionHandler = new PeriodicTaskExceptionHandler() {
 *     public void onLongExecution(PeriodicTask task) {
 *         Logger logger = task.getLogger();
 *         if (logger != null) {
 *             logger.log(getLevel(), "Task "+ task.getTaskName() +" is taking more than its period.", (Object[])null);
 *         }
 *         alertOperator();
 *     }
 * };
 * exceptionHandler.setSkipOverdueExecutions(true);
 * exceptionHandler.setDetectLongExecutions(true);
 * exceptionHandler.setMaxFailures(2);
 * exceptionHandler.setResetFailureCountOnSuccess(false);
 * scheduler.scheduleAtFixedRate(command, 0, 1, TimeUnit.SECONDS, "Task name", exceptionHandler);
 * ...
 * }</pre>
 *
 *
 * @author onoprien
 */
public class PeriodicTaskExceptionHandler {

// -- Fields : -----------------------------------------------------------------
    
    private boolean skipOverdueExecutions = false;
    private int longExecutionThreshold = 0;
    private volatile int maxFailures = -1;
    private boolean resetFailureCountOnSuccess = true;
    private Level level;
    
// -- Setters/getters : --------------------------------------------------------

    /**
     * Sets a policy on skipping executions that cannot start on time.
     * This flag is only relevant for fixed rate tasks, and can be set to {@code true}
     * to prevent executions from piling up when a previous execution takes a long time,
     * and then running in rapid succession when it finally finishes. This option  is
     * usually not suitable for task periods less than 100 milliseconds.
     * The default is {@code false}.
     * 
     * @param skipOverdueExecutions If {@code true}, executions that cannot start on time are skipped.
     */
    public void setSkipOverdueExecutions(boolean skipOverdueExecutions) {
        this.skipOverdueExecutions = skipOverdueExecutions;
    }

    /**
     * Sets a policy on whether or not executions that take longer than the task period to complete
     * should be detected and {@code onLongExecution(...)} called. Note that setting this flag to
     * {@code true} adds significant overhead, and is usually not suitable for task periods of less
     * than 100 milliseconds. The default is {@code false}.
     * <p>
     * Calling this method with {@code true} is equivalent to {@code setLongExecutionThreshold(1, null)};
     * Calling this method with {@code false} is equivalent to {@code setLongExecutionThreshold(0, null)};
     * 
     * @param detectLongExecutions {@code true} if long executions should be detected.
     */
    public void setDetectLongExecutions(boolean detectLongExecutions) {
        longExecutionThreshold = detectLongExecutions ? -1 : 0;
    }

    /**
     * Sets a policy on detecting long executions.
     * Executions that take longer than {@code longExecutionThreshold} to complete
     * should be detected and {@code onLongExecution(...)} called. Note that detecting
     * long executions adds significant overhead, and is usually not suitable for task
     * periods of less than 100 milliseconds.
     * <p>
     * If the threshold is set to zero (default), long executions are not detected.
     * If the {@code unit} is {@code null}, the value of {@code longExecutionThreshold}
     * is interpreted as the number of task periods.
     * 
     * @param longExecutionThreshold Threshold.
     * @param unit Init for threshold.
     */
    public void setLongExecutionThreshold(int longExecutionThreshold, TimeUnit unit) {
        if (longExecutionThreshold == 0) {
            this.longExecutionThreshold = 0;
        } else if (unit == null) {
            this.longExecutionThreshold = - longExecutionThreshold;
        } else {
            this.longExecutionThreshold = (int) unit.toMillis(longExecutionThreshold);
        }
    }

    /**
     * Sets the number of times the task can throw an exception before it is terminated.
     * 
     * @param maxFailures Allowed number of failures.
     */
    public void setMaxFailures(int maxFailures) {
        this.maxFailures = maxFailures;
    }

    /**
     * Sets a policy on whether or not only consecutive failures should be counted towards the
     * allowed maximum. The default is {@code true}.
     * 
     * @param resetFailureCountOnSuccess If {@code true}, only consecutive failures are counted towards the allowed maximum.
     */
    public void setResetFailureCountOnSuccess(boolean resetFailureCountOnSuccess) {
        this.resetFailureCountOnSuccess = resetFailureCountOnSuccess;
    }

    /**
     * Sets logging level.
     * @param level Level.
     */
    public void setLevel(Level level) {
        this.level = level;
    }

    public boolean isSkipOverdueExecutions() {
        return skipOverdueExecutions;
    }

    public boolean isDetectLongExecutions() {
        return longExecutionThreshold != 0;
    }

    public int getLongExecutionThreshold() {
        return longExecutionThreshold;
    }

    public int getMaxFailures() {
        return maxFailures;
    }

    public boolean isResetFailureCountOnSuccess() {
        return resetFailureCountOnSuccess;
    }

    public Level getLevel() {
        return level;
    }

// -- Callbacks : --------------------------------------------------------------
    
    /**
     * Called when a fixed rate task executes after skipping one or more previous executions
     * that could not start on time. The default implementation logs a message.
     * 
     * @param task Periodic task.
     * @param nSkipped Number of skipped executions.
     */
    public void onSkippedExecutions(PeriodicTask task, int nSkipped) {
        Logger logger = task.getLogger();
        if (logger != null) {
            StringBuilder sb = new StringBuilder();
            sb.append("Periodic task ").append(task.getTaskName()).append(" skipped ").append(nSkipped).append(" executions.");

            sb.append(" Timing relative to ");
            long t = task.timeBegin;
            sb.append(millisToTime(t)).append(", ms: PREVIOUS: from 0 to ");
            long delta = task.timeEnd - t;
            sb.append(delta);
            if (delta > 1L) {
                sb.append(" (");
                if (task.timeAfterOnSkippedExecutions > 0L) {
                    delta = task.timeBeforeOnSkippedExecutions - t;
                    if (delta > 0) sb.append(" before onSkipped : ").append(delta);
                    delta = task.timeAfterOnSkippedExecutions - task.timeBeforeOnSkippedExecutions;
                    if (delta > 0) sb.append(" onSkipped : ").append(delta);
                    t = task.timeAfterOnSkippedExecutions;
                }
                if (task.timeAfterTimer > 0L) {
                    delta = task.timeBeforeTimer - t;
                    if (delta > 0) sb.append(" before timer : ").append(delta);
                    delta = task.timeAfterTimer - task.timeBeforeTimer;
                    if (delta > 0) sb.append(" timer : ").append(delta);
                    t = task.timeAfterTimer;
                }
                delta = task.timeBeforeRun - t;
                if (delta > 0) sb.append(" before run : ").append(delta);
                delta = task.timeAfterRun - task.timeBeforeRun;
                if (delta > 0) sb.append(" run : ").append(delta);
                t = task.timeAfterRun;
                if (task.timeBeforeCancel > 0L) {
                    delta = task.timeBeforeCancel - t;
                    if (delta > 0) sb.append(" before cancel : ").append(delta);
                    delta = task.timeEnd - task.timeBeforeCancel;
                    if (delta > 0) sb.append(" cancel : ").append(delta);
                } else {
                    delta = task.timeEnd - t;
                    if (delta > 0) sb.append(" after run : ").append(delta);
                }
                sb.append(")");
            }
            t = task.timeFirstSkipped - task.timeBegin;
            sb.append("; FIRST SKIPPED: expected ").append(t + task.delayFirstSkipped).append(", started ").append(t);
            long currentTime = System.currentTimeMillis();
            sb.append("; NOW: ").append(currentTime - task.timeBegin);
            
            try {
                final RuntimeMXBean rt = ManagementFactory.getRuntimeMXBean();
                long base = task.timeBegin - rt.getStartTime();
                sb.append("; GC: ");
                boolean hasGC = false;
                for (GarbageCollectorMXBean gc : ManagementFactory.getGarbageCollectorMXBeans()) {
                    if (gc instanceof com.sun.management.GarbageCollectorMXBean) {
                        com.sun.management.GcInfo info = ((com.sun.management.GarbageCollectorMXBean) gc).getLastGcInfo();
                        if (info != null) {
                            sb.append(gc.getName()).append(" from ").append(info.getStartTime() - base).append(" to ").append(info.getEndTime() - base).append(" for ").append(info.getDuration()).append(", ");
                            hasGC = true;
                        }
                    }
                }
                if (hasGC) sb.delete(sb.length()-2, sb.length());
            } catch (RuntimeException x) {
            }

            sb.append(".");
            logger.log(level, sb.toString(), (Object[])null);
        }
    }
    
    /**
     * Called when an execution that takes longer than the task period is detected.
     * This method is only called if {@code detectLongExecutions} property has been set to {@code true}.
     * The default implementation logs a message.
     * 
     * @param task Periodic task.
     */
    public void onLongExecution(PeriodicTask task) {
        Logger logger = task.getLogger();
        if (logger != null) {
            StringBuilder sb = new StringBuilder();
            sb.append("Periodic task ").append(task.getTaskName()).append(" did not finish execution by the end of its period of ");
            sb.append(task.getPeriod(TimeUnit.MILLISECONDS)).append(" ms. ");
            sb.append(task.getThread() != null ? "Still runnung" : "Finished by now");
            sb.append(". Timing (unreliable!) relative to ");
            long t = task.timeBegin;
            sb.append(millisToTime(t)).append(", ms: started 0, ");
            do { // executed once, breakable sequence
                long delta = 0L;
                if (task.timeBeforeOnSkippedExecutions > 0L) {
                    delta = task.timeBeforeOnSkippedExecutions - t;
                    if (delta > 0) {
                        sb.append(" before onSkipped : ").append(delta);
                    } else if (delta < 0) {
                        sb.append(" before onSkipped : ").append("unfinished");
                        break;
                    }
                    delta = task.timeAfterOnSkippedExecutions - task.timeBeforeOnSkippedExecutions;
                    if (delta > 0) {
                        sb.append(" onSkipped : ").append(delta);
                    } else if (delta < 0) {
                        sb.append(" onSkipped : ").append("unfinished");
                        break;
                    }
                    t = task.timeAfterOnSkippedExecutions;
                }
                if (task.timeBeforeTimer > 0L) {
                    delta = task.timeBeforeTimer - t;
                    if (delta > 0) {
                        sb.append(" before timer : ").append(delta);
                    } else if (delta < 0) {
                        sb.append(" before timer : ").append("unfinished");
                        break;
                    }
                    delta = task.timeAfterTimer - task.timeBeforeTimer;
                    if (delta > 0) {
                        sb.append(" timer : ").append(delta);
                    } else if (delta < 0) {
                        sb.append(" timer : ").append("unfinished");
                        break;
                    }
                    t = task.timeAfterTimer;
                }
                delta = task.timeBeforeRun - t;
                if (delta > 0) {
                    sb.append(" before run : ").append(delta);
                } else if (delta < 0) {
                    sb.append(" before run : ").append("unfinished");
                    break;
                }
                delta = task.timeAfterRun - task.timeBeforeRun;
                if (delta > 0) {
                    sb.append(" run : ").append(delta);
                } else if (delta < 0) {
                    sb.append(" run : ").append("unfinished");
                    break;
                }
                t = task.timeAfterRun;
                if (task.timeBeforeCancel > 0L) {
                    delta = task.timeBeforeCancel - t;
                    if (delta > 0) {
                        sb.append(" before cancel : ").append(delta);
                    } else if (delta < 0) {
                        sb.append(" before cancel : ").append("unfinished");
                        break;
                    }
                    delta = task.timeEnd - task.timeBeforeCancel;
                    if (delta > 0) {
                        sb.append(" cancel : ").append(delta);
                    } else if (delta < 0) {
                        sb.append(" cancel : ").append("unfinished");
                        break;
                    }
                } else {
                    delta = task.timeEnd - t;
                    if (delta > 0) {
                        sb.append(" after run : ").append(delta);
                    } else if (delta < 0) {
                        sb.append(" after run : ").append("unfinished");
                        break;
                    }
                }
            } while (false);
            long currentTime = System.currentTimeMillis();
            sb.append("; NOW: ").append(currentTime - task.timeBegin);
            
            try {
                final RuntimeMXBean rt = ManagementFactory.getRuntimeMXBean();
                long base = task.timeBegin - rt.getStartTime();
                sb.append("; GC: ");
                boolean hasGC = false;
                for (GarbageCollectorMXBean gc : ManagementFactory.getGarbageCollectorMXBeans()) {
                    if (gc instanceof com.sun.management.GarbageCollectorMXBean) {
                        com.sun.management.GcInfo info = ((com.sun.management.GarbageCollectorMXBean) gc).getLastGcInfo();
                        if (info != null) {
                            sb.append(gc.getName()).append(" from ").append(info.getStartTime() - base).append(" to ").append(info.getEndTime() - base).append(" for ").append(info.getDuration()).append(", ");
                            hasGC = true;
                        }
                    }
                }
                if (hasGC) sb.delete(sb.length()-2, sb.length());
            } catch (RuntimeException x) {
            }

            sb.append(".");
            logger.log(level, sb.toString(), (Object[])null);
        }
    }
    
    /**
     * Called when a task throws an exception, but is not going to be terminated.
     * If this method throws an exception, the task will terminate.
     * The default implementation logs a message.
     * 
     * @param task Periodic task.
     * @param exception Thrown exception.
     */
    public void onException(PeriodicTask task, Throwable exception) {
        Logger logger = task.getLogger();
        if (logger != null) {
            logger.log(level, "Exception thrown by periodic task " + task.getTaskName(), exception);
        }
    }
    
    /**
     * Called when a task throws an exception, and is going to be terminated.
     * The default implementation logs a message.
     * 
     * @param task Periodic task.
     * @param exception Thrown exception.
     */
    public void onFinalException(PeriodicTask task, Throwable exception) {
        Logger logger = task.getLogger();
        if (logger != null) {
            logger.log(level, "Exception thrown by periodic task " + task.getTaskName() + ", task terminated.", exception);
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private final DateFormat formatter = new SimpleDateFormat("HH:mm:ss.SSS");
    private String millisToTime(long time) {
//        formatter.setTimeZone(TimeZone.getTimeZone("UTC"));
        return formatter.format(new Date(time));
    }

}
