package org.lsst.ccs.utilities.scheduler;

import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Thread factory to be used by executors that need to customize thread name, group, and daemon status.
 *
 * @author onoprien
 */
public class BasicThreadFactory implements ThreadFactory {

// -- Fields : -----------------------------------------------------------------
    
        private static final AtomicInteger FACTORY_NUMBER = new AtomicInteger(1);
        
        private final ThreadGroup group;
        private final AtomicInteger threadNumber = new AtomicInteger(1);
        private final String namePrefix;
        private final boolean daemon;

        
// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Creates thread factory.
     * 
     * @param name Thread name prefix: threads will be named "name-N".
     *             If {@code null}, threads will be named BasicThreadFactory-K-N".
     * @param group Thread group. If {@code null}, the group of the thread that creates
     *              this factory will be used, unless {@code SecurityManager} exists.
     * @param daemon Whether or not threads made by this factory are daemons.
     */
    BasicThreadFactory(String name, ThreadGroup group, boolean daemon) {
        if (group == null) {
            SecurityManager s = System.getSecurityManager();
            this.group = (s != null) ? s.getThreadGroup() : Thread.currentThread().getThreadGroup();
        } else {
            this.group = group;
        }
        namePrefix = name == null ? "BasicThreadFactory-" + FACTORY_NUMBER.getAndIncrement() + "-" : name + "-";
        this.daemon = false;
    }

    
// -- Making threads : ---------------------------------------------------------
    
    public Thread newThread(Runnable r) {
        Thread t = new Thread(group, r, namePrefix + threadNumber.getAndIncrement(), 0);
        if (t.isDaemon() != daemon) {
            t.setDaemon(daemon);
        }
        if (t.getPriority() != Thread.NORM_PRIORITY) {
            t.setPriority(Thread.NORM_PRIORITY);
        }
        return t;
    }
}
