package org.lsst.ccs.utilities.scheduler;

import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Defines behavior of {@link PeriodicTask} under abnormal circumstances.
 * Setters provided by this class can be used to establish policies the periodic task
 * should follow when it throws an exception or runs for an unusually long time.
 * Callback methods {@code onXXX(PeriodicTask task, ...} can be overridden to execute
 * client code under these circumstances. They should be implemented to finish quickly,
 * any time consuming or potentially blocking operations should be offloaded to other
 * threads. Default implementations provided by this class log messages at the level 
 * set through a call to {@code setLevel(...)} or inherited from the {@link Scheduler}
 * where the task runs.
 * <p>
 * <b>Sample Usage.</b> The following code sketch creates a fixed rate task that will
 * skip execution if the previous one has not finished, log a message if an execution
 * is taking longer than the task period, and quit after 2 failures.
 *
 *  <pre> {@code
 * Scheduler scheduler = new Scheduler("tester", 1);
 * ...
 * PeriodicTaskExceptionHandler exceptionHandler = new PeriodicTaskExceptionHandler() {
 *     public void onLongExecution(PeriodicTask task) {
 *         Logger logger = task.getLogger();
 *         if (logger != null) {
 *             logger.log(getLevel(), "Task "+ task.getTaskName() +" is taking more than its period.", (Object[])null);
 *         }
 *         alertOperator();
 *     }
 * };
 * exceptionHandler.setSkipOverdueExecutions(true);
 * exceptionHandler.setDetectLongExecutions(true);
 * exceptionHandler.setMaxFailures(2);
 * exceptionHandler.setResetFailureCountOnSuccess(false);
 * scheduler.scheduleAtFixedRate(command, 0, 1, TimeUnit.SECONDS, "Task name", exceptionHandler);
 * ...
 * }</pre>
 *
 *
 * @author onoprien
 */
public class PeriodicTaskExceptionHandler {

// -- Fields : -----------------------------------------------------------------
    
    private boolean skipOverdueExecutions = false;
    private boolean detectLongExecutions = false;
    private volatile int maxFailures = -1;
    private boolean resetFailureCountOnSuccess = true;
    private Level level;
    
// -- Setters/getters : --------------------------------------------------------

    /**
     * Sets a policy on skipping executions that cannot start on time.
     * This flag is only relevant for fixed rate tasks, and can be set to {@code true}
     * to prevent executions from piling up when a previous execution takes a long time,
     * and then running in rapid succession when it finally finishes. This option  is
     * usually not suitable for task periods less than 100 milliseconds.
     * The default is {@code false}.
     * 
     * @param skipOverdueExecutions If {@code true}, executions that cannot start on time are skipped.
     */
    public void setSkipOverdueExecutions(boolean skipOverdueExecutions) {
        this.skipOverdueExecutions = skipOverdueExecutions;
    }

    /**
     * Sets a policy on whether or not executions that take longer than the task period to complete
     * should be detected and {@code onLongExecution(...)} called. Note that setting this flag to
     * {@code true} adds significant overhead, and is usually not suitable for task periods less than
     * 100 milliseconds. The default is {@code false}.
     * 
     * @param detectLongExecutions {@code true} if long executions should be detected.
     */
    public void setDetectLongExecutions(boolean detectLongExecutions) {
        this.detectLongExecutions = detectLongExecutions;
    }

    /**
     * Sets the number of times the task can throw an exception before it is terminated.
     * 
     * @param maxFailures Allowed number of failures.
     */
    public void setMaxFailures(int maxFailures) {
        this.maxFailures = maxFailures;
    }

    /**
     * Sets a policy on whether or not only consecutive failures should be counted towards the
     * allowed maximum. The default is {@code true}.
     * 
     * @param resetFailureCountOnSuccess If {@code true}, only consecutive failures are counted towards the allowed maximum.
     */
    public void setResetFailureCountOnSuccess(boolean resetFailureCountOnSuccess) {
        this.resetFailureCountOnSuccess = resetFailureCountOnSuccess;
    }

    /**
     * Sets logging level.
     * @param level Level.
     */
    public void setLevel(Level level) {
        this.level = level;
    }

    public boolean isSkipOverdueExecutions() {
        return skipOverdueExecutions;
    }

    public boolean isDetectLongExecutions() {
        return detectLongExecutions;
    }

    public int getMaxFailures() {
        return maxFailures;
    }

    public boolean isResetFailureCountOnSuccess() {
        return resetFailureCountOnSuccess;
    }

    public Level getLevel() {
        return level;
    }

// -- Callbacks : --------------------------------------------------------------
    
    /**
     * Called when a fixed rate task executes after skipping one or more previous executions
     * that could not start on time. The default implementation logs a message.
     * 
     * @param task Periodic task.
     * @param nSkipped Number of skipped executions.
     */
    public void onSkippedExecutions(PeriodicTask task, int nSkipped) {
        Logger logger = task.getLogger();
        if (logger != null) {
            StringBuilder sb = new StringBuilder();
            sb.append("Periodic task ").append(task.getTaskName()).append(" skipped ").append(nSkipped).append(" executions.");
            logger.log(level, sb.toString(), (Object[])null);
        }
    }
    
    /**
     * Called when an execution that takes longer than the task period is detected.
     * This method is only called if {@code detectLongExecutions} property has been set to {@code true}.
     * The default implementation logs a message.
     * 
     * @param task Periodic task.
     */
    public void onLongExecution(PeriodicTask task) {
        Logger logger = task.getLogger();
        if (logger != null) {
            StringBuilder sb = new StringBuilder();
            sb.append("Periodic task ").append(task.getTaskName()).append(" did not finish execution by the end of its period of ");
            sb.append(task.getPeriod(TimeUnit.MILLISECONDS)).append(" milliseconds.");
            logger.log(level, sb.toString(), (Object[])null);
        }
    }
    
    /**
     * Called when a task throws an exception, but is not going to be terminated.
     * If this method throws an exception, the task will terminate.
     * The default implementation logs a message.
     * 
     * @param task Periodic task.
     * @param exception Thrown exception.
     */
    public void onException(PeriodicTask task, Throwable exception) {
        Logger logger = task.getLogger();
        if (logger != null) {
            logger.log(level, "Exception thrown by periodic task " + task.getTaskName(), exception);
        }
    }
    
    /**
     * Called when a task throws an exception, and is going to be terminated.
     * The default implementation logs a message.
     * 
     * @param task Periodic task.
     * @param exception Thrown exception.
     */
    public void onFinalException(PeriodicTask task, Throwable exception) {
        Logger logger = task.getLogger();
        if (logger != null) {
            logger.log(level, "Exception thrown by periodic task " + task.getTaskName() + ", task terminated.", exception);
        }
    }

}
