package org.lsst.ccs.utilities.logging;


import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.util.*;
import java.util.logging.*;
import java.util.logging.Formatter;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.utilities.pattern.PatternUtils;


/**
 * The role of this class is:
 * <UL>
*     <LI/> to read the logging.properties and offer other codes (namely handlers) access
 *     to the properties described (a thing not permitted by the standard logManager)
 *     <LI/>  to be sure that the Handlers are lazily loaded  (the standard LogManager may
 *     not be able to load the Handlers class due to ClassLoading problems)
*     <LI/> to delegate informations to the LogManager so it can behave properly (e.g. transmit
 *     logging properties, register Loggers)
* /UL>
 * @ImplSpec
 * Specifications
 * <BR/>
 * Instances of this class must be created during load time by the <TT>LogManagement</TT>
 * the static code must:
 * <UL>
 *     <LI/> create an instance of this class (so there is one instance per ClassLoader)
 *     this instance load the Handlers  if not already done
 * </UL>
 * When operating as a code that delegates to the Top Manager
 * <UL>
 *     <LI/> the methods that deal with handlers creation should avoid duplications.
 *     they should query the corresponding <TT>Logger</TT> from the "top" manager and verify
 *     that the requested Handler is not already registered to this Logger.
 *     <LI/> static services that require information from the logging.properties should
 *     be addressed to  the LogManager
 * </UL>
 * Technical hurdles:
 * <UL>
 *     <LI/>  the "top" manager is the standard manager, there may be many ContextLogManagers
 *     that delegate to it: what happens when we ask the top manager to read again the properties
 *     (do we need to "reset" each time), how to know that the manager has all the needed information
 *     and that it is useless to duplicate configuration?
 *     <BR/> a special property <TT>ccsLoaded</TT>is  used for this purpose
 * </UL>
 *
 *
 */
public class LogPropertiesLoader {
    public static final String LOG_PROPS = "/logging.properties";
    /**
     * if the property to define a config file is set then the corresponding file
     * is used for properties otherwise it is "LOG_PROPS"
     */
    public static final String LOG_CONF_FILE = System.getProperty("java.util.logging.config.file", LOG_PROPS);


    // ------------------------------------------------------------Constructors
    
    Properties loggingProperties ;

    public LogPropertiesLoader() {
        /* */
        readInitialConfiguration();
        /**/ 
    }



    ///////////////// INITIALIZATION METHODS


    /**
     * invoked by constructor (that should be itself invoked during load time).
     * since this method will be called each time a new ClassLoader fires load time and
     * that we decided there should be only one logging context when CCS classes are loaded
     * then the standard LogManager should be initialized once.
     * for this purpose we create a dummy property and we check each time if this property is present.
     */
    void readInitialConfiguration() {
        
        String resourceName =  LOG_CONF_FILE ;

        loggingProperties = new Properties(); 
        Properties props = BootstrapResourceUtils.getBootstrapProperties(resourceName, this.getClass());
        Set<Object> keys = BootstrapResourceUtils.getAllKeysInProperties(props);
        for (Object key : keys) {
            loggingProperties.put(key, props.getProperty((String) key));
        }


        //HERE DO something for files
        Set<String> set = loggingProperties.stringPropertyNames();
        for (String key : set) {
            //TODO: reinforce this code : just ending by .pattern may be prone to bugs
            if (key.endsWith(".pattern")) {
                String pattern = loggingProperties.getProperty(key);
                try {
                    pattern = checkForFile(pattern);
                    loggingProperties.put(key, pattern) ;
                } catch (IOException e) {
                    System.err.println("problem while creating log directory " + e);
                }
                break;
            }
        }

        // Temp code block to facilitate transition between 1.6 and 2.0
        String handlersN = loggingProperties.getProperty("handlersN");
        if (handlersN != null){
           throw new RuntimeException("handlersN property and custom handler are no longer in use. Use 'handler' property and standard handlers");
        }
        // end
        
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        try {
            loggingProperties.store(bos, "");
            byte[] bytes = bos.toByteArray();
            ByteArrayInputStream bis = new ByteArrayInputStream(bytes);
            LogManager.getLogManager().readConfiguration(bis);
        } catch (IOException e) {
            System.err.println(" can't load properties for LogManager" + e);
        }
    }

    /**
     * System property replacement in the given string.
     *
     * @param str The original string
     * @return the modified string
     */
    protected String replace(String str, Properties props) {
        String result = str;
        int pos_start = result.indexOf("${");
        if (pos_start != -1) {
            int pos_end = result.indexOf('}');
            if (pos_end != -1) {
                String propName = result.substring(pos_start + 2, pos_end);
                String replacement = props.getProperty(propName);
                if (replacement != null) {
                    if (pos_start > 0) {
                        result = result.substring(0, pos_start) +
                                replacement + replace(result.substring(pos_end + 1), props);
                    } else {
                        result = replacement + replace(result.substring(pos_end + 1), props);
                    }
                }
            }
        }
        return result;
    }


// ---------------------------------------------------- LogNode Inner Class



    /**
     * read the file pattern until last  separator (this will give the directory)
     * then will try to create missing directories.
     *
     * @param filePattern
     */
    String checkForFile(String filePattern) throws IOException {
        filePattern = PatternUtils.resolvePattern(filePattern);
        int lastSlash = filePattern.lastIndexOf('/');
        if (lastSlash >= 0) {
            String directorySpec = filePattern.substring(0, lastSlash);
            Files.createDirectories(FileSystems.getDefault().getPath(directorySpec));
        }
        return filePattern ;
    }

    ///////////////// END INITIALIZATION
    public static String loaderGetProperty(String name) {        
        try {
            String res = LogManager.getLogManager().getProperty(name) ;
            return res;
        } catch (Exception e) {
            System.err.println(" loaderGetProperty :" + e);
            return null;
        }

    }
    public static String loaderGetStringProperty(String name, String defaultValue) {
        String val = loaderGetProperty(name);
        if (val == null) {
            return defaultValue;
        }
        return val.trim();
    }
    public static int loaderGetIntProperty(String name, int defaultValue) {

        String val = loaderGetProperty(name);
        if (val == null) {
            return defaultValue;
        }
        try {
            return Integer.parseInt(val.trim());
        } catch (Exception ex) {
            return defaultValue;
        }
    }


    public static Level loaderGetLevelProperty(String name, Level defaultValue) {
        String val = loaderGetProperty(name);
        if (val == null) {
            return defaultValue;
        }
        try {
            return Level.parse(val.trim());
        } catch (Exception ex) {
            return defaultValue;
        }

    }






    public static Formatter loaderGetFormatterProperty(String name, Formatter defaultValue) {

        String val = loaderGetProperty(name);
        try {
            if (val != null) {
                //Class clz = ClassLoader.getSystemClassLoader().loadClass(val);
                Class clazz = LogPropertiesLoader.class.getClassLoader().loadClass(val) ;
                return (Formatter) clazz.newInstance();
            }
        } catch (Exception ex) {
            // We got one of a variety of exceptions in creating the
            // class or creating an instance.
            // Drop through.
        }
        // We got an exception.  Return the defaultValue.
        return defaultValue;
    }






    synchronized private void setLevelOnExistingLogger(String loggerName, java.util.logging.Logger logger) {
        Level level = getLevelProperty(loggerName + ".level", null) ;
        //LSSTCCS-296
        if ( level != null ) {
            logger.setLevel(level);
        }
    }

    public String getProperty(String property) {
        return LogManager.getLogManager().getProperty(property) ;
    }

    
    //  method to get a String property.
    // If the property is not defined we return the given
    // default value.
    public String getStringProperty(String name, String defaultValue) {
        String val = getProperty(name);
        if (val == null) {
            return defaultValue;
        }
        return val.trim();
    }


    // If the property is not defined or cannot be parsed
    // we return the given default value.
    public int getIntProperty(String name, int defaultValue) {
        String val = getProperty(name);
        if (val == null) {
            return defaultValue;
        }
        try {
            return Integer.parseInt(val.trim());
        } catch (Exception ex) {
            return defaultValue;
        }
    }


    // If the property is not defined or cannot be parsed
    // we return the given default value.
    public boolean getBooleanProperty(String name, boolean defaultValue) {
        String val = getProperty(name);
        if (val == null) {
            return defaultValue;
        }
        val = val.toLowerCase();
        if (val.equals("true") || val.equals("1")) {
            return true;
        } else if (val.equals("false") || val.equals("0")) {
            return false;
        }
        return defaultValue;
    }

    // we return the given default value.
    public Level getLevelProperty(String name, Level defaultValue) {
        String val = getProperty(name);
        if (val == null) {
            return defaultValue;
        }
        try {
            return Level.parse(val.trim());
        } catch (Exception ex) {
            return defaultValue;
        }
    }


    //  method to get a filter property.
    // We return an instance of the class named by the "name"
    // property. If the property is not defined or has problems
    // we return the defaultValue.
    public Filter getFilterProperty(String name, Filter defaultValue) {
        String val = getProperty(name);
        try {
            if (val != null) {
                Class clz = ClassLoader.getSystemClassLoader().loadClass(val);
                return (Filter) clz.newInstance();
            }
        } catch (Exception ex) {
            // We got one of a variety of exceptions in creating the
            // class or creating an instance.
            // Drop through.
        }
        // We got an exception.  Return the defaultValue.
        return defaultValue;
    }


    //  method to get a formatter property.
    // We return an instance of the class named by the "name"
    // property. If the property is not defined or has problems
    // we return the defaultValue.
    public Formatter getFormatterProperty(String name, Formatter defaultValue) {
        String val = getProperty(name);
        try {
            if (val != null) {
                Class clz = ClassLoader.getSystemClassLoader().loadClass(val);
                return (Formatter) clz.newInstance();
            }
        } catch (Exception ex) {
            // We got one of a variety of exceptions in creating the
            // class or creating an instance.
            // Drop through.
        }
        // We got an exception.  Return the defaultValue.
        return defaultValue;
    }



}





