package org.lsst.ccs.utilities.structs;


import java.io.Serializable;

/**
 * Path of a configuration parameter.
 * @author LSST CCS Team
 */
public class ParameterPath implements Serializable {
    
    private static final long serialVersionUID = -8374146985540286600L;
    
    /**
     * unique name of Module/component in subsystem context
     */
    final String componentName;

    /**
     * name of parameter (may be the number of a positional parameter starting at 0)
     */
    final String parameterName ;
    
    //////////////////////////////////// CONSTRUCTORS

    /**
     * @param componentName should not be null
     * @param parameterName should not be null
     */
    public ParameterPath(String componentName, String parameterName) {
        if(componentName == null || parameterName == null) {
            throw new IllegalArgumentException("null argument in ParameterPath");
        }
        this.componentName = componentName;
        this.parameterName = parameterName;
    }
    
    ////////////////////////////////// ACCESSORS/MUTATORS

    /**
     * Gets the component this parameter belongs to.
     * @return The component name.
     */
    public String getComponentName() {
        return componentName;
    }

    /**
     * Gets the parameter name.
     * @return the parameter name
     */
    public String getParameterName() {
        return parameterName;
    }

    ////////////////////////////// IDENT METHODS

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ParameterPath that = (ParameterPath) o;

        if (!componentName.equals(that.componentName)) return false;
        if (!parameterName.equals(that.parameterName)) return false;

        return true;
    }


    @Override
    public int hashCode() {
        int result = componentName.hashCode();
        result = 31 * result + parameterName.hashCode();
        return result;
    }
    
    @Override
    public String toString() {
        return this.componentName + '/'+this.parameterName ;
    }

    /**
     * reverse operation from toString(): creates a ParameterPath from a String.
     * It supports both old and new configuration parameter formatting.
     * @param pathString should be of the form {@code componentName/parameterName"}
     * where {@code componentPath} could also be a path (example: "carousel/tickMillis", "main/tick/taskPeriodMillis")
     * @return a ParameterPath out of the given path String
     * @throws IllegalArgumentException if {@code pathString} is not a path.
     */
    public static ParameterPath valueOf(String pathString) {
        if (pathString.contains("//")) {
            // old format
            String[] split = pathString.split("//");
            return new ParameterPath(split[0], split[1]);
        } else {
            int lastSep = pathString.lastIndexOf('/');
            if (lastSep < 0 ) {
                throw new IllegalArgumentException(pathString + " is is not a configuration parameter path");
            }
            String parameterName = pathString.substring(lastSep+1);
            String componentName = pathString.substring(0, lastSep);            
            
            return new ParameterPath(componentName, parameterName) ;
        }

    }
}