package org.lsst.ccs.utilities.scheduler;

import java.util.Date;
import java.util.concurrent.Callable;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Customized scheduling executor for running periodic and delayed tasks.
 * Methods that schedule periodic actions return instances of {@link PeriodicTask}
 * that provide control over the task execution: cancellation, change of period,
 * suspension and resumption. It is also possible to provide a logger for reporting
 * uncaught exceptions thrown by periodic tasks.
 *
 * <p><b>Sample Usage.</b> The following code sketch creates a periodic task that 
 * beeps for 10 seconds, changing period after 3 seconds, then pausing for 2 seconds.
 *
 *  <pre> {@code
 * Scheduler scheduler = new Scheduler("tester", 1);
 * ...
 * Runnable run = () -> java.awt.Toolkit.getDefaultToolkit().beep();
 * PeriodicTask task = scheduler.scheduleAtFixedRate(run, 0, 1, TimeUnit.SECONDS, "bipper", Level.SEVERE);
 * scheduler.schedule(() -> task.setPeriod(500, TimeUnit.MILLISECONDS), 3, TimeUnit.SECONDS);
 * scheduler.schedule(() -> task.stop(), 6, TimeUnit.SECONDS);
 * scheduler.schedule(() -> task.start(), 8, TimeUnit.SECONDS);
 * scheduler.schedule(() -> task.cancel(false), 10, TimeUnit.SECONDS);
 * ...
 * scheduler.shutdown();
 * }</pre>
 *
 *
 * @author onoprien
 */
public final class Scheduler {
  
// -- Fields : -----------------------------------------------------------------
    
    final ScheduledThreadPoolExecutor executor;
    volatile Logger log;
    volatile int maxFailures = 10;
    
    private volatile Level defaultLogLevel = Level.WARNING;
    
//-- Life cycle : --------------------------------------------------------------
    
    /**
     * Constructs a scheduler with a fixed number of executor threads.
     * 
     * @param name name for threads created by this scheduler.
     * @param nThreads number of threads.
     */
    public Scheduler(String name, int nThreads) {
        executor = new SchedulerExecutor(name, nThreads);
    }

    /** Initiates orderly shutdown of this scheduler. */
    public void shutdown() {
        executor.shutdown();
    }

    /** Initiates forced shutdown of this scheduler. */
    public void shutdownNow() {
        executor.shutdownNow();
    }
    
    
// -- Setters : ----------------------------------------------------------------
    
    /** Sets logger to be used for reporting exceptions thrown by scheduled tasks. */
    public void setLogger(Logger logger) {
        log = logger;
    }

    /** 
     * Modifies default level at which uncaught exceptions thrown by scheduled tasks
     * are logged if no level is specified at task submission.
     * By default, this level is WARNING.
     */
    public void setDefaultLogLevel(Level defaultLogLevel) {
        this.defaultLogLevel = defaultLogLevel;
    }

    /**
     * Sets the default maximum number of failures (uncaught exceptions) fixedRate tasks
     * can encounter before being suppressed. The default is 10.
     * The limit associated with the scheduler can be overridden on a task-by-task basis.
     * 
     * @param maxFailures Maximum number of failures; if negative, the tasks are never suppressed.
     */
    public void setMaxFailures(int maxFailures) {
        this.maxFailures = maxFailures;
    }
    
    
// -- Getters : ----------------------------------------------------------------

    /** 
     * Returns the default level at which uncaught exceptions thrown by scheduled tasks
     * are logged if no level is specified at task submission.
     */
    public Level getDefaultLogLevel() {
        return defaultLogLevel;
    }

    /** Returns true if this scheduler has been shut down. */
    public boolean isShutdown() {
        return executor.isShutdown();
    }

   /** Returns true if all tasks have completed following shut down. */
    public boolean isTerminated() {
        return executor.isTerminated();
    }
    
// -- Scheduling : -------------------------------------------------------------

    /**
     * Creates and executes a one-shot action that becomes enabled after the given delay.
     *
     * @param command the task to execute
     * @param delay the time from now to delay execution
     * @param unit the time unit of the delay parameter
     * @return a PeriodicTask representing pending completion of the task, whose {@code get()} 
     *         method will return {@code null} upon completion
     * @throws RejectedExecutionException if the task cannot be scheduled for execution
     * @throws NullPointerException if command is null
     */
    public  ScheduledFuture<Void> schedule(Runnable command, long delay, TimeUnit unit) {
        return (ScheduledFuture<Void>) executor.schedule(command, delay, unit);
    }

    /**
     * Creates and executes a one-shot action that becomes enabled at the given time in the future.
     * If the specified moment in time has already passed, the execution is started immediately.
     *
     * @param command the task to execute.
     * @param date the time when the execution should start
     * @return a PeriodicTask representing pending completion of the task, whose {@code get()} 
     *         method will return {@code null} upon completion
     * @throws RejectedExecutionException if the task cannot be scheduled for execution
     * @throws NullPointerException if command is null
     */
    public  ScheduledFuture<Void> schedule(Runnable command, Date date) {
        return (ScheduledFuture<Void>) executor.schedule(command, date.getTime() - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
    }

    /**
     * Creates and executes a one-shot action that becomes enabled after the given delay.
     *
     * @param callable the function to execute
     * @param delay the time from now to delay execution
     * @param unit the time unit of the delay parameter
     * @param <V> the type of the result
     * @return a PeriodicTask that can be used to extract result or cancel
     * @throws RejectedExecutionException if the task cannot be scheduled for execution
     * @throws NullPointerException if callable is null
     */
    public <V> ScheduledFuture<V> schedule(Callable<V> callable, long delay, TimeUnit unit) {
        return executor.schedule(callable, delay, unit);
    }
    
    /**
     * Creates and executes a one-shot action that becomes enabled at the given time in the future.
     * If the specified moment in time has already passed, the execution is started immediately.
     *
     * @param callable the function to execute
     * @param date the time when the execution should start
     * @param <V> the type of the result
     * @return a PeriodicTask that can be used to extract result or cancel
     * @throws RejectedExecutionException if the task cannot be
     *         scheduled for execution
     * @throws NullPointerException if callable is null
     */
    public <V> ScheduledFuture<V> schedule(Callable<V> callable, Date date) {
        return executor.schedule(callable, date.getTime() - System.currentTimeMillis(), TimeUnit.MILLISECONDS);
    }
    
    /**
     * Creates and executes a fixedRate action that becomes enabled first after
     * the given initial delay, and subsequently with the given period. If any
     * execution of the task encounters an exception, subsequent executions are
     * suppressed. Otherwise, the task will only terminate via cancellation or
     * termination of the executor.  If any execution of this task
     * takes longer than its period, then subsequent executions
     * may start late, but will not concurrently execute.
     *
     * @param command the task to execute
     * @param initialDelay the time to delay first execution
     * @param period the period between successive executions
     * @param unit the time unit of the initialDelay and period parameters
     * @param taskName task name, used to identify failed tasks in the log
     * @param logLevel level at which exceptions thrown by scheduled tasks should be logged
     * @return a PeriodicTask representing pending completion of the task, and whose 
     *         {@code get()} method will throw an exception upon cancellation
     * @throws RejectedExecutionException if the task cannot be scheduled for execution
     * @throws NullPointerException if command is null
     * @throws IllegalArgumentException if period less than or equal to zero
     */
    public PeriodicTask scheduleAtFixedRate(Runnable command, long initialDelay, long period, TimeUnit unit, String taskName, Level logLevel) {
        PeriodicTask task = new PeriodicTask(this, command, true, taskName, logLevel, period, unit);
        task.start(initialDelay, unit);
        return task;
    }
    
    /**
     * Creates and executes a fixedRate action that becomes enabled first after
     * the given initial delay, and subsequently with the given period. If any
     * execution of the task encounters an exception, subsequent executions are suppressed.
     * Otherwise, the task will only terminate via cancellation or
     * termination of the executor.  If any execution of this task
     * takes longer than its period, then subsequent executions
     * may start late, but will not concurrently execute.
     *
     * @param command the task to execute
     * @param initialDelay the time to delay first execution
     * @param period the period between successive executions
     * @param unit the time unit of the initialDelay and period parameters
     * @return a PeriodicTask representing pending completion of the task, and whose 
     *         {@code get()} method will throw an exception upon cancellation
     * @throws RejectedExecutionException if the task cannot be scheduled for execution
     * @throws NullPointerException if command is null
     * @throws IllegalArgumentException if period less than or equal to zero
     */
    public PeriodicTask scheduleAtFixedRate(Runnable command, long initialDelay, long period, TimeUnit unit) {
        return scheduleAtFixedRate(command, initialDelay, period, unit, "", defaultLogLevel);
    }

    /**
     * Creates and executes a fixedRate action that becomes enabled first after
     * the given initial delay, and subsequently with the given delay between
     * the termination of one execution and the commencement of the next.
     *
     * @param command the task to execute
     * @param initialDelay the time to delay first execution
     * @param delay the delay between the termination of one
     *              execution and the commencement of the next
     * @param unit the time unit of the initialDelay and delay parameters
     * @param taskName task name, used to identify failed tasks in the log
     * @param logLevel level at which exceptions thrown by scheduled tasks should be logged
     * @return a PeriodicTask representing pending completion of the task, and whose 
     *         {@code get()} method will throw an exception upon cancellation
     * @throws RejectedExecutionException if the task cannot be scheduled for execution
     * @throws NullPointerException if command is null
     * @throws IllegalArgumentException if delay less than or equal to zero
     */
    public PeriodicTask scheduleWithFixedDelay(Runnable command, long initialDelay, long delay, TimeUnit unit, String taskName, Level logLevel) {
        PeriodicTask task = new PeriodicTask(this, command, false, taskName, logLevel, delay, unit);
        task.start(initialDelay, unit);
        return task;
    }

    /**
     * Creates and executes a fixedRate action that becomes enabled first after
     * the given initial delay, and subsequently with the given delay between
     * the termination of one execution and the commencement of the next.
     *
     * @param command the task to execute
     * @param initialDelay the time to delay first execution
     * @param delay the delay between the termination of one
     * execution and the commencement of the next
     * @param unit the time unit of the initialDelay and delay parameters
     * @return a PeriodicTask representing pending completion of the task, and whose 
     *         {@code get()} method will throw an exception upon cancellation
     * @throws RejectedExecutionException if the task cannot be scheduled for execution
     * @throws NullPointerException if command is null
     * @throws IllegalArgumentException if delay less than or equal to zero
     */
    public PeriodicTask scheduleWithFixedDelay(Runnable command, long initialDelay, long delay, TimeUnit unit) {
        return scheduleWithFixedDelay(command, initialDelay, delay, unit, "", defaultLogLevel);
    }
    
    /**
     * Blocks until all tasks have completed execution after a shutdown request, or the 
     * timeout occurs, or the current thread is interrupted, whichever happens first.
     * 
     * @param timeout the maximum time to wait.
     * @param unit the time unit of the timeout argument.
     * @return true if this executor terminated and false if the timeout elapsed before termination.
     * @throws InterruptedException if interrupted while waiting
     */
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        return executor.awaitTermination(timeout, unit);
    } 
    
    
// -- Customized ScheduledThreadPoolExecutor : ---------------------------------
    
    private class SchedulerExecutor extends ScheduledThreadPoolExecutor {
        
        SchedulerExecutor(String name, int nThreads) {
            super(nThreads, r -> new Thread(r, name));
            setRemoveOnCancelPolicy(true);
        }
        
    }
    
}
