package org.lsst.ccs.utilities.logging;

import java.util.Optional;
import org.lsst.ccs.utilities.tracers.Tracer;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;

/**
 * Just a wrapper class around java.util.Logger to add specific logging methods.
 * The instances are not cached (so better have only one per package)
 * <BR/>
 * Most methods allow "multi-dimensional" logging (log the same data to various
 * loggers).
 * <BR/>
 * Many of those are intended to replace log4j invocations by just changing the
 * <TT>import</TT> directives. The mapping from log4J Levels is operated this
 * way
 * <PRE>
 * finest -> finest
 * trace -> finer
 * debug-> fine
 * info -> info
 * warn -> warning
 * error -> severe
 * fatal -> severe
 * </PRE>
 * <BR/>
 * Logging methods that return a boolean can be used through <TT>assert</TT>
 * calls (they always return true).
 * <p>
 * Do not use this <TT>Logger</TT> for admin purpose (setting a level, a Filter,
 * a Formatter or a Handler): use the corresponding
 * <TT>java.util.logging.Logger</TT> instead.
 *
 * @author bamade
 */
// Date: 29/05/13
public class Logger {

    static {
        assert Tracer.version("$Rev$", Logger.class, "org-lsst-ccs-utilities");
        // Forces Logger Initialization
        // See https://jira.slac.stanford.edu/browse/LSSTCCS-215
        /*
        * this is to ensure that the first time we use our Logger
        * then static initialization of codes of LogManagement and thus LogPropertiesLoader
        * are load-time initialized.
        * (though this is probably the case but depends on JVM load strategy)
        */
        LogPropertiesLoader.getSystemLoaderLogManager();
    }

    /**
     * the real JUL logger for which we delegate log4J lookalike methods
     * and all our logs.
     */
    private java.util.logging.Logger julDelegate;

    /**
     * scheduler for delayed logs
     */
    protected ScheduledExecutorService executor = Executors.newScheduledThreadPool(1,
            new ThreadFactory() {
                @Override
                public Thread newThread(Runnable r) {
                    Thread res = new Thread(r, "LoggerDelayedExecutor");
                    res.setDaemon(true);
                    return res;
                }
            });

    /**
     * private constructor: we need a JUL delegate to deliver our own Logger.
     * see factory methods to obtain a Logger.
     * @param delegate
     */
    private Logger(java.util.logging.Logger delegate) {
        this.julDelegate = delegate;
        Optional<LogManager> optManager = LogPropertiesLoader.getSystemLoaderLogManager();
        if (optManager.isPresent()) {
            LogManager manager = optManager.get();
            manager.addLogger(delegate);
        }
    }

    /**
     * factory method to obtain a <TT>Logger</TT> proxy.
     *
     * @param name usually a package name (top of hierarchy is "" empty
     *             String)
     * @return
     */
    public static Logger getLogger(String name) {
        java.util.logging.Logger realLogger = java.util.logging.Logger.getLogger(name);
        return new Logger(realLogger);
    }

    /**
     * factory method to obtain a <TT>Logger</TT> proxy.
     *
     * @param name               usually a package name (top of hierarchy is "" empty
     *                           String)
     * @param resourceBundleName
     * @return
     */
    public static Logger getLogger(String name, String resourceBundleName) {
        java.util.logging.Logger realLogger = java.util.logging.Logger.getLogger(name, resourceBundleName);
        return new Logger(realLogger);
    }

    /**
     * @return the name of the Logger
     */
    public String getName() {
        return julDelegate.getName();
    }

    /**
     * @return the level of the associated JUL logger
     */
    public Level getLevel() {
        return julDelegate.getLevel();
    }

    public void setLevel (Level level){
        julDelegate.setLevel(level);
    }

    /**
     * @return the parent of the associated JUL logger
     */
    protected java.util.logging.Logger getParent() {
        return julDelegate.getParent();
    }

    /**
     * creates a simple log record. This factory manipulates the caller
     * information (className, method Name) by getting rid of any
     * information coming from a package that contains "logging" (or
     * "log4j").
     * <BR/>
     * note that the LogRecord is automatically marked with a sequenceNumber
     * that can be used by <TT>Handlers</TT>
     * that want to detect duplicate publications.
     *
     * @param level   JUL Level
     * @param message (avoid null values)
     * @return a simple LogRecord
     */
    public LogRecord createLogRecord(Level level, String message) {
        LogRecord res = new LogRecord(level, message);
        //now modifies the method and class calls
        //Throwable throwable = new Throwable();
        //StackTraceElement[] stackTraceElements = throwable.getStackTrace();
        StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
        for (StackTraceElement stackTraceElement : stackTraceElements) {
            String className = stackTraceElement.getClassName();
            if (!className.startsWith("org.lsst.ccs.utilities.logging")) {
                if (!className.startsWith("org.apache.log4j") && !className.contains(".logging.")) {
                    String methodName = stackTraceElement.getMethodName();
                    if (!"getStackTrace".equals(methodName)) {
                        res.setSourceClassName(className);
                        res.setSourceMethodName(methodName);
                        break;
                    }
                }
            }
        }

        return res;
    }

    /* LOG4J compatible calls
     finest -> finest
     trace -> finer
     debug-> fine
     info -> info
     warn -> warning
     error -> severe
     fatal -> severe
     isDebugEnabled
     */

    /**
     * utility method to log a message with a Level
     *
     * @param level    JUL level
     * @param message  (avoid null values)
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     * @return true
     * @deprecated use logMessage(Level level, Object message) instead
     */
    @Deprecated
    protected boolean logMessage(Level level, Object message, String... concerns) {
        return logMessage(level, message);
    }
    
    protected boolean logMessage(Level level, Object message) {
        String msg = String.valueOf(message);
        LogRecord record = createLogRecord(level, msg);
        this.log(record);
        return true;
    }

    /**
     * logs a message at FINEST level
     *
     * @param message
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     * @return true
     */
    @Deprecated
    public boolean finest(Object message, String... concerns) {
        return finest(message);
    }
    
    public boolean finest(Object message) {
        return logMessage(Level.FINEST, message);
    }

    /**
     * logs a message at FINER level
     *
     * @param message
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     * @return true
     */
    @Deprecated
    public boolean trace(Object message, String... concerns) {
        return trace(message);
    }

    public boolean trace(Object message) {
        return logMessage(Level.FINER, message);
    }
    /**
     * logs a message at FINER level
     *
     * @param message
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     * @return true
     */
    @Deprecated
    public boolean finer(Object message, String... concerns) {
        return finer(message);
    }

    public boolean finer(Object message) {
        return logMessage(Level.FINER, message);
    }
    /**
     * logs a message at FINE Level
     *
     * @param message
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     * @return true
     */
    @Deprecated
    public boolean debug(Object message, String... concerns) {
        return debug(message);
    }

    public boolean debug(Object message) {
        return logMessage(Level.FINE, message);
    }
    /**
     * logs a message at FINE Level
     *
     * @param message
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     * @return true
     */
    @Deprecated
    public boolean fine(Object message, String... concerns) {
        return fine(message);
    }
    public boolean fine(Object message) {
        return logMessage(Level.FINE, message);
    }

    /**
     * logs a message at INFO Level
     *
     * @param message
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     * @return true
     */
    @Deprecated
    public boolean info(Object message, String... concerns) {
        return info(message);
    }

    public boolean info(Object message) {
        return logMessage(Level.INFO, message);
    }
    /**
     * logs a message at WARNING Level
     *
     * @param message
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     * @return true
     */
    @Deprecated
    public boolean warn(Object message, String... concerns) {
        return warn(message);
    }
    
    public boolean warn(Object message) {
        return logMessage(Level.WARNING, message);
    }
    /**
     * logs a message at WARNING Level
     *
     * @param message
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     * @return true
     */
    @Deprecated
    public boolean warning(Object message, String... concerns) {
        return warning(message);
    }
    
    public boolean warning(Object message) {
        return logMessage(Level.WARNING, message);
    }

    /**
     * logs a message at SEVERE Level (use preferably the error method that
     * uses a <TT>Throwable</TT> parameter)
     *
     * @param message
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     */
    @Deprecated
    public void error(Object message, String... concerns) {
        error(message);
    }
    
    public void error(Object message) {
        logMessage(Level.SEVERE, message);
    }

    /**
     * logs a message at SEVERE Level (use preferably the fatal method that
     * uses a <TT>Throwable</TT> parameter)
     *
     * @param message
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     */
    @Deprecated
    public void fatal(Object message, String... concerns) {
        fatal(message);
    }
    public void fatal(Object message) {
        logMessage(Level.SEVERE, message);
    }
    /**
     * logs a message at SEVERE Level (use preferably the severe method that
     * uses a <TT>Throwable</TT> parameter)
     *
     * @param message
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     */
    @Deprecated
    public void severe(Object message, String... concerns) {
        severe(message);
    }

    public void severe(Object message) {
        logMessage(Level.SEVERE, message);
    }
    /**
     * tells if the FINE level is activated for the corresponding JUL
     * Logger. (to be used if a subsequent logging call is costly)
     *
     * @return
     */
    public boolean isDebugEnabled() {
        return julDelegate.isLoggable(Level.FINE);
    }

    /**
     * tells if the INFO level is activated for the corresponding JUL
     * Logger. (to be used if a subsequent logging call is costly)
     *
     * @return
     */
    public boolean isInfoEnabled() {
        return julDelegate.isLoggable(Level.INFO);
    }

    /**
     * utility method to forward a <TT>Throwable</TT> and a message to the
     * loggers.
     *
     * @param level     JUL level
     * @param message
     * @param throwable
     * @param concerns  a list of additional JUL loggers name (such as
     *                  "INIT", "CONFIG" ,...)
     */
    @Deprecated
    protected void logSimpleThrowable(Level level, Object message, Throwable throwable, String... concerns) {
        logSimpleThrowable(level, message, throwable);
    }

    protected void logSimpleThrowable(Level level, Object message, Throwable throwable) {
        String msg = String.valueOf(message);
        LogRecord record = createLogRecord(level, msg);
        record.setThrown(throwable);
        this.log(record);
    }
    
    /**
     * logs a message at SEVERE Level
     *
     * @param message
     * @param throwable
     * @param concerns  a list of additional JUL loggers name (such as
     *                  "INIT", "CONFIG" ,...)
     */
    @Deprecated
    public void fatal(Object message, Throwable throwable, String... concerns) {
        fatal(message, throwable);
    }

    public void fatal(Object message, Throwable throwable) {
        logSimpleThrowable(Level.SEVERE, message, throwable);
    }
    /**
     * logs a message at SEVERE Level
     *
     * @param message
     * @param throwable
     * @param concerns  a list of additional JUL loggers name (such as
     *                  "INIT", "CONFIG" ,...)
     */
    @Deprecated
    public void severe(Object message, Throwable throwable, String... concerns) {
        severe(message, throwable);
    }

    public void severe(Object message, Throwable throwable) {
        logSimpleThrowable(Level.SEVERE, message, throwable);
    }
    /**
     * logs a message at SEVERE Level
     *
     * @param message
     * @param throwable
     * @param concerns  a list of additional JUL loggers name (such as
     *                  "INIT", "CONFIG" ,...)
     */
    @Deprecated
    public void error(Object message, Throwable throwable, String... concerns) {
        error(message, throwable);
    }
    public void error(Object message, Throwable throwable) {
        logSimpleThrowable(Level.SEVERE, message, throwable);
    }
    /**
     * logs a message at WARNING Level
     *
     * @param message
     * @param throwable
     * @param concerns  a list of additional JUL loggers name (such as
     *                  "INIT", "CONFIG" ,...)
     */
    @Deprecated
    public void warn(Object message, Throwable throwable, String... concerns) {
        warn(message, throwable);
    }
    public void warn(Object message, Throwable throwable) {
        logSimpleThrowable(Level.WARNING, message, throwable);
    }

    /**
     * logs a message at WARNING Level
     *
     * @param message
     * @param throwable
     * @param concerns  a list of additional JUL loggers name (such as
     *                  "INIT", "CONFIG" ,...)
     */
    @Deprecated 
    public void warning(Object message, Throwable throwable, String... concerns) {
        warning(message, throwable);
    }
    public void warning(Object message, Throwable throwable) {
        logSimpleThrowable(Level.WARNING, message, throwable);
    }
    /**
     * invokes the <TT>throwing</TT> method on the corresponding JUL logger.
     *
     * @param sourceClass
     * @param sourceMethod
     * @param throwable
     */
    public void throwing(String sourceClass, String sourceMethod, Throwable throwable) {
        julDelegate.throwing(sourceClass, sourceMethod, throwable);
    }

    /**
     * logs a message at INFO Level
     *
     * @param message
     * @param throwable
     * @param concerns  a list of additional JUL loggers name (such as
     *                  "INIT", "CONFIG" ,...)
     */
    @Deprecated
    public void info(Object message, Throwable throwable, String... concerns) {
       info(message, throwable);
    }
    public void info(Object message, Throwable throwable) {
        logSimpleThrowable(Level.INFO, message, throwable);
    }

    /**
     * logs a message at FINE Level
     *
     * @param message
     * @param throwable
     * @param concerns  a list of additional JUL loggers name (such as
     *                  "INIT", "CONFIG" ,...)
     */
    @Deprecated
    public void debug(Object message, Throwable throwable, String... concerns) {
        debug(message, throwable);
    }
    public void debug(Object message, Throwable throwable) {
        logSimpleThrowable(Level.FINE, message, throwable);
    }

    //END LOG4J
    // BEGIN JUL-LIKE calls

    /**
     * tells if the corresponding JUL level is activated for the
     * corresponding JUL Logger. (to be used if a subsequent logging call is
     * costly)
     *
     * @return
     */
    public boolean isLoggable(Level level) {
        return julDelegate.isLoggable(level);
    }

    /**
     * utility method to send a <TT>LogRecord</TT> to the corresponding JUL
     * logger and to a list of other loggers.
     *
     * @param record   (should be created with the <TT>createLogRecord</TT>
     *                 factory :otherwise stack information will be wrong)
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     */
    @Deprecated
    public void log(LogRecord record, String... concerns) {
        log(record);
    }
    
    public void log(LogRecord record){
        record.setLoggerName(julDelegate.getName());
        julDelegate.log(record);
    }
    /*
    public static Logger[] emptyArg = new Logger[0] ;
    public void log(LogRecord record, Logger... otherLoggers) {
        record.setLoggerName(julDelegate.getName());
        julDelegate.log(record);
        for(Logger logger: otherLoggers) {
            logger.log(record, emptyArg) ;
        }
    }
    */

    /**
     * same method as <TT>log(LogRecord, String... concerns</TT>
     * but executed in a different Thread.
     * <BR/>
     * this might be interesting when the Thread that logs should be
     * different from the Thread that handles the LogRecord
     *
     * @param delay    in millis
     * @param record
     * @param concerns
     */
    @Deprecated
    public void decoupledLog(long delay, final LogRecord record, final String... concerns) {
        decoupledLog(delay, record);

    }
    public void decoupledLog(long delay, final LogRecord record) {
        executor.schedule(new Runnable() {
            public void run() {
                log(record);
            }
        }, delay, TimeUnit.MILLISECONDS);

    }
    
    /**
     * does the same as the equivalent standard JUL method but forwards also
     * to other Loggers.
     *
     * @param level    JUL level
     * @param message
     * @param argument
     * @param concerns a list of additional JUL loggers name (such as
     *                 "INIT", "CONFIG" ,...)
     * @return true
     */
    @Deprecated
    public boolean log(Level level, String message, Object argument, String... concerns) {
        return log(level,message,argument);
    }
    public boolean log(Level level, String message, Object argument) {
        LogRecord record = createLogRecord(level, message);
        record.setParameters(new Object[]{argument});
        this.log(record);
        return true;
    }

    /**
     * same method as <TT>public boolean log(Level level, String message,
     * Object argument, String... concerns)</TT>
     * but executed in a different Thread.
     * <BR/>
     * this might be interesting when the Thread that logs should be
     * different from the Thread that handles the LogRecord
     *
     * @param delay    in Milliseconds
     * @param level
     * @param message
     * @param argument
     * @param concerns
     * @return
     */
    @Deprecated
    public boolean decoupledLog(long delay, Level level, String message, Object argument, String... concerns) {
        return decoupledLog(delay, level, message, argument);
    }
    public boolean decoupledLog(long delay, Level level, String message, Object argument) {
        LogRecord record = createLogRecord(level, message);
        record.setParameters(new Object[]{argument});
        this.decoupledLog(delay, record);
        return true;
    }

    /**
     * does the same as the equivalent standard JUL method but forwards also
     * to other Loggers.
     *
     * @param level     JUL level
     * @param message
     * @param arguments
     * @param concerns  a list of additional JUL loggers name (such as
     *                  "INIT", "CONFIG" ,...)
     * @return true
     */
    @Deprecated
    public boolean log(Level level, String message, Object[] arguments, String... concerns) {
        return log(level,message,arguments);
    }
    public boolean log(Level level, String message, Object[] arguments) {
        LogRecord record = createLogRecord(level, message);
        record.setParameters(arguments);
        this.log(record);
        return true;
    }

    /**
     * does the same as the corresponding <TT>log</TT> method but executed
     * in a different Thread.
     * <BR/>
     * this might be interesting when the Thread that logs should be
     * different from the Thread that handles the LogRecord
     *
     * @param delay     in millis
     * @param level
     * @param message
     * @param arguments
     * @param concerns
     * @return
     */
    @Deprecated
    public boolean decoupledLog(long delay, Level level, String message, Object[] arguments, String... concerns) {
        return decoupledLog(delay, level, message, arguments);
    }
    public boolean decoupledLog(long delay, Level level, String message, Object[] arguments) {
        LogRecord record = createLogRecord(level, message);
        record.setParameters(arguments);
        this.decoupledLog(delay, record);
        return true;
    }

    /**
     * does the same as the equivalent standard JUL method but forwards also
     * to other Loggers.
     *
     * @param level     JUL level
     * @param message
     * @param throwable
     * @param concerns  a list of additional JUL loggers name (such as
     *                  "INIT", "CONFIG" ,...)
     * @return true
     */
    @Deprecated
    public void log(Level level, String message, Throwable throwable, String... concerns) {
        log(level, message, throwable);
    }
    public void log(Level level, String message, Throwable throwable) {
        LogRecord record = createLogRecord(level, message);
        record.setThrown(throwable);
        this.log(record);
    }

    /**
     * does the same as the corresponding <TT>log</TT> method but executed
     * in a different Thread.
     * <BR/>
     * this might be interesting when the Thread that logs should be
     * different from the Thread that handles the LogRecord
     *
     * @param delay     in millis
     * @param level
     * @param message
     * @param throwable
     * @param concerns
     */
    @Deprecated
    public void decoupledLog(long delay, Level level, String message, Throwable throwable, String... concerns) {
        decoupledLog(delay, level, message, throwable);
    }
    public void decoupledLog(long delay, Level level, String message, Throwable throwable) {
        LogRecord record = createLogRecord(level, message);
        record.setThrown(throwable);
        this.decoupledLog(delay, record);
    }


}
