package org.lsst.ccs.utilities.tracers;

import java.util.ServiceLoader;

/**
 * Utility to add traces to a code.
 * <p/>
 * Though it is a hack on assertions this can be used that way
 * <PRE>
 * import static org.lsst.ccs.utilities.tracers.Tracer.* ;
 * ....
 * public void myMethod(int arg) {
 *      assert trace("myMethod called with" +arg) ;
 * // code
 * }
 * <p/>
 * public void anotherMethod(String arg) {
 *      assert stackTrace() ;
 * }
 * </PRE>
 * A developper can also use home-grown filters
 * <p/>
 * <PRE>
 * import static org.lsst.ccs.utilities.tracers.Tracer.* ;
 * import org.lsst.ccs.mypackage.MyEnum ;
 * ....
 * public void myMethod(int arg) {
 *     assert trace(MyEnum.DOING_THIS, "myMethod called with" +arg) ;
 * // code
 * }
 * <p/>
 * public void anotherMethod(String arg) {
 *    assert stackTrace(MyEnum.DOING_THAT) ;
 * }
 * </PRE>
 * then if System property <TT>org.lsst.ccs.traces</TT> is set
 * the traces can be produced or not. Example of option
 * <TT>-Dorg.lsst.ccs.traces=DOING_THIS,DOING_THAT</TT>
 * <p/>
 * By default the strings are printed to System.out but ths can be changed.
 * If you want to delegate the reporting to another code (such as a Logger)
 * write a code implementing <TT>Tracer.Strategy</TT>
 * and deploy it using the <TT>ServiceLoader</TT> (or put the canonical
 * className as value for property <TT>org.lsst.ccs.traceReporter</TT>).
 *
 * @author bamade
 */
// Date: 23/04/13

public class Tracer {
    /**
     * defines the behaviour of a tracer.
     * Implementations can forward traces to a Logger for instance.
     */
    public interface Strategy {
        /**
         * starts a trace and reports a String.
         * Each "report" (mostly one line reports) is terminated by a call
         * to <TT>endReport</TT>
         * @param report
         */
        void report(String report);

        /**
         * long reports that span many lines (such as stackTraces) may have intermediate
         * calls to <TT>addReportLine</TT> before a call to <TT>endReport</TT>
         * @param line
         */
        void addReportLine(String line);

        /**
         * marks the end of a trace (this can be used with implementations that use
         * parenthesized expressions -such as XML-).
         * it is guaranteed that each call to <TT>report</TT> is closed by a call
         * to <TT>endReport</TT> (in the meantime many invocations to <TT>addReportLine</TT>
         * may have been fired).
         */
        void endReport();
    }

    /**
     * Convenience class to create a code that reports to many report "handlers".
     * (for instance to a Logger AND  System.out).
     */
    public static class MultiReporter implements Strategy {
        Strategy[] strategies;

        public MultiReporter(Strategy... strategies) {
            this.strategies = strategies;
        }

        @Override
        public void report(String report) {
            for (Strategy strategy : strategies) {
                strategy.report(report);
            }
        }

        @Override
        public void addReportLine(String line) {
            for (Strategy strategy : strategies) {
                strategy.addReportLine(line);
            }
        }

        @Override
        public void endReport() {
            for (Strategy strategy : strategies) {
                strategy.endReport();
            }
        }
    }

    public static Strategy reporter;

    static {
        ServiceLoader<Tracer.Strategy> loader = ServiceLoader.load(Tracer.Strategy.class);
        boolean found = false;
        try {
            for (Tracer.Strategy rep : loader) {
                reporter = rep;
                found = true;
                break;
            }

        } catch (Exception exc) { /*IGNORE*/ }
        if (!found) {
            String className = System.getProperty("org.lsst.ccs.traceReporter");
            if (className == null) {
                reporter = new Strategy() {
                    @Override
                    public void report(String report) {
                        System.out.println(report);
                    }

                    @Override
                    public void addReportLine(String line) {
                        System.out.println('\t' + line);
                    }

                    @Override
                    public void endReport() {
                    }
                };
            } else {
                try {
                    Class clazz = Class.forName(className);
                    reporter = (Strategy) clazz.newInstance();
                } catch (Exception exc) {
                    throw new Error("incorrect class for trace configuration", exc);
                }
            }

        }
    }

    public static void setReporter(Strategy reporter) {
        Tracer.reporter = reporter;
    }

    public static Strategy getReporter() {
        return reporter;
    }

    protected static String[] keys;

    static {
        String properties = System.getProperty("org.lsst.ccs.traces");
        if (properties != null) {
            keys = properties.split(",");
        } else {
            keys = new String[0];
        }
    }

    /**
     * utility method to fire a trace.
     * Though it is not mandatory this method could be used as a side-effect of an <TT>assert</TT>
     * (this is not a politically correct use of the assert feature).
     * @param traceString will always be  transmitted to code for reporting the trace
     * @return
     */
    public static boolean trace(String traceString) {
        reporter.report(traceString);
        reporter.endReport();
        return true;
    }

    /**
     * reports a trace only if the <TT>filter</TT> string is part of the <TT>org.lsst.ccs.traces</TT>
     * property.
     * It is highly recommended to use enums instead of Strings to operate this filtering.
     * @param filter
     * @param traceString
     * @return
     */
    public static boolean trace(String filter, String traceString) {
        for (String key : keys) {
            if (key.equalsIgnoreCase(filter)) {
                reporter.report(traceString);
                reporter.endReport();
                return true;
            }
        }
        return true;
    }

    /**
     * uses an enum constant to filter the message passed to the tracer.
     * @param enumElement
     * @param traceString
     * @param <T>
     * @return
     */
    public static <T extends Enum> boolean trace(T enumElement, String traceString) {
        return trace(String.valueOf(enumElement), traceString);
    }

    /**
     * reports a complete stack trace of the current  code execution.
     * It is highly recommended to use with an assert and preferably with a filter.
     * @return
     */
    public static boolean stackTrace() {
        stackTrace(3);
        return true;
    }

    /**
     * reports a complete stack trace of the current  code execution
     * only if the <TT>filter</TT> string is part of the <TT>org.lsst.ccs.traces</TT>
     * property.
     * @param filter
     * @return
     */
    public static boolean stackTrace(String filter) {
        for (String key : keys) {
            if (key.equalsIgnoreCase(filter)) {
                stackTrace(3);
                return true;
            }
        }
        return true;
    }

    /**
     * reports a complete stack trace of the current  code execution
     * only if the <TT>filter</TT> enum constant is cited in the <TT>org.lsst.ccs.traces</TT>
     * property.
     *
     * @param enumElement
     * @param <T>
     * @return
     */
    public static <T extends Enum> boolean stackTrace(T enumElement) {
        String filter = String.valueOf(enumElement);
        for (String key : keys) {
            if (key.equalsIgnoreCase(filter)) {
                stackTrace(1);
                return true;
            }
        }
        return true;
    }

    /**
     * reports part a stacktrace starting at index <TT>levelOut</TT>
     * @param levelOut
     */
    protected static void stackTrace(int levelOut) {
        StackTraceElement[] stack = Thread.currentThread().getStackTrace();
        reporter.report("-----------------");
        for (int ix = levelOut; ix < stack.length; ix++) {
            StackTraceElement element = stack[ix];
            reporter.addReportLine(element.toString());
        }
        reporter.endReport();
    }
}
