package org.lsst.ccs.utilities.dispatch;

import java.util.LinkedHashMap;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Typical use of this pattern is to have two threads:
 * <UL>
 *     <LI> one is going to write commands to a stream. Before writing the command data
 *     the calling thread registers a code that may  handle  the corresponding asynchronous reply.
 *     This is done by invoking one of the <TT>register</TT> methods.
 *     So there must be a token that must be sent with the command and must be delivered
 *     with the correponding response.
 *     <LI> The other Thread is reading responses on a stream.
 *     Each time it receives a response it gathers: the token needed for correlation, and the data
 *     needed to execute the response code. The correlation token and the data is passed
 *     to the <TT>callBack</TT> method of the InvocationDispenser</LI>
 * </UL>
 * examples:
 * <PRE>
 *     public synchronized void replytoA(Object elem1, elem2) {
 *         //do something
 *     }
 *     public void sendCommandA(Object someData) {
 *        String token = this.myInvocationDispenser.register( new Invoker() {
 *        public void invoke(Object... args) {
 *            //check args ?
 *            replytoA(args[0], args[1]) ;
 *        }
 *
 *        } ) ;
 *        ACommand toBeSent = buildCommand(token, someData) ;
 *        sendCommand(toBeSent)
 *     }
 * </PRE>
 *
 * The listening Thread :
 * <PRE>
 *     public void run() {
 *         while(doItAgain) {
 *             // don't forget exception handling!
 *             CommandData receivedData = readReply() ;
 *             String correlationString =  receivedData.getToken() ;
 *             Object[] responseData = receivedData.getData() ;
 *             myInvocationDispenser.callBack(correlationString, responseData) ;
 *         }
 *     }
 * </PRE>
 * @author bamade
 */
// Date: 30/05/12

public class InvocationDispenser {
    public static class TokenLostException extends Exception {
        public final String token ;

        public TokenLostException(String token) {
            this.token = token;
        }

        public String toString() {
            return super.toString() + " " + token ;
        }
    }
    class DelegateInvoker implements Invoker{
        Invoker delegate ;

        Object res ;

        Object getRes() {
            return res ;
        }

        FutureTask task = new FutureTask(new Callable() {
            @Override
            public Object call() throws Exception {
                return getRes();
            }
        });

        public DelegateInvoker(Invoker delegate) {
            this.delegate = delegate;
        }


        @Override
        public Object invoke(Object... args) throws Exception{
            res =  delegate.invoke(args);
            task.run() ;
            return res ;
        }

        public FutureTask getTask() {
            return task ;
        }
    }
    //TODO: naive seed!!! use UUID ?
    private static AtomicLong seed = new AtomicLong(System.nanoTime()) ;

    // TODO :get rid of "old" registrations that are did not receive a response
    private LinkedHashMap<String, Invoker> map = new LinkedHashMap<String, Invoker>() ;

    /**
     * generates a unique token (for this ClassLoader context!)
     * @return
     */
    public String generateToken() {
        long longKey = seed.getAndIncrement() ;
        String key = Long.toString(longKey,Character.MAX_RADIX);
        return key ;
    }
    /**
     * registers an Invoker and returns a token to be used to run the invocation code
     * @param invoker should not be null.
     * @return
     */
    public synchronized String register(Invoker invoker) {
        if(invoker == null) {
            throw new IllegalArgumentException("null Invoker") ;
        }
        String key =  generateToken() ;
        map.put(key, invoker) ;
        return key ;
    }

    /**
     * registers an invoker and the token is given by the calling code.
     * @param token
     * @param invoker
     * @return
     */
    public synchronized String register(String token ,Invoker invoker) {
        if(invoker == null) {
            throw new IllegalArgumentException("null Invoker") ;
        }
        map.put(token, invoker) ;
        return token ;
    }

    /**
     * Gets a <TT>Future</TT> object from this task registration.
     * This can be used by methods that are "nearly synchronous" : that is
     * it is a synchronous call but the process that reads the response may be polluted
     * by other incoming events.
     * <P>
     *  Exemple of call may be :
     *  <PRE>
     *      public String pleaseCall(String arg) {
     *           String token = myInvocationDispenser.generateToken() ;
     *          Future future = registerFuture(token, new Invoker() {
     *              public Object invoke(Object... args) {
     *                  return args[0] ;
     *              }
     *          }
     *        ACommand toBeSent = buildCommand(token, arg) ;
     *        sendCommand(toBeSent)
     *          return (String) future.get() ;
     *      }
     *  </PRE>
     * @param token
     * @param invoker
     * @return
     */
    public synchronized Future registerFuture(String token, Invoker invoker) {
        if(invoker == null) {
            throw new IllegalArgumentException("null Invoker") ;
        }
        DelegateInvoker delegateInvoker = new DelegateInvoker(invoker) ;
        map.put(token, delegateInvoker) ;
        return delegateInvoker.getTask() ;
    }

    /**
     * Removes a response task
     * @param token
     */
    public synchronized void dumpTask(String token) {
        map.remove(token) ;
    }


    /**
     * tells if the token is known.
     * <B>beware</B> : in a multihreaded environment there may be race condition
     * where this method returns true but where  an Exception is fired upon call of
     * callback Method.
     * @param token
     * @return
     */
    public synchronized boolean isTokenUsed(String token) {
        if(token == null) return false ;
        return map.containsKey(token) ;
    }

    /**
     * calls the registered invoker which corresponds to the token.
     * The token is no longer valid.
     * @param token
     * @param args
     * @return
     * @throws Exception
     */
    public Object callBack(String token, Object... args) throws Exception{
        Invoker invoker ;
        synchronized (this) {
            invoker = map.remove(token) ;
            if(invoker == null) {
                throw new TokenLostException(token) ;
            }
        }
        return invoker.invoke(args) ;
    }

    /**
     * calls the registered invoker which corresponds to the token.
     * The token is still valid.
     * That is to be used when there is a registration of a code
     * that is to treat purely incoming events (without a corresponding send).
     * The invoker should be able to run many times.
     * @param token
     * @param args
     * @return
     * @throws Exception
     */
    public Object call(String token, Object... args) throws Exception{
        Invoker invoker ;
        synchronized (this) {
            invoker = map.get(token) ;
            if(invoker == null) {
                throw new TokenLostException(token) ;
            }
        }
        return invoker.invoke(args) ;

    }
}
