package org.lsst.ccs.utilities.sa;

import java.io.PrintStream;
import java.util.LinkedList;
import java.util.Scanner;

/**
 ***************************************************************************
 **
 **  Class to implement command processing, new style
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class CmndProcess {

    public final static int
        CMD_AMBIG       = -2,
        CMD_UNKNOWN     = -1;

    private final static PrintStream out = System.out;

    private LinkedList<Dispatch> disps = new LinkedList<Dispatch>();
    private LinkedList<Command> cmnds = new LinkedList<Command>();


   /**
    ***************************************************************************
    **
    **  Inner interface to define command dispatching
    **
    ***************************************************************************
    */
    public interface Dispatch {

        boolean dispatch(int code, int found, Object[] args);

    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement command decoding
    **
    ***************************************************************************
    */
    public static class Command {

        private int nCmnd = 0;
        private int[] cmdCode;
        private String[] cmdName;
        private String[] cmdArgDesc;
        private Lookup[][] cmdLookup;
        private String[][] cmdHelp;

       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public Command(int count)
        {
            cmdCode    = new int[count];
            cmdName    = new String[count];
            cmdArgDesc = new String[count];
            cmdLookup  = new Lookup[count][];
            cmdHelp    = new String[count][];
        }


       /**
        ***********************************************************************
        **
        **  Adds a command to the table
        **
        ***********************************************************************
        */
        public void add(String command, int code, String[] help, String argDesc,
                        Lookup ... lookup)
        {
            cmdName[nCmnd]    = command;
            cmdCode[nCmnd]    = code;
            cmdHelp[nCmnd]    = help;
            cmdArgDesc[nCmnd] = argDesc;
            cmdLookup[nCmnd]  = lookup;
            nCmnd++;
        }


       /**
        ***********************************************************************
        **
        **  Gets the number of commands
        **
        ***********************************************************************
        */
        public int count()
        {
            return nCmnd;
        }


       /**
        ***********************************************************************
        **
        **  Gets the index of a command, given its name
        **
        **  This routine compares the command string with the first part of
        **  all the table entries and returns one of the following:
        **
        **    index (>= 0) If the command is fully matched
        **    CMD_UNKNOWN  If the command is not in the table
        **    CMD_AMBIG    If the command matches more than one entry
        **    CMD_AMBIG - 1 - index  If the command is partially matched
        **
        ***********************************************************************
        */
        public int find(String command)
        {
            int index = 0, match = 0;

            for (int j = 0; j < nCmnd; j++) {
                String name = cmdName[j];
                if (name != null && name.startsWith(command)) {
                    if (name.equals(command)) return j;
                    index = j;
                    match++;
                }
            }

            if (match == 0) return CMD_UNKNOWN;
            if (match > 1) return CMD_AMBIG;

            return CMD_AMBIG - 1 - index;
        }


       /**
        ***********************************************************************
        **
        **  Disables a command, given its index
        **
        ***********************************************************************
        */
        public void disable(int index)
        {
            if (index >= 0) cmdName[index] = null;
        }


       /**
        ***********************************************************************
        **
        **  Gets the name of a command, given its index
        **
        ***********************************************************************
        */
        public String name(int index)
        {
            return (index >= 0) ? cmdName[index] : null;
        }


       /**
        ***********************************************************************
        **
        **  Gets the code for a command, given its index
        **
        ***********************************************************************
        */
        public int code(int index)
        {
            return (index >= 0) ? cmdCode[index] : index;
        }


       /**
        ***********************************************************************
        **
        **  Gets the argument descriptor for a command, given its index
        **
        ***********************************************************************
        */
        public String argDesc(int index)
        {
            return (index >= 0) ? cmdArgDesc[index] : null;
        }


       /**
        ***********************************************************************
        **
        **  Gets the lookup tables for a command, given its index
        **
        ***********************************************************************
        */
        public Lookup[] lookup(int index)
        {
            return (index >= 0) ? cmdLookup[index] : null;
        }


       /**
        ***********************************************************************
        **
        **  Gets the help for a command, given its index
        **
        ***********************************************************************
        */
        public String[] help(int index)
        {
            return (index >= 0) ? cmdHelp[index] : null;
        }


       /**
        ***********************************************************************
        **
        **  Shows help for a command, given its index
        **
        ***********************************************************************
        */
        public void showHelp(int index)
        {
            String[] text = cmdHelp[index];
            if (text == null) return;
            out.println("Description:");
            out.println("  " + text[0]);
            out.println("Usage:");
            out.println("  " + text[1]);
            if (text.length > 2)
                out.println("Parameters:");
            for (int j = 2; j < text.length; j++)
                out.println("  " + text[j]);
            
        }


       /**
        ***********************************************************************
        **
        **  Gets the maximum length of matching commands
        **
        ***********************************************************************
        */
        public int getMaxLeng(String command)
        {
            int maxLeng = 0;
            for (int j = 0; j < nCmnd; j++) {
                String name = cmdName[j];
                if (name == null) continue;
                if (command == null || name.startsWith(command))
                    if (name.length() > maxLeng) maxLeng = name.length();
            }

            return maxLeng;
        }


       /**
        ***********************************************************************
        **
        **  Shows help summary for matching commands
        **
        ***********************************************************************
        */
        public void showHelpSumm(String command)
        {
            showHelpSumm(command, getMaxLeng(command));
        }

        public void showHelpSumm(String command, int maxLeng)
        {
            StringBuilder blanks = new StringBuilder(maxLeng + 2);
            for (int j = 0; j < blanks.capacity(); j++) blanks.append(" ");
            for (int j = 0; j < nCmnd; j++) {
                String name = cmdName[j];
                if (name == null) continue;
                if (command == null || name.startsWith(command)) {
                    String[] text = cmdHelp[j];
                    if (text == null) continue;
                    out.format("  %s%s%s\n", name,
                               blanks.substring(name.length()), text[0]);
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement lookup tables
    **
    ***************************************************************************
    */
    public static class Lookup {

        private int nItem = 0;
        private String[] names;
        private int[] codes;


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public Lookup(int count)
        {
            codes = new int[count];
            names = new String[count];
        }


       /**
        ***********************************************************************
        **
        **  Adds a new entry
        **
        ***********************************************************************
        */
        public void add(String name, int code)
        {
            names[nItem] = name;
            codes[nItem] = code;
            nItem++;
        }


       /**
        ***********************************************************************
        **
        **  Encodes a name to its equivalent code
        **
        ***********************************************************************
        */
        public int encode(String name, boolean report)
        {
            int index = 0, match = 0;
            String lcName = name.toLowerCase();

            for (int j = 0; j < names.length; j++) {
                String tName = names[j].toLowerCase();
                if (tName.startsWith(lcName)) {
                    index = j;
                    match++;
                    if (tName.equals(lcName)) {
                        match = 1;
                        break;
                    }
                }
            }
            if (match == 1) return codes[index];
            if (report) out.println((match == 0 ? "Unrecognized" : "Ambiguous")
                                      + " value: " + name);
            return (match == 0) ? -1 : -2;
        }


       /**
        ***********************************************************************
        **
        **  Decodes a code to its equivalent name
        **
        ***********************************************************************
        */
        public String decode(int code)
        {
            for (int j = 0; j < codes.length; j++)
                if (code == codes[j]) return names[j];
            return "Invalid";
        }


       /**
        ***********************************************************************
        **
        **  Gets the number of items
        **
        ***********************************************************************
        */
        public int count()
        {
            return nItem;
        }


       /**
        ***********************************************************************
        **
        **  Gets the code given its index
        **
        ***********************************************************************
        */
        public int code(int index)
        {
            return codes[index];
        }


       /**
        ***********************************************************************
        **
        **  Gets the name given its index
        **
        ***********************************************************************
        */
        public String name(int index)
        {
            return names[index];
        }
    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement common command processing
    **
    ***************************************************************************
    */
    private final static int
        CMD_HELPSUMM    = 0,
        CMD_HELP        = 1,
        CMD_EXIT        = 2;

    private final static String[] helpHelp = {
        "Display help",
        "help  [<command>]",
        "command    Optional command for which help is wanted",
    };

    private final static String[] helpExit = {
        "Exit the program",
        "exit",
    };

    private final static String[] helpQuit = {
        "Exit the program",
        "quit",
    };

    private final static Command cmnd = new Command(4);
    static {
        cmnd.add("help",  CMD_HELP,      helpHelp, "s");
        cmnd.add("?",     CMD_HELPSUMM,  null,     "");
        cmnd.add("exit",  CMD_EXIT,      helpExit, "");
        cmnd.add("quit",  CMD_EXIT,      helpQuit, "");
    }


    private class Common implements Dispatch {

       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public Common()
        {
            add(this, cmnd);
        }


       /**
        ***********************************************************************
        **
        **  Dispatches common commands
        **
        ***********************************************************************
        */
        @Override
        public boolean dispatch(int code, int found, Object[] args)
        {
            boolean cont = true;

            switch (code) {

            case CMD_HELP:
                if (found != 0) {
                    String command = ((String)args[0]).toLowerCase();
                    int index = findCmnd(command);
                    if (index >= 0)
                        cmnds.get(index >> 16).showHelp(index & 0xffff);
                    else if (index == CMD_AMBIG) {
                        out.println("Matching commands are:");
                        showHelpSumm(command);
                    }
                    else
                        out.println("No matching commands");
                    break;
                }

            case CMD_HELPSUMM:
                out.println("Valid commands are:");
                showHelpSumm(null);
                break;

            case CMD_EXIT:
                cont = false;
                break;
            }

            return cont;
        }

    }


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public CmndProcess()
    {
        new Common();
    }


   /**
    ***************************************************************************
    **
    **  Adds a command processor
    **
    ***************************************************************************
    */
    public void add(Dispatch disp, Command cmnd)
    {
        for (int j = 0; j < cmnd.count(); j++) {
            int code = findCmnd(cmnd.name(j));
            if (code >= 0) cmnd.disable(j);
        }
        disps.add(disp);
        cmnds.add(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Processes a command line
    **
    ***************************************************************************
    */
    public boolean process(String line)
    {
        Scanner scan = new Scanner(line);
        if (!scan.hasNext()) return true;
        String command = scan.next().toLowerCase();
        int index = findCmnd(command);
        if (index < 0) {
            out.println((index == CMD_UNKNOWN ? "Unknown" : "Ambiguous")
                          + " command (" + command + ")");
            return true;
        }
        int tIndex = index >> 16, cIndex = index & 0xffff;
        Command cmd = cmnds.get(tIndex);
        int code = cmd.code(cIndex);
        String argDesc = cmd.argDesc(cIndex);
        Object[] args = new Object[argDesc.length()];
        int found = scanArgs(scan, argDesc, cmd.lookup(cIndex), args);
        if (found < 0) return true;
        return disps.get(tIndex).dispatch(code, found, args);
    }


   /**
    ***************************************************************************
    **
    **  Scans the arguments in a command line
    **
    ***************************************************************************
    */
    private final static String[] typeDesc =
        {"string", "byte", "short", "integer", "long", "keywdint", "keyword",
         "float", "double"};

    private static int scanArgs(Scanner scan, String types, Lookup[] lookup,
                                Object[] args)
    {
        int found = 0, kInc = 0;

        for (int j = 0, k = 0; j < types.length(); j++, k += kInc) {
            String type = types.substring(j, j + 1);
            String upType = type.toUpperCase();
            boolean required = type.equals(upType);
            int iType = "SBWILJKFDE".indexOf(upType);

            if (iType < 0) {
                out.println("Invalid argument type specified (" + type + ")");
                return -1;
            }

            kInc = (iType == 5 || iType == 6) ? 1 : 0;

            if (!scan.hasNext()) {
                if (required) {
                    out.println("Missing argument (" + (j + 1) + ")");
                    return -1;
                }
                continue;
            }

            String arg;
            if (iType == 9) {
                scan.skip(" *");
                arg = scan.findInLine(".*");
            }
            else {
                arg = scan.next();
                if (!required && arg.equals("*")) continue;
            }

            try {
                if (iType == 0 || iType == 9)
                    args[j] = arg;
                else if (iType < 6) {
                    long value = Long.decode(arg);
                    long sMask = -1L << ((iType == 5) ? 32 : 1 << (iType + 2));
                    long sBits = value & sMask;
                    if ((value >= 0 && sBits != 0) ||
                        (value < 0 && (sBits != sMask || value == sMask)))
                        throw new NumberFormatException();
                    if (iType == 1)
                        args[j] = (byte)value;
                    else if (iType == 2)
                        args[j] = (short)value;
                    else if (iType == 4)
                        args[j] = value;
                    else
                        args[j] = (int)value;
                }
                else if (iType == 6)
                    throw new NumberFormatException();
                else if (iType == 7)
                    args[j] = Float.valueOf(arg);
                else
                    args[j] = Double.valueOf(arg);
            }
            catch (NumberFormatException e) {
                int value = -1;
                if (iType == 5 || iType == 6) {
                    value = lookup[k].encode(arg, false);
                    if (value >= 0)
                        args[j] = value;
                }
                if (value < 0) {
                    out.println((value == -1 ? "Invalid " : "Ambiguous ")
                                + typeDesc[iType] + " argument (" + arg + ")");
                    return -1;
                }
            }
            found |= (1 << j);
        }

        if (scan.hasNext()) {
            out.println("Too many arguments");
            return -1;
        }

        return found;
    }


   /**
    ***************************************************************************
    **
    **  Finds a command in the set of command definitions
    **
    ***************************************************************************
    */
    private int findCmnd(String command)
    {
        int index = CMD_UNKNOWN, cIndx = 0;

        for (int j = 0; j < cmnds.size(); j++) {
            int tIndex = cmnds.get(j).find(command);
            if (tIndex >= 0) {
                index = tIndex;
                cIndx = j;
                break;
            }
            if (tIndex == CMD_AMBIG) index = CMD_AMBIG;
            if (tIndex < CMD_AMBIG) {
                if (index == CMD_UNKNOWN) {
                    index = tIndex;
                    cIndx = j;
                }
                else index = CMD_AMBIG;
            }
        }

        if (index < CMD_AMBIG) index = CMD_AMBIG - 1 - index;
        if (index >= 0) index |= (cIndx << 16);

        return index;
    }


   /**
    ***************************************************************************
    **
    **  Displays help summary for matching commands
    **
    ***************************************************************************
    */
    private void showHelpSumm(String command)
    {
        int maxLeng = 0;
        for (int j = 0; j < cmnds.size(); j++) {
            int leng = cmnds.get(j).getMaxLeng(command);
            if (leng > maxLeng) maxLeng = leng;
        }
        for (int j = 0; j < cmnds.size(); j++)
            cmnds.get(j).showHelpSumm(command, maxLeng);
    }

}
