package org.lsst.ccs.utilities.sa;

/**
 ***************************************************************************
 **
 **  \file  CmndProc.java
 **
 **  Class to implement command processing
 **
 **  \author Owen Saxton
 **
 ***************************************************************************
 */
import java.util.LinkedList;
import java.util.Scanner;
import java.io.PrintStream;

public class CmndProc {

    public final static int
        CMD_AMBIG       = -2,
        CMD_UNKNOWN     = -1;

    private final static int
        CMD_HELPSUMM    = 0,
        CMD_HELP        = 1,
        CMD_EXIT        = 2;

    private final static String[] helpHelp = {
        "Display help",
        "help  [<command>]",
        "command    Optional command for which help is wanted",
    };

    private final static String[] helpExit = {
        "Exit the program",
        "exit",
    };

    private final static String[] helpQuit = {
        "Exit the program",
        "quit",
    };

    private final static PrintStream out = System.out;

    private LinkedList<Dispatch> disps = new LinkedList<Dispatch>();
    private LinkedList<Command> cmnds = new LinkedList<Command>();


   /**
    ***************************************************************************
    **
    **  Inner interface to define command dispatching
    **
    ***************************************************************************
    */
    public interface Dispatch {

        boolean dispatch(int code, Scanner scan);
    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement command decoding
    **
    ***************************************************************************
    */
    public static class Command {

        private int nCmnd = 0;
        private int[] cmdCode;
        private String[] cmdName;
        private String[][] cmdHelp;


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public Command(int count)
        {
            cmdCode = new int[count];
            cmdName = new String[count];
            cmdHelp = new String[count][];
        }


       /**
        ***********************************************************************
        **
        **  Add a command to the table
        **
        ***********************************************************************
        */
        public void add(String command, int code, String[] help)
        {
            cmdName[nCmnd] = command;
            cmdCode[nCmnd] = code;
            cmdHelp[nCmnd] = help;
            nCmnd++;
        }


       /**
        ***********************************************************************
        **
        **  Get the number of commands
        **
        ***********************************************************************
        */
        public int count()
        {
            return nCmnd;
        }


       /**
        ***********************************************************************
        **
        **  Get the index of a command, given its name
        **
        **  This routine compares the command string with the first part of
        **  all the table entries and returns one of the following:
        **
        **    index (>= 0) If the command is fully matched
        **    CMD_UNKNOWN  If the command is not in the table
        **    CMD_AMBIG    If the command matches more than one entry
        **    CMD_AMBIG - 1 - index  If the command is partially matched
        **
        ***********************************************************************
        */
        public int find(String command)
        {
            int index = 0, match = 0;

            for (int j = 0; j < nCmnd; j++) {
                String name = cmdName[j];
                if (name != null && name.startsWith(command)) {
                    if (name.equals(command)) return j;
                    index = j;
                    match++;
                }
            }

            if (match == 0) return CMD_UNKNOWN;
            if (match > 1) return CMD_AMBIG;

            return CMD_AMBIG - 1 - index;
        }


       /**
        ***********************************************************************
        **
        **  Disable a command, given its index
        **
        ***********************************************************************
        */
        public void disable(int index)
        {
            if (index >= 0) cmdName[index] = null;
        }


       /**
        ***********************************************************************
        **
        **  Get the name of a command, given its index
        **
        ***********************************************************************
        */
        public String name(int index)
        {
            return (index >= 0) ? cmdName[index] : null;
        }


       /**
        ***********************************************************************
        **
        **  Get the code for a command, given its index
        **
        ***********************************************************************
        */
        public int code(int index)
        {
            return (index >= 0) ? cmdCode[index] : index;
        }


       /**
        ***********************************************************************
        **
        **  Get the help for a command, given its index
        **
        ***********************************************************************
        */
        public String[] help(int index)
        {
            return (index >= 0) ? cmdHelp[index] : null;
        }


       /**
        ***********************************************************************
        **
        **  Show help for a command, given its index
        **
        ***********************************************************************
        */
        public void showHelp(int index)
        {
            String[] text = cmdHelp[index];
            if (text == null) return;
            out.println("Description:");
            out.println("  " + text[0]);
            out.println("Usage:");
            out.println("  " + text[1]);
            if (text.length > 2)
                out.println("Parameters:");
            for (int j = 2; j < text.length; j++)
                out.println("  " + text[j]);
            
        }


       /**
        ***********************************************************************
        **
        **  Show help summary for matching commands
        **
        ***********************************************************************
        */
        public void showHelpSumm(String command)
        {
            for (int j = 0; j < nCmnd; j++) {
                String name = cmdName[j];
                if (name == null) continue;
                if (command == null || name.startsWith(command)) {
                    String[] text = cmdHelp[j];
                    if (text == null) continue;
                    out.format("  %-16s%s\n", name, text[0]);
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement lookup tables
    **
    ***************************************************************************
    */
    public static class Lookup {

        private int nItem = 0;
        private String[] names;
        private int[] codes;


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public Lookup(int count)
        {
            codes = new int[count];
            names = new String[count];
        }


       /**
        ***********************************************************************
        **
        **  Add a new entry
        **
        ***********************************************************************
        */
        public void add(String name, int code)
        {
            names[nItem] = name;
            codes[nItem] = code;
            nItem++;
        }


       /**
        ***********************************************************************
        **
        **  Encode a name to its equivalent code
        **
        ***********************************************************************
        */
        public int encode(String name, boolean report)
        {
            int index = 0, match = 0;
            String lcName = name.toLowerCase();

            for (int j = 0; j < names.length; j++) {
                String tName = names[j].toLowerCase();
                if (tName.startsWith(lcName)) {
                    index = j;
                    match++;
                    if (tName.equals(lcName)) {
                        match = 1;
                        break;
                    }
                }
            }
            if (match == 1) return codes[index];
            if (report) out.println((match == 0 ? "Unrecognized" : "Ambiguous")
                                      + " value: " + name);
            return (match == 0) ? -1 : -2;
        }


       /**
        ***********************************************************************
        **
        **  Decode a code to its equivalent name
        **
        ***********************************************************************
        */
        public String decode(int code)
        {
            for (int j = 0; j < codes.length; j++)
                if (code == codes[j]) return names[j];
            return "Invalid";
        }


       /**
        ***********************************************************************
        **
        **  Get the number of items
        **
        ***********************************************************************
        */
        public int count()
        {
            return nItem;
        }


       /**
        ***********************************************************************
        **
        **  Get the code given its index
        **
        ***********************************************************************
        */
        public int code(int index)
        {
            return codes[index];
        }


       /**
        ***********************************************************************
        **
        **  Get the name given its index
        **
        ***********************************************************************
        */
        public String name(int index)
        {
            return names[index];
        }
    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement common command processing
    **
    ***************************************************************************
    */
    private class Common implements Dispatch {


       /**
        ***********************************************************************
        **
        **  Constructor
        **
        ***********************************************************************
        */
        public Common()
        {
            Command cmnd = new Command(4);
            cmnd.add("help",        CMD_HELP,        helpHelp);
            cmnd.add("?",           CMD_HELPSUMM,    null);
            cmnd.add("exit",        CMD_EXIT,        helpExit);
            cmnd.add("quit",        CMD_EXIT,        helpQuit);

            add(this, cmnd);
        }


       /**
        ***********************************************************************
        **
        **  Dispatch common commands
        **
        ***********************************************************************
        */
        @Override
        public boolean dispatch(int code, Scanner scan)
        {
            int found;
            boolean cont = true;
            Object[] args = new Object[1];

            switch (code) {

            case CMD_HELP:
                if ((found = scanArgs(scan, "s", args)) == -1) break;
                if (found != 0) {
                    String command = ((String)args[0]).toLowerCase();
                    int index = findCmnd(command);
                    if (index >= 0)
                        cmnds.get(index >> 16).showHelp(index & 0xffff);
                    else if (index == CMD_AMBIG) {
                        out.println("Matching commands are:");
                        showHelpSumm(command);
                    }
                    else
                        out.println("No matching commands");
                    break;
                }

            case CMD_HELPSUMM:
                out.println("Valid commands are:");
                showHelpSumm(null);
                break;

            case CMD_EXIT:
                cont = false;
                break;
            }

            return cont;
        }

    }


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public CmndProc()
    {
        new Common();
    }


   /**
    ***************************************************************************
    **
    **  Add a command processor
    **
    ***************************************************************************
    */
    public void add(Dispatch disp, Command cmnd)
    {
        for (int j = 0; j < cmnd.count(); j++) {
            int code = findCmnd(cmnd.name(j));
            if (code >= 0) cmnd.disable(j);
        }
        disps.add(disp);
        cmnds.add(cmnd);
    }


   /**
    ***************************************************************************
    **
    **  Process a command line
    **
    ***************************************************************************
    */
    public boolean process(String line)
    {
        Scanner scan = new Scanner(line);
        if (!scan.hasNext()) return true;
        String command = scan.next().toLowerCase();
        int index = findCmnd(command);
        if (index < 0) {
            out.println((index == CMD_UNKNOWN ? "Unknown" : "Ambiguous")
                          + " command (" + command + ")");
            return true;
        }
        int code = cmnds.get(index >> 16).code(index & 0xffff);
        return disps.get(index >> 16).dispatch(code, scan);
    }


   /**
    ***************************************************************************
    **
    **  Scan the arguments in a command line
    **
    ***************************************************************************
    */
    private final static String[] typeDesc =
        {"string", "byte", "short", "integer", "float", "double"};

    public static int scanArgs(Scanner scan, String types, Object[] args)
    {
        int found = 0;

        for (int j = 0; j < types.length(); j++) {
            String type = types.substring(j, j + 1);
            String upType = type.toUpperCase();
            boolean required = type.equals(upType);
            int iType = "SBWIFD".indexOf(upType);

            if (iType < 0) {
                out.println("Invalid argument type specified (" + type + ")");
                return -1;
            }

            if (!scan.hasNext()) {
                if (required) {
                    out.println("Missing argument (" + (j + 1) + ")");
                    return -1;
                }
                continue;
            }

            String arg = scan.next();
            if (!required && arg.equals("*")) continue;

            try {
                if (iType == 0)
                    args[j] = arg;
                else if (iType < 4) {
                    long value = Long.decode(arg);
                    long sMask = -1L << (1 << (iType + 2));
                    long sBits = value & sMask;
                    if ((value >= 0 && sBits != 0) ||
                        (value < 0 && (sBits != sMask || value == sMask)))
                        throw new NumberFormatException();
                    if (iType == 1)
                        args[j] = Byte.valueOf((byte)value);
                    else if (iType == 2)
                        args[j] = Short.valueOf((short)value);
                    else
                        args[j] = Integer.valueOf((int)value);
                }
                else {
                    if (iType == 4)
                        args[j] = Float.valueOf(arg);
                    else
                        args[j] = Double.valueOf(arg);
                }
                found |= (1 << j);
            }
            catch (NumberFormatException e) {
                out.println("Invalid " + typeDesc[iType] + " argument ("
                              + arg + ")");
                return -1;
            }
        }

        if (scan.hasNext()) {
            out.println("Too many arguments");
            return -1;
        }

        return found;
    }


   /**
    ***************************************************************************
    **
    **  Find a command in the set of command definitions
    **
    ***************************************************************************
    */
    private int findCmnd(String command)
    {
        int index = CMD_UNKNOWN, cIndx = 0;

        for (int j = 0; j < cmnds.size(); j++) {
            int tIndex = cmnds.get(j).find(command);
            if (tIndex >= 0) {
                index = tIndex;
                cIndx = j;
                break;
            }
            if (tIndex == CMD_AMBIG) index = CMD_AMBIG;
            if (tIndex < CMD_AMBIG) {
                if (index == CMD_UNKNOWN) {
                    index = tIndex;
                    cIndx = j;
                }
                else index = CMD_AMBIG;
            }
        }

        if (index < CMD_AMBIG) index = CMD_AMBIG - 1 - index;
        if (index >= 0) index |= (cIndx << 16);

        return index;
    }


   /**
    ***************************************************************************
    **
    **  Display help summary for matching commands
    **
    ***************************************************************************
    */
    private void showHelpSumm(String command)
    {
        for (int j = 0; j < cmnds.size(); j++)
            cmnds.get(j).showHelpSumm(command);
    }

}
