package org.lsst.ccs.utilities.dispatch;


import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.logging.Level;

/**
 * Dispatches a Command object to registered "<I>executants</I>" able to handle the command;
 * each command execution may be handled by a different Thread so every registered agent does not have to wait
 * for termination of code from the previous  call.
 * <P>
 * <B>BEWARE</B> :
 * <UL>
 * <LI>
 *     The default constructor implementation uses a "cached" Thread Pool :
 *     <UL>
 *         <LI>
 *             It is not guaranteed that events will be really executed in parallel.
 *             A fixed Thread pool knowing about the number of processors  or a pure sequential code may be
 *             more efficient: test! (and ask for changes!)
 *             <BR>
 *             Better use this class if the Command's code takes quite a lot of time to execute.
 *         <LI>
 *             The Thread pool may "choke"  if commands to be executed are registered at a higher rate than
 *             they can be executed.
 *     </UL>
 * <LI>
 *     The Threads handling the commands are deamon Threads. This may be a problem if you need an application Thread ...
 *     Moreover it is not guaranteed that all commands will be executed if the JVM abruptly terminates.
 *     <BR>
 *      This can be changed .... just ask!
 *
 * </UL>
 *
 *
 * @param <T> the nature of the code able to handle the <TT>CommandFor&lt;T&gt;</TT> Objects
 * @author bernard AMADE
 */
public class ParallelCommandDispatcher<T> {

    /**
     * Object of this embedded class are for registering a task to be handled by the executing Threads.
     * @param <T2> code able to handle the corresponding <TT>CommandFor</TT> Objects.
     */
    static class DoIt<T2> implements Runnable {
        public final T2 executant ;
        public final CommandFor<T2> command ;

        DoIt(T2 executant, CommandFor<T2> command) {
            //System.out.println("#### command submitted : " + command);
            this.executant = executant;
            this.command = command;
        }

        @Override
        public void run() {
            try{
                //System.out.println("#### command   executed " + command  );
                command.invokeOn(executant);
            } catch (Exception exc) {
               PackCst.CURLOG.log(Level.SEVERE, "while executing :", exc);
            }
        }
    }

    /**
     *   ExecutorService that deals with the Threads handling the commands.
     */
    protected  ExecutorService loop ;

    /**
     * List of all registered agent that may take action.
     */
     protected CopyOnWriteArrayList<T> list = new java.util.concurrent.CopyOnWriteArrayList<T>() ;


    /**
     * Creates an asynchronous Dispatcher for Commands.
     * Uses the standard common ExecutorService for this package.
     */
    public ParallelCommandDispatcher() {
        this.loop = PackCst.EXECUTOR_SERVICE ;
    }

    /**
     * Creates an asynchronous Dispatcher for Commands: uses a specific ExecutorService.
     */
    public ParallelCommandDispatcher(ExecutorService execService) {
        this.loop = execService ;
    }

    public ParallelCommandDispatcher(boolean singleThread) {
        if(singleThread) {
            this.loop   =  PackCst.SINGLE_THREAD ;
        } else {
            this.loop = PackCst.EXECUTOR_SERVICE ;
        }
    }
    /**
     * adds an executant to the list of those going to execute Commands when needed.
     * @param executant
     */
    public void addExecutant(T executant) {
        list.add(executant) ;
    }

    /**
     * removes an executant form the list
     *
     * @param executant
     */

    public void removeExecutant(T executant) {
        list.remove(executant) ;
    }

    /**
     * Send a Command request for all registered executants.
     * @param command
     */
     public void dispatchCommand(CommandFor<T> command) {
         //System.out.println("# dispatching to " + list.size());
        for(T executant : list) {
            loop.submit(new DoIt<T>(executant, command)) ;
        }
    }
}
