package org.lsst.ccs.utilities.dispatch;

/**
 * This interface defines the behaviour of a class implementing a Command pattern.
 * <P>
 * Example:
 * <UL>
 *     Lets define a class <TT>Doer</TT> that has a method <TT>show(String string, int anInt)</TT>.
 *     When do we need to send a "command" to such an Object instead of just calling directly this method?
 *     It could be that the code that "relays" this invocation does not know much about the sender , the receiver
 *     and the method itself.
 *     <BR>
 *      for instance  we may want to send a "command" to such an Object through the wire and invoke the method across
 *      the network (without predefined RMI) or we have a generic code that dispatches a call to many receivers.
 *      <BR>
 *      In programming parlance we are going to "<I>reify</I>" the method call: that is transform it into an Object.
 *      <BR>
 *      So the object representing this method call and implementing the Command could look like that:
 *      <PRE>
 *          class ShowCommand implements CommandFor<Doer> , Serializable{
 *              String arg1 ;
 *              int arg2 ;
 *              // constructor and everything
 *              public void invokeOn(Doer someInstance) {
 *                  someInstance.show(arg1, arg2)
 *              }
 *          }
 *      </PRE>
 *      Now suppose the receiving code reads the command from a Stream
 *      <PRE>
 *          Doer actor ; // initialized
 *          // then later
 *          ShowCommand command = (ShowCommand) objectStream.readObject() ;
 *          command.invokeOn(actor); // and that's it!
 *      </PRE>
 * </UL>
 *@see org.lsst.ccs.utilities.dispatch.SynchronousCommandFor
 *@see org.lsst.ccs.utilities.dispatch.SynchronousCommandDispatcher
 *@see org.lsst.ccs.utilities.dispatch.ParallelCommandDispatcher
 *
 *@author bernard Amade
 */
public interface CommandFor<T> {
    public  void invokeOn(T instance);
}
