package org.lsst.ccs.utilities.dispatch;


import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;

/**
 * Registers codes interested in receiving commands.
 * Each code should know that it is handling an instance of <TT>SynchronousCommandFor</TT>
 * and should behave accordingly.
 * <P>
 * The <TT>dispatchCommand</TT>   methods execute this command sequentially on each
 * "<I>executant</I>" object. The order of calls is not specified.
 *
 * @author bernard AMADE
 */
public class SynchronousCommandDispatcher<T> {
    protected CopyOnWriteArrayList<T> list = new java.util.concurrent.CopyOnWriteArrayList() ;

    /**
     * Registers a code ready to receive "synchronous" commands
     * @param executant
     */
    public  void addExecutant(T executant) {
        list.add(executant) ;
    }

    /**
     * Remove the code from the listeners list
     *
     * @param executant
     */
    public  void removeExecutant(T executant){
        list.remove(executant) ;
    }

/**
 * Sends a command to all registered codes ready to handle it.
 * <P>
 *     If an exception is  fired it is catched and logged.
 *     proper handling of Exception should be performed by the implementing commands.
 * </P>
 */
    public void dispatchCommand(SynchronousCommandFor<? super T> command) {
        for(T executant : list) {
            try{
                command.invokeOn(executant);
            } catch (Exception exc) {
               PackCst.CURLOG.log(Level.SEVERE, "while dispatching :", exc);
            }
        }
    }
}
