/**
 * This package provides  proxies to the standard java.util.logging package.
 * <BR/>
 * Its <TT>Logger</TT> class is only for codes that log messages or data (not for codes that deal with <TT>Level</TT>,
 * <TT>Formatters</TT> or <TT>Handlers</TT>). It provides methods that can simply replace logging invocations
 * meant to be handled by <TT>Log4J</TT> or simple standard java logging invocations: you can then in your codes
 * just change the <TT>import</TT> directives and keep you code "as is".
 * <P/>
 * So you have for instance :
 * <PRE>
 *     Logger logger = Logger.getLogger("org.lsst.ccs.mypackage") ;
 *     //....
 *     logger.info("doing the right thing!") ;
 *     //...
 *     if( logger.isInfoEnabled() ) {
 *         logger.log(Level.INFO, "still better", methodCall()) ;
 *     }
 *     //...
 *     try {
 *         //...
 *     } catch(Exception exc) {
 *         logger.error("Oops!", exc) ;
 *     }
 * </PRE>
 * But the <TT>Logger</TT> class is not only to get syntactic sugar: it provides access to "multi-dimensional logging".
 * <P/>
 * What's that?
 * <P/>
 * The idea is that can log the same call to different loggers in a single invocation.
 * So, for instance, you have the usual logger linked to a package but you want also to address different "concerns" that
 * are common to different classes or package. Then you can write:
 * <PRE>
 *     logger.log(Level.INFO, "Object being initialized", theObject, "INIT");
 * </PRE>
 * this means that the created <TT>LogRecord</TT> will be submitted to the current <TT>Logger</TT>
 * and also to another one named "INIT".
 * <BR/>
 * Thus you can create many "transversal" loggers named "INIT", "CONFIG", "MOVE" (or whatever you want)
 * and then you will be able to handle log informations reporting specifically about one of this "concerns".
 * <PRE>
 *     logger.log(Level.INFO, "Object being moved at initialisation", theObject, "INIT", "MOVE");
 * </PRE>
 * Then you can set the level for the corresponding package to <TT>SEVERE</TT> and set the level
 * of the "INIT" <I>concern</I> to <TT>INFO</TT> .
 * <BR/>
 * All the logging methods of the <TT>Logger</TT> class takes an optional <TT>String... </TT> variable argument list
 * where you can name <I>concerns</I>.
 * <P/>
 * Then you can handle real <TT>java.util.logging.Loggers</TT> that bear a name of package or a name of <I>concern</I>.
 * Then you can use <TT>setLevel</TT> or <TT>addHandler</TT> methods to this actual loggers (of type <TT>java.util.logging.Logger</TT>).
 * <B>But</B> you should note there are changes to the way <TT>Handlers</TT> should be defined and to the configuration
 * file (if you want to pilot logging through a configuration file).
 * <H3>Handler design</H3>
 * Since the same <TT>LogRecord</TT> might be provided to different <TT>Loggers</TT> then
 * a <TT>Handler</TT> may receive many <TT>publish</TT> requests for the <B>same</B> <TT>LogRecord</TT>!
 * It means that the <TT>Handlers</TT> should memorize which records are published and not report twice the same record.
 * <BR/>
 * Special handlers are provided to check for duplicates: <TT>ConsoleHandlerN, StreamHandlerN, SocketHandlerN, ...</TT>
 * all rely on the service of a <TT>IsLoggableDelegate</TT> that checks the records for multiple publishing.
 *
 * <H3>Configuration file</H3>
 * There is a <TT>logging.properties</TT> file to define the behaviour of loggers ans handlers:
 * though this file is in the standard format of <TT>java.util.logging</TT> it is positioned in a different way:
 * <UL>
 *     <LI/> it is considered a better policy to put the file at a global level of the CLASSPATH.
 *     ("/logging.propertie" in resource handling syntax).
 *     <LI/> to obtain this behaviour call <TT>Class.forName("org.lsst.ccs.utilities.logging.LogManagement</TT>
 *     early in your <TT>main</TT> (this will call <TT>static</TT> initialization code)
 * </UL>
 * Example of a <TT>logging.properties</TT> file:
 * <PRE>
*
 # Properties file which configures the operation of the CCS
 # logging facility.

 # Global logging properties.
 # ------------------------------------------
 # The set of handlers to be loaded upon startup.
 # Comma-separated list of class names.
 # (? LogManager docs say no comma here, but JDK example has comma.)
 # do not put space characters in this list!
 # handlers are loaded by the primordial log manager

 handlers=
 ## use HandlersN to avoid classLoading problems
 ## these are loaded by the ClassLoader that knows about thoses classes
 handlersN=org.lsst.ccs.utilities.logging.ConsoleHandlerN,org.lsst.ccs.utilities.logging.FileHandlerN

 ## BEWARE: you CAN'T set  org.lsst.ccs.bus.utils.LogBusHandler HERE!
 ## because it is initialized later (when the buses are activated)

 # Default global logging level.
 # Loggers and Handlers may override this level
 # SEE LSSTCCS-290
 .level=WARNING

 #The level of the CCS Root logger LSSTCCS-297
 org.lsst.ccs.level=INFO
 # Loggers
 # ------------------------------------------
 # Loggers are usually attached to packages.
 # Here, the level for each package is specified.
 # The global level is used by default, so levels
 # specified here simply act as an override.
 #myapp.ui.level=ALL
 #myapp.business.level=CONFIG
 #myapp.data.level=SEVERE


 # Handlers
 # -----------------------------------------

 # --- ConsoleHandler ---
 # Override of global logging level
 #java.util.logging.ConsoleHandler.level=SEVERE

 org.lsst.ccs.utilities.logging.ConsoleHandlerN.level=INFO

 ## now you can set the level of the LogBusHandler here
 org.lsst.ccs.bus.utils.LogBusHandler.level=WARNING

 # --- FileHandler ---
 # Override of global logging level

 org.lsst.ccs.utilities.logging.FileHandlerN.level=ALL

 # Naming style for the output file:
 # (The output file is placed in the directory
 # defined by the "user.home" System property.)
 # use %t if temporary directory to be used (instead of %h)
 # see below other CCS options
 # org.lsst.ccs.utilities.logging.FileHandlerN.pattern=%h/ccs/ccs-logs-%u.log

 #org.lsst.ccs.utilities.logging.FileHandlerN.pattern=%t/ccs-logs-%u.log

 # the case where we use our own log property from Property "org.lsst.ccs.logdir"

 org.lsst.ccs.utilities.logging.FileHandlerN.pattern=%L/ccs-logs-%A.log

 # the case where we use our own log property from Property "org.lsst.ccs.workdir"

 #org.lsst.ccs.utilities.logging.FileHandlerN.pattern=%W/logs/ccs-logs-%u.log

 # Limiting size of output file in bytes:
 org.lsst.ccs.utilities.logging.FileHandlerN.limit=5000000

 # Number of output files to cycle through, by appending an
 # integer to the base file name:
 org.lsst.ccs.utilities.logging.FileHandlerN.count=20

 # Style of output (Simple or XML):
 org.lsst.ccs.utilities.logging.FileHandlerN.formatter=java.util.logging.SimpleFormatter
 # a special formatter that deals with StackTraces
 org.lsst.ccs.bus.utils.LogBusHandler.formatter=org.lsst.ccs.utilities.logging.TextFormatter
 org.lsst.ccs.utilities.logging.ConsoleHandlerN.formatter=org.lsst.ccs.utilities.logging.TextFormatter
 # change that one if you want to modify the way StackTraces are printed
 # negative value means all the stacktrace will be printed
 StackTraceFormats.depth=2
 #org.lsst.ccs.utilities.logging.FileHandlerN.formatter=java.util.logging.XMLFormatter

 # Example to customize the SimpleFormatter output format
 # to print one-line log message like this:
 #     level: log message [date/time]
 #
 java.util.logging.SimpleFormatter.format=%4$s: %5$s [%1$tc]%n
 # index starts at 1 : date, source, Logger, Level, message, throwableStr
 # here we have source :log_message throwable
 org.lsst.ccs.utilities.logging.TextFormatter.format=%4$s: %5$s[%2$s]%n%6$s

 </PRE>
 * @ImplNote
 * the difficult part of this package lies in the <TT>ContextLogManager</TT> class.
 * The JUL standard classes have many drawbacks:
 * <UL>
 *     <LI/>  implementation of base classes (such as <TT>StreamHandler</TT>) rely on the code of <TT>LogManager</TT>
 *     which is inaccessible outside the package. So reading properties in the logging.properties is almost impossible
 *     from codes (outside JUL package).
 *     <LI/> the instances of Handlers should be lazily loaded : the LogManager should have finished to be constructed
 *     to have Handler registration (again the base classes rely on <TT>LogManager.getManager()</TT> which can be null
 *     if your class is the LogManager and has not finished registration.)
 * </UL>
 * So the <TT>ContextLogManager</TT> class should be loaded as soon as possible either by being declared as
 * <TT>java.util.logging.Manager</TT> or through load-time initialization (class <TT>LogManagement</TT>).
 * The class reads the logging.properties with special entry for handlers (named "handlerN") that avoid
 * to have the standard logManager load classes that he can't initialize properly.
 * <BR/>
 *  The properties object read from logging.properties should be passed along the properties
 *  initialization of the <TT>LogManager</TT> base class (if the current class IS the LogManager by calling the
 *  super.read of properties and if it is not then the properties are forcibly fed upon the LogManager).
 *  <BR/>
 *  the properties have two different parts:
 *  <UL>
 *      <LI/> the logger configuration which can be run as soon as possible (when the getManager does not yield a null value)
 *      <LI/> the handler configuration which is to be run  after (but before any log of our log happens)
 *  </UL>
 *  <BR/>
 *  The codes have a dependency on <TT>ClassLoaders</TT>: if the ContextLogManager is  build by different
 *  <TT>ClassLoaders</TT> there might be duplications  and reconfiguration.
 *  So if there are many ClassLoaders that are activated at different moments in the life of the application
 *  is is advisable to make the <TT>ContextLogManager</TT> the default LogManager (by setting property <TT>java.util.logging.manager</TT>)
 */
package org.lsst.ccs.utilities.logging ;

