package org.lsst.ccs.utilities.logging;


import org.lsst.ccs.bootstrap.resources.BootstrapResourceUtils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.lang.reflect.Method;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.util.*;
import java.util.logging.*;
import java.util.logging.Formatter;


/**
 * instance of this class should be built only during load time. That is: when java.util.logging.manager property
 * is set to the name of this class, or by call of LogManagement load time in a ClassLoader.
 * <BR/>
 *  <UL>
 *      <LI/> When the class is instanciated as the default LogManager the properties that are read from
 *      "logging.properties" should be modified and passed to the super-class, but the Handlers must
 *       not be created at this stage (otherwise NullPointerException will ensue)
 *       </LI> If the class is not the defaultLogManager then the default log Manager should be reinitialized
 *       with the modified Properties.
 *  </UL>
 *  The Big Problem is the lazy initialization of Handlers (HandlersN are subsistued to the standard "handler" property)
 *
 */
public class ContextLogManager extends LogManager {
    public static final String LOG_PROPS = "/logging.properties";
    public static final String LOG_CONF_FILE = System.getProperty("java.util.logging.config.file", LOG_PROPS);

    private final class Cleaner extends Thread {

        @Override
        public void run() {
            if (useShutdownHook) {
                shutdown();
            }
        }

    }


    // ------------------------------------------------------------Constructors
    static Properties loggingProperties ;
    static Map<String, java.util.logging.Logger> loggerMap = new HashMap<>() ;
    static Map<String, Handler> handlerMap = new HashMap<>() ;
    static LogManager systemLoaderLogManager;
    static Class systemManagerClass;
    static Method systemManagerPropertyMethod;
    boolean isInInitialManagerState ;
    boolean delegateToStandardLogger ;
    LogManager stdLogManager ;
    boolean handlersLoaded ;

    public ContextLogManager() {
        super();
        try {
            //TODO : check if all handlers are cleaned
            //Runtime.getRuntime().addShutdownHook(new Cleaner());
        } catch (IllegalStateException ise) {
            // We are probably already being shutdown. Ignore this error.
        }
        stdLogManager = LogManager.getLogManager() ;
        if(stdLogManager == null) {
            isInInitialManagerState = true ;
            systemManagerClass = this.getClass();
            systemLoaderLogManager = this;
        } else if (stdLogManager instanceof ContextLogManager) {
            systemManagerClass = this.getClass();
            systemLoaderLogManager = this;
        } else {
            systemLoaderLogManager = this;
            systemManagerClass = stdLogManager.getClass();
            delegateToStandardLogger = true ;
        }

        try {
            systemManagerPropertyMethod = systemManagerClass.getMethod("getProperty", new Class[]{String.class});
        } catch (NoSuchMethodException e) {
            System.err.println("getProperty not found for ContextClassLoader");
        }
        /* */
        readInitialConfiguration(true);
        /**/ 
        
        // Default CCS Root logger LSSTCCS-297
        // comment: this may lead to a bug if ContextLogManager is tagged as Manager
        // so initialization of rootLogger is conditional
        if(!isInInitialManagerState) {
            org.lsst.ccs.utilities.logging.Logger.getLogger("");
        }
        
    }


    public static org.lsst.ccs.utilities.beanutils.Optional<LogManager> getSystemLoaderLogManager() {
        org.lsst.ccs.utilities.beanutils.Optional<LogManager> res = org.lsst.ccs.utilities.beanutils.Optional.empty();
        try {
            if (systemLoaderLogManager == null) {

                Class clazz = ClassLoader.getSystemClassLoader().loadClass("org.lsst.ccs.utilities.logging" +
                        ".ContextLogManager");
                systemManagerClass = clazz;
                /**
                 Method method = res.getMethod("getSystemLoaderLogManager") ;
                 systemLoaderLogManager = method.invoke(res) ;
                 */
                systemLoaderLogManager = (LogManager) clazz.newInstance();
                //TODO : this line not necessary if newintance
                //TODO: make an instance of a delegating singleton
                systemManagerPropertyMethod = systemManagerClass.getMethod("getProperty", new Class[]{String.class});
            }
            res = org.lsst.ccs.utilities.beanutils.Optional.of(systemLoaderLogManager);
        } catch (Exception exc) {
            System.err.println(" ERROR " + exc);
        }
        return res;
    }

    public static void loaderLazyLoadHandler() {
        org.lsst.ccs.utilities.beanutils.Optional<LogManager> optionalManager = getSystemLoaderLogManager() ;
        if(optionalManager.isPresent()) {
            LogManager manager = optionalManager.get() ;
            if(manager instanceof ContextLogManager) {
                try {
                    ((ContextLogManager) manager).handlersLazyLoad();
                } catch (Exception exc) {
                    System.err.println("error in Handlers initialization ");
                    exc.printStackTrace();
                }
            }
        }
    }
    public static String loaderGetProperty(String name) {
        getSystemLoaderLogManager();
        try {
            String res = (String) systemManagerPropertyMethod.invoke(systemLoaderLogManager, name);
            return res;
        } catch (Exception e) {
            System.err.println(" loaderGetProperty :" + e);
            return null;
        }

    }
    public static String loaderGetStringProperty(String name, String defaultValue) {
        String val = loaderGetProperty(name);
        if (val == null) {
            return defaultValue;
        }
        return val.trim();
    }
    public static int loaderGetIntProperty(String name, int defaultValue) {

        String val = loaderGetProperty(name);
        if (val == null) {
            return defaultValue;
        }
        try {
            return Integer.parseInt(val.trim());
        } catch (Exception ex) {
            return defaultValue;
        }
    }


    public static Level loaderGetLevelProperty(String name, Level defaultValue) {
        String val = loaderGetProperty(name);
        if (val == null) {
            return defaultValue;
        }
        try {
            return Level.parse(val.trim());
        } catch (Exception ex) {
            return defaultValue;
        }

    }






    public static Formatter loaderGetFormatterProperty(String name, Formatter defaultValue) {

        String val = loaderGetProperty(name);
        try {
            if (val != null) {
                Class clz = ClassLoader.getSystemClassLoader().loadClass(val);
                return (Formatter) clz.newInstance();
            }
        } catch (Exception ex) {
            // We got one of a variety of exceptions in creating the
            // class or creating an instance.
            // Drop through.
        }
        // We got an exception.  Return the defaultValue.
        return defaultValue;
    }


    // -------------------------------------------------------------- Variables



    /**
     * Determines if the shutdown hook is used to perform any necessary
     * clean-up such as flushing buffered handlers on JVM shutdown. Defaults to
     * <code>true</code> but may be set to false if another component ensures
     * that {@link #shutdown()} is called.
     */
    protected volatile boolean useShutdownHook = true;


    // ------------------------------------------------------------- Properties


    public boolean isUseShutdownHook() {
        return useShutdownHook;
    }


    public void setUseShutdownHook(boolean useShutdownHook) {
        this.useShutdownHook = useShutdownHook;
    }


    // --------------------------------------------------------- Public Methods


    /**
     * Add the specified logger to the  local configuration.
     *
     * @param logger The logger to be added
     */
    public synchronized boolean addLogger(final java.util.logging.Logger logger) {

        final String loggerName = logger.getName();
        boolean res = super.addLogger(logger) ;
        if(res == false) {
            return res ;
        }
        try {
            loggerMap.put(loggerName, logger) ;
            loadLoggerHandlers(loggerName, logger) ;
        } catch (Exception e) {
            System.err.println(" pb with logger handler " + loggerName);
            e.printStackTrace();
            return false ;
        }

        return true;
    }




    /**
     * Get an enumeration of the logger names currently defined in the
     * classloader local configuration.
     */
    public synchronized Enumeration<String> getLoggerNames() {
        //TODO use the map ?
        return super.getLoggerNames();
    }


    /**
     * Get the value of the specified property in the classloader local
     * configuration.
     *
     * @param name The property name
     */
    @Override
    public String getProperty(String name) {
        String result =loggingProperties.getProperty(name);
            if (result == null) {
                result = super.getProperty(name);
            }
        // Simple property replacement (mostly for folder names)
        if (result != null) {
            Properties props = BootstrapResourceUtils.getBootstrapSystemProperties();
            result = replace(result, props);
        }
        return result;
    }



    synchronized private void setLevelOnExistingLogger(String loggerName, java.util.logging.Logger logger) {
        Level level = getLevelProperty(loggerName + ".level",null) ;
        //LSSTCCS-296
        if ( level != null ) {
            logger.setLevel(level);
        }
    }

    
    // TO BE REMOVED IF NOT USED
//    synchronized private void setLevelsOnExistingLoggers(Properties props) {
//        Enumeration enum_ = props.propertyNames();
//        while (enum_.hasMoreElements()) {
//            String key = (String) enum_.nextElement();
//            if (!key.endsWith(".level")) {
//                // Not a level definition.
//                continue;
//            }
//            int ix = key.length() - 6;
//            String name = key.substring(0, ix);
//            Level level = getLevelProperty(key, null);
//            if (level == null) {
//                System.err.println("Bad level value for property: " + key);
//                continue;
//            }
//            java.util.logging.Logger l = getLogger(name);
//            if (l == null) {
//                continue;
//            }
//            l.setLevel(level);
//        }
//    }

    //  method to get a String property.
    // If the property is not defined we return the given
    // default value.
    public String getStringProperty(String name, String defaultValue) {
        String val = getProperty(name);
        if (val == null) {
            return defaultValue;
        }
        return val.trim();
    }


    // If the property is not defined or cannot be parsed
    // we return the given default value.
    public int getIntProperty(String name, int defaultValue) {
        String val = getProperty(name);
        if (val == null) {
            return defaultValue;
        }
        try {
            return Integer.parseInt(val.trim());
        } catch (Exception ex) {
            return defaultValue;
        }
    }


    // If the property is not defined or cannot be parsed
    // we return the given default value.
    public boolean getBooleanProperty(String name, boolean defaultValue) {
        String val = getProperty(name);
        if (val == null) {
            return defaultValue;
        }
        val = val.toLowerCase();
        if (val.equals("true") || val.equals("1")) {
            return true;
        } else if (val.equals("false") || val.equals("0")) {
            return false;
        }
        return defaultValue;
    }

    // we return the given default value.
    public Level getLevelProperty(String name, Level defaultValue) {
        String val = getProperty(name);
        if (val == null) {
            return defaultValue;
        }
        try {
            return Level.parse(val.trim());
        } catch (Exception ex) {
            return defaultValue;
        }
    }


    //  method to get a filter property.
    // We return an instance of the class named by the "name"
    // property. If the property is not defined or has problems
    // we return the defaultValue.
    public Filter getFilterProperty(String name, Filter defaultValue) {
        String val = getProperty(name);
        try {
            if (val != null) {
                Class clz = ClassLoader.getSystemClassLoader().loadClass(val);
                return (Filter) clz.newInstance();
            }
        } catch (Exception ex) {
            // We got one of a variety of exceptions in creating the
            // class or creating an instance.
            // Drop through.
        }
        // We got an exception.  Return the defaultValue.
        return defaultValue;
    }


    //  method to get a formatter property.
    // We return an instance of the class named by the "name"
    // property. If the property is not defined or has problems
    // we return the defaultValue.
    public Formatter getFormatterProperty(String name, Formatter defaultValue) {
        String val = getProperty(name);
        try {
            if (val != null) {
                Class clz = ClassLoader.getSystemClassLoader().loadClass(val);
                return (Formatter) clz.newInstance();
            }
        } catch (Exception ex) {
            // We got one of a variety of exceptions in creating the
            // class or creating an instance.
            // Drop through.
        }
        // We got an exception.  Return the defaultValue.
        return defaultValue;
    }




    /**
     * Shuts down the logging system.
     */
    public void shutdown() {

            for (java.util.logging.Logger logger : loggerMap.values()) {
                Handler[] handlers = logger.getHandlers();
                for (Handler handler : handlers) {
                    logger.removeHandler(handler);
                }
            }
         for (Handler handler : handlerMap.values()) {
             handler.close() ;
         }
    }


    void readInitialConfiguration(boolean fromCtor) {

        String resourceName = LOG_CONF_FILE != null ? LOG_CONF_FILE : LOG_PROPS;
        loggingProperties = BootstrapResourceUtils.getBootstrapProperties(resourceName, this.getClass());
        //HERE DO something for files
        Set<String> set = loggingProperties.stringPropertyNames();
        for (String key : set) {
            if (key.endsWith(".pattern")) {
                String pattern = loggingProperties.getProperty(key);
                try {
                     pattern = checkForFile(pattern);
                    loggingProperties.put(key, pattern) ;
                } catch (IOException e) {
                    System.err.println("problem while creating log directory " + e);
                }
                break;
            }
        }
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        try {
            loggingProperties.store(bos, "");
            byte[] bytes = bos.toByteArray();
            ByteArrayInputStream bis = new ByteArrayInputStream(bytes);
            if(isInInitialManagerState) {
                //System.out.println("CALLING MANAGER INITIAL :" + this.getClass().getClassLoader());
                reset() ;
                super.readConfiguration(bis);
            } else if(delegateToStandardLogger) {
                //System.out.println("CALLING MANAGER WITH DELEGATION: " + this.getClass().getClassLoader() + " "  +stdLogManager);
                stdLogManager.reset() ;
                stdLogManager.readConfiguration(bis);
                handlersLazyLoad();
            } else {
                //System.out.println("CALLING MANAGER ANOTHER TIME for: " + this.getClass().getClassLoader() + " " + stdLogManager);
                //stdLogManager.reset() ;
                stdLogManager.readConfiguration(bis);
                handlersLazyLoad();
            }
        } catch (IOException e) {
            System.err.println(" can't load properties for LogManager" + e);
        }
    }


    protected void handlersLazyLoad() throws IOException {
         if(this.handlersLoaded) return ;
         if(stdLogManager == null) return ;
         Enumeration<String> loggerNames = stdLogManager.getLoggerNames() ;
        while(loggerNames.hasMoreElements()) {
            String loggerName = loggerNames.nextElement() ;
            //System.out.println(" found logger : " + loggerName);
            java.util.logging.Logger logger = stdLogManager.getLogger(loggerName) ;
            if(logger != null) {
               loadLoggerHandlers(loggerName, logger);
            }

        }
        handlersLoaded = true ;
    }
    protected void loadLoggerHandlers(String loggerName, java.util.logging.Logger logger)
            throws IOException {

        Handler[] logHandlers = logger.getHandlers();
        Set<String> currentHandlersNames = new HashSet<>() ;
        for(Handler handler : logHandlers) {
            currentHandlersNames.add(handler.getClass().getName()) ;
        }
        // Create handlers for the root logger of this classloader
        String handlers = loggingProperties.getProperty(loggerName +".handlersN");
        if(handlers == null && "".equals(loggerName)) {
            handlers = loggingProperties.getProperty("handlersN");
        }
        if (handlers != null) {
            StringTokenizer tok = new StringTokenizer(handlers, ",");
            while (tok.hasMoreTokens()) {
                String handlerName = (tok.nextToken().trim());
                String handlerClassName = handlerName;
                String prefix = "";
                if (handlerClassName.length() <= 0) {
                    continue;
                }
                // Parse and remove a prefix (prefix start with a digit, such as
                // "10WebappFooHanlder.")
                if (Character.isDigit(handlerClassName.charAt(0))) {
                    int pos = handlerClassName.indexOf('.');
                    if (pos >= 0) {
                        prefix = handlerClassName.substring(0, pos + 1);
                        handlerClassName = handlerClassName.substring(pos + 1);
                    }
                }
                if(currentHandlersNames.contains(handlerClassName)) {
                    continue ;
                }
                try {
                    ClassLoader classLoader = this.getClass().getClassLoader() ;
                    Handler handler =
                            (Handler) classLoader.loadClass(handlerClassName).newInstance();
                    // The specification strongly implies all configuration should be done
                    // during the creation of the handler object.
                    // This includes setting level, filter, formatter and encoding.
                    configureHandler(handlerName, handler);
                    handlerMap.put(handlerName, handler);
                    logger.addHandler(handler);
                } catch (Exception e) {
                    // Report error
                    System.err.println("Handler error");
                    e.printStackTrace();
                }
            }

        }
        //TODO: this was added .... is it relevant?
        //setLevelsOnExistingLoggers(info.props) ;
        setLevelOnExistingLogger(loggerName, logger) ;

    }


    /**
     * should set level, formatter, filter
     *
     * @param handlerName
     * @param handler
     */
    protected void configureHandler(String handlerName, Handler handler) {
        Level level = getLevelProperty(handlerName + ".level", null);
        if (level == null) {
            level = getLevelProperty(".level", Level.OFF);
        }
        handler.setLevel(level);
        Filter filter = getFilterProperty(handlerName + ".filter", null);
        if (filter == null) {
            filter = getFilterProperty(".filter", null);
        }
        if (filter != null) {
            handler.setFilter(filter);
        }
        java.util.logging.Formatter formatter = getFormatterProperty(handlerName + ".formatter", new SimpleFormatter());
        handler.setFormatter(formatter);
    }



    /**
     * System property replacement in the given string.
     *
     * @param str The original string
     * @return the modified string
     */
    protected String replace(String str, Properties props) {
        String result = str;
        int pos_start = result.indexOf("${");
        if (pos_start != -1) {
            int pos_end = result.indexOf('}');
            if (pos_end != -1) {
                String propName = result.substring(pos_start + 2, pos_end);
                String replacement = props.getProperty(propName);
                if (replacement != null) {
                    if (pos_start > 0) {
                        result = result.substring(0, pos_start) +
                                replacement + replace(result.substring(pos_end + 1), props);
                    } else {
                        result = replacement + replace(result.substring(pos_end + 1), props);
                    }
                }
            }
        }
        return result;
    }


// ---------------------------------------------------- LogNode Inner Class



    /**
     * read the file pattern until last  separator (this will give the directory)
     * then will try to create missing directories.
     *
     * @param filePattern
     */
    String checkForFile(String filePattern) throws IOException {
        //our own specifications
        Properties bootstrapSystemProperties = BootstrapResourceUtils.getBootstrapSystemProperties();
        if (filePattern.contains("%W")) {
            String logdir = bootstrapSystemProperties.getProperty("org.lsst.ccs.workdir","%h") ;
            filePattern = filePattern.replace("%W", logdir);
        } 
        if (filePattern.contains("%L")) {
            String logdir = bootstrapSystemProperties.getProperty("org.lsst.ccs.logdir","%h") ;
            filePattern = filePattern.replace("%L", logdir);
        }
        if (filePattern.contains("%A")) {
            String applicationName = bootstrapSystemProperties.getProperty("org.lsst.ccs.application.name","%u") ;
            filePattern = filePattern.replace("%A", applicationName);
        }

        int lastSlash = filePattern.lastIndexOf('/');
        if (lastSlash >= 0) {
            String directorySpec = filePattern.substring(0, lastSlash);

            if (directorySpec.contains("%h")) {
                directorySpec = directorySpec.replace("%h", System.getProperty("user.home"));
            } else if (directorySpec.contains("%t")) {
                directorySpec = directorySpec.replace("%t", System.getProperty("java.io.tmpdir"));
            }
            Files.createDirectories(FileSystems.getDefault().getPath(directorySpec));
        }
        return filePattern ;
    }


}





