package org.lsst.ccs.utilities.logging;

import org.lsst.ccs.utilities.tracers.Tracer;

import java.awt.*;
import java.io.*;
import java.util.logging.*;

/**
 * Ths static initialization of this class should be fired early in an application by doing
 * <PRE>
 * Class.forName("org.lsst.ccs.utilities.logging.LogManagement") ;
 * </PRE>
 * This code will relod the logging configuration from a <TT>logging.properties</TT> file
 * situated at global level of the CLASSPATH (resource "/logging.properties" for a Class).
 * If it does not find this resource it will nonetheless get rid of the standard <TT>ConsoleHandler</TT>
 * described in "lib/logging.properties".
 *
 * @author bamade
 */
// Date: 31/05/13

public class LogManagement {
    /**
     * a simple global <TT>ErrorManager</TT> that print logging framework problems to the console.
     * Should be changed to something that fires a network message
     */
    public static final ErrorManager ERROR_MANAGER = new ErrorManager();
    private static boolean configInitialized = true;

    public static boolean isConfigInitialized() {
        return configInitialized;
    }

    public static void setConfigInitialized(boolean configInitialized) {
        LogManagement.configInitialized = configInitialized;
    }

    /**
     * gets the top logger (name "") and removes the standard <TT>ConsoleHandler</TT>
     */
    public static Level cleanConsole() {
        //this is to get rid of other Consoles at the top
        java.util.logging.Logger globalHandler = java.util.logging.Logger.getLogger("");
        Handler[] handlers = globalHandler.getHandlers();
        // BEWARE used to know if ConsoleHandler was there!
        Level consoleLevel = null;
        for (Handler handler : handlers) {
            if (handler.getClass().equals(ConsoleHandler.class)) {
                consoleLevel = handler.getLevel();
                globalHandler.removeHandler(handler);
            }
        }
        return consoleLevel;
    }

    public static void setDefaultConsoleHandler(Level consoleLevel) {
        java.util.logging.Logger globalHandler = java.util.logging.Logger.getLogger("");
        if (consoleLevel != null && !GraphicsEnvironment.isHeadless()) {
            Handler consoleHandlerN = new ConsoleHandlerN();
            consoleHandlerN.setLevel(consoleLevel);
            globalHandler.addHandler(consoleHandlerN);
        }
    }


    static {
        //TODO: change that and use property "java.util.logging.config.file" to load classes?
        //TODO: make it a dynamic code linked to ClassLoader (see: ContextLogManager)
        LogManager manager = LogManager.getLogManager();
        if (!(manager instanceof ContextLogManager)) {
            //try to find the name and put it in the property ?
            String configName = System.getProperty("java.util.logging.config.file");
            InputStream is = null;
            if (configName == null) {
                is = LogManagement.class.getResourceAsStream("/logging.properties");
            }

            if (is != null) {
                reloadConfigurationFrom(" global resource logging.properties ", is);
            } else {
                setConfigInitialized(false);
                Level defaultLevel = cleanConsole();
                setDefaultConsoleHandler(defaultLevel);
                ERROR_MANAGER.error("WARNING: no access to global resource " + "logging.properties (but cleaning Console access)", null, ErrorManager.OPEN_FAILURE);
            }

        }
    }

    /**
     * used mainly to read resource "/logging.properties"
     *
     * @param name only for tracing purposes
     * @param is   InputStream to read the configuration from
     */
    static void reloadConfigurationFrom(String name, InputStream is) {
        LogManager manager = LogManager.getLogManager();
        //System.out.println("log manager system classLoader" + ClassLoader.getSystemClassLoader());
        //System.out.println("log management classLoader" + LogManagement.class.getClassLoader());
        try {
            //System.out.println(" RELOADING");
            //ClassLoader systemLoader= ClassLoader.getSystemClassLoader() ;
            //ClassLoader localLoader = Thread.currentThread().getContextClassLoader() ;
            manager.readConfiguration(is);
            assert Tracer.trace("handlers=" + manager.getProperty("handlers"));
        } catch (IOException e) {
            ERROR_MANAGER.error("can't open " + name, e, ErrorManager.OPEN_FAILURE);
        }
    }


}
