package org.lsst.ccs.utilities.logging;

import org.lsst.ccs.utilities.tracers.Tracer;

import java.awt.GraphicsEnvironment;
import java.io.IOException;
import java.io.InputStream;
import java.util.logging.ConsoleHandler;
import java.util.logging.ErrorManager;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogManager;

/**
 * Ths static initialization of this class should be fired early in an application by doing
 * <PRE>
 *     Class.forName("org.lsst.ccs.utilities.logging.LogManagement") ;
 * </PRE>
 * This code will relod the logging configuration from a <TT>logging.properties</TT> file
 * situated at global level of the CLASSPATH (resource "/logging.properties" for a Class).
 * If it does not find this resource it will nonetheless get rid of the standard <TT>ConsoleHandler</TT>
 * described in "lib/logging.properties".
 * @author bamade
 */
// Date: 31/05/13

public class LogManagement {
    /**
     * a simple global <TT>ErrorManager</TT> that print logging framework problems to the console.
     * Should be changed to something that fires a network message
     */
    public static final ErrorManager ERROR_MANAGER = new ErrorManager() ;
    private static boolean configInitialized = true;

    public static boolean isConfigInitialized() {
        return configInitialized;
    }

    public static void setConfigInitialized(boolean configInitialized) {
        LogManagement.configInitialized = configInitialized;
    }

    /**
     * gets the top logger (name "") and removes the standard <TT>ConsoleHandler</TT>
     */
    public static Level cleanConsole() {
        //this is to get rid of other Consoles at the top
        java.util.logging.Logger globalHandler = java.util.logging.Logger.getLogger( "");
        Handler[] handlers = globalHandler.getHandlers() ;
        // BEWARE used to know if ConsoleHandler was there!
        Level consoleLevel = null;
        for(Handler handler: handlers){
            if(handler.getClass().equals( ConsoleHandler.class)) {
                consoleLevel = handler.getLevel() ;
                globalHandler.removeHandler(handler);
            }
        }
        return consoleLevel ;
    }

    public static void setDefaultConsoleHandler(Level consoleLevel) {
        java.util.logging.Logger globalHandler = java.util.logging.Logger.getLogger( "");
        if( consoleLevel!= null && ! GraphicsEnvironment.isHeadless()) {
            Handler consoleHandlerN = new ConsoleHandlerN();
            consoleHandlerN.setLevel(consoleLevel);
            globalHandler.addHandler(consoleHandlerN);
        }
    }


    static {
        String resourcesDir = System.getProperty("org.lsst.ccs.resources.directory", "");
        InputStream is = LogManagement.class.getResourceAsStream(resourcesDir+"/logging.properties") ;
        if(is != null ) {
            reloadConfigurationFrom(" global resource logging.properties ", is);
        } else {
            setConfigInitialized(false);
            Level defaultLevel = cleanConsole();
            setDefaultConsoleHandler(defaultLevel);
            ERROR_MANAGER.error("WARNING: no access to global resource " +resourcesDir +"/logging.properties (but cleaning Console access)", null, ErrorManager.OPEN_FAILURE);
        }
    }

    /**
     * used mainly to read resource "/logging.properties"
     * @param name only for tracing purposes
     * @param is InputStream to read the configuration from
     */
    static void reloadConfigurationFrom(String name, InputStream is) {
        LogManager manager = LogManager.getLogManager();
        try {
            manager.readConfiguration(is);
            assert Tracer.trace("handlers=" + manager.getProperty("handlers"));
        } catch (IOException e) {
            ERROR_MANAGER.error("can't open " + name,e, ErrorManager.OPEN_FAILURE);
        }
    }
}
