package org.lsst.ccs.utilities.jars;

import org.lsst.ccs.utilities.logging.Logger;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Properties;
import java.util.jar.Manifest;
import java.util.logging.Level;

/**
 * This class deals with "common resources": that is resources that
 * are not linked to a specific package (such resources are often queried
 * using an initial slash when querying a resource through the <TT>Class</TT>
 * resource methods such as <TT>getResource</TT> : <TT>getResource("/global.properties")</TT>).
 * <p/>
 * <B>Beware</B>: in the codeusing this class these resource names <B>should not</B> start with a slash (same protocol as with the <TT>ClassLoader</TT>
 * resource methods.
 * <p/>
 * The general idea is that when dealing with multiple jars there may be many "common resources" with the same name.
 * <BR/>
 * So:
 * <UL>
 * <LI/> the <TT>getURLsFrom</TT> method will return all the URL of this resource known by a <TT>ClassLoader</TT>
 * (chances are that there may be many of these)
 * <LI/> the <TT>getURLsFor</TT> method will return the resource linked to a specific object. So if there
 * are many resources with the same name this will pick up only the one that is in the jar of the current class.
 * (chances are that there may be only one such resource: but this is not 100% sure if different versions
 * of the same class are referenced in different jars! -an unlikely occurence-)
 * </UL>
 * <p/>
 * Other utilities methods:
 * <UL>
 * <LI/> <TT>getPropertiesFrom(URL)</TT> loads a property from a common resource
 * <LI/> <TT>getResourceContainer</TT> gives the name of the jar that contains a resource
 * </UL>
 *
 * @author bamade
 */
// Date: 14/05/13

public class CommonResources {
    public static final URL[] URL_ARRAY_MODEL = new URL[0];
    public static final Logger CURLOG = Logger.getLogger("org.lsst.ccs.utilities");

    /**
     * As "seen" from a <TT>ClassLoader</TT> the various URLs that point to resources with a given name.
     * Normally these are ordered along the ClassPath order so if used for a hierarchical exploration
     * the resulting array should be exploited backwards.
     *
     * @param loader             a current ClassLoader
     * @param commonResourceName name of resource (without initial slash!)
     * @return an Array of URL that point to resources with that name
     */
    public static URL[] getURLsFrom(ClassLoader loader, String commonResourceName) {
        ArrayList<URL> list = new ArrayList<URL>();
        try {
            Enumeration<URL> enumURL = loader.getResources(commonResourceName);
            while (enumURL.hasMoreElements()) {
                list.add(enumURL.nextElement());
            }
        } catch (IOException e) {
            CURLOG.warn("common resource Name error: " + commonResourceName, e);
        }
        return list.toArray(URL_ARRAY_MODEL);
    }

    /**
     * As "seen" from the context ClassLoader the various URLs that point to resources with a given name
     *
     * @param commonResourceName name of resource (without initial slash!)
     * @return an Array of URL that point to resources with that name
     */
    public static URL[] getURLsFrom(String commonResourceName) {
        return getURLsFrom(Thread.currentThread().getContextClassLoader(), commonResourceName);
    }

    /**
     * return the URL of a resource that lies in the same jar as a class.
     *
     * @param obj                any instance of a Class object if no instance is at hand
     * @param commonResourceName name of resource (without initial slash!)
     * @return an Array (with normally only one element)
     */
    public static URL[] getURLsFor(Object obj, String commonResourceName) {
        Class clazz;
        if (obj instanceof Class) {
            clazz = (Class) obj;
        } else {
            clazz = obj.getClass();
        }
        ArrayList<URL> list = new ArrayList<URL>();
        ClassLoader loader = clazz.getClassLoader();
        URL[] allResources = getURLsFrom(loader, commonResourceName);
        //System.out.println(" name path : " + classNamePath);
        String[] urlNames = getClassURLNamesFor(clazz);
        for (String urlName : urlNames) {
            //System.out.println(" URL name : " + urlName);
            for (URL urlResource : allResources) {
                //System.out.println(" URL resource : " + urlResource);
                if (urlResource.toString().startsWith(urlName)) {
                    list.add(urlResource);
                }
            }

        }
        return list.toArray(URL_ARRAY_MODEL);
    }

    /**
     * get the names of places where a Class lies in the classPath
     *
     * @param obj any object or Class
     * @return mostly an array of size one
     */
    public static String[] getClassURLNamesFor(Object obj) {
        Class clazz;
        if (obj instanceof Class) {
            clazz = (Class) obj;
        } else {
            clazz = obj.getClass();
        }
        String classNamePath = clazz.getCanonicalName().replace('.', '/') + ".class";
        ClassLoader loader = clazz.getClassLoader();
        URL[] classRepositories = getURLsFrom(loader, classNamePath);
        String[] res = new String[classRepositories.length];
        for (int ix = 0; ix < classRepositories.length; ix++) {
            String urlName = classRepositories[ix].toString().replace(classNamePath, "");
            res[ix] = urlName.substring(0, urlName.length() - 1);
        }
        return res;
    }

    /**
     * fills a Property Object from an <TT>URL the references</TT> a file in ".properties" format
     *
     * @param url
     * @return a Properties object that may be empty if URL incorrect (or read error of the resource)
     */
    public static Properties getPropertiesFrom(URL url) {
        Properties props = new Properties();
        try {
            InputStream is = url.openStream();
            props.load(is);
        } catch (Exception e) {
            CURLOG.warn("URL stream error for: " + url, e);
        }
        return props;
    }

    /**
     * gets the name of a jar that contains a "common resource"
     *
     * @param url
     * @param commonResourceName name of resource (without initial slash!)
     * @return the name of a jar that "contains" the object referenced by the URL (can also be a directory)
     */
    public static String getResourceContainer(URL url, String commonResourceName) {
        String path = url.getPath();
        int idx = path.lastIndexOf(commonResourceName);
        if (idx < 0) return null;
        String pathWithoutResource = path.substring(0, idx - 1);
        //System.out.println("pathWithoutResource :" +pathWithoutResource);
        int lastSlash = pathWithoutResource.lastIndexOf('/');
        String res = pathWithoutResource.substring(lastSlash + 1);
        if (res.endsWith("!")) {
            res = res.substring(0, res.length() - 1);
        } else {
            res = pathWithoutResource;
        }
        return res;
    }

    /**
     * returns a Manifest for an object
     *
     * @param obj an object instance or a Class
     * @return a Manifest or null if not found
     */
    public static Manifest getManifestFor(Object obj) {
        URL[] jarURLs = getURLsFor(obj, "META-INF/MANIFEST.MF");
        if (jarURLs.length > 0) {
            URL jarURL = jarURLs[0];
            try {
                InputStream is = jarURL.openStream();
                Manifest res = new Manifest(is);
                return res;
            } catch (IOException e) {
                CURLOG.warn("Manifest opening problem at " + jarURL, e);
                return null;
            }

        } else {
            return null;
        }
    }

    /**
     * searches for a file as a local file or a resource linked to a class or a global resource.
     *
     * @param clazz
     * @param pathName
     * @return
     * @throws IOException
     */
    // TODO: possible duplication of code in core utils
    public static InputStream getInput(Class clazz, String pathName) throws IOException {
        InputStream is = null;
        File file = new File(pathName);
        if (file.exists()) {
            is = new FileInputStream(file);

        } else {
            is = clazz.getResourceAsStream(pathName);
            if (is == null) {
                is = clazz.getResourceAsStream("/" + pathName);
            }
            if (is == null) {
                throw new IllegalArgumentException("no resource " + pathName);
            }
        }
        return is;
    }

    /**
     * @param name
     * @return
     */
    public static Properties getPropertiesFromAny(String name) {
        Properties res = new Properties();
        try {
            InputStream is = getInput(CommonResources.class, name);
            res.load(is);
            is.close(); //TODO : use automatic closing 1.7
        } catch (IOException e) {
            CURLOG.log(Level.WARNING, "no general resource for :" + name, e);
        }
            return res;
    }

    /**
     *  this method will create a <TT>Properties</TT> object filled with ALL the ".properties" resources
     *  and Files that happen to bear the same name in the ClassPath and in the current directory.
     *  It uses a hierarchical reading of these files: any property which is in a file "before" the others
     *  in the current directory and in the classPath will "shadow" other values.
     * @param loader  ClassLoader
     * @param loaderResourceName the name of the Resource according to ClassLoader conventions
     *                           (this means that the usual ResourceName with Objects should not be used)
     * @return a Properties object with content read from these various files
     */
    public static Properties gatherAllPropertiesFrom(ClassLoader loader, String loaderResourceName) {
        Properties res = new Properties();
        // then from all resources
        URL[] urls = getURLsFrom(loader, loaderResourceName);
        for (int ix = urls.length - 1; ix >= 0; ix--) {
            URL url = urls[ix];
            try {
                InputStream is = url.openStream();
                res.load(is);
                is.close();
            } catch (IOException e) {
                CURLOG.log(Level.WARNING, "reading for properties url : " + url, e);  //To change body of catch statement use File | Settings | File Templates.
            }
        }
        //first from a file
        File file = new File(loaderResourceName);
        if (file.exists()) {
            try {
                InputStream is = new FileInputStream(file);
                res.load(is);
                is.close(); //TODO : use 1.7 closing
            } catch (IOException e) {
                CURLOG.log(Level.WARNING, "", e);
            }
        }

        return res;

    }

    /**
     * same as the method with a ClassLoader argument but uses the current ClassLoader
     * @param loaderResourceName file name or resource name that uses the ClassLoader resource naming convention
     * @return a filled Properties object
     */
    public static Properties gatherAllPropertiesFrom( String loaderResourceName) {
        return gatherAllPropertiesFrom(Thread.currentThread().getContextClassLoader(), loaderResourceName) ;
    }
}
