package org.lsst.ccs.utilities.dispatch;


import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;

/**
 * Instances of this class creates a dynamic proxy that dispatches method call
 * to registered agents implementing an interface type.
 * <P>
 *  <B>beware</B> : if the implemented interface is itself a parameterized type
 *  no type control is exercised on these type parameters.
 *
 * @author Bernard AMADE
 */
public class SynchronousDispatchProxy <I> {
    protected CopyOnWriteArrayList<Object> list = new CopyOnWriteArrayList<Object>() ;
    class Invoker implements InvocationHandler {
        @Override
        public Object invoke(Object o, Method method, Object[] parms){
            for(Object goal : list) {
                try {
                method.invoke(goal, parms);
                } catch (Exception exc) {
                    PackCst.CURLOG.log(Level.SEVERE, "dispatching to " + goal, exc);
                }
            }
            return null;
        }
    }

    protected I proxy ;

    /**
     *  prepares a generator able to deliver a Proxy object able to dispatch method calls to registered agents.
     *   The methods of the interface will necessarily be of procedural nature (the return values will be dumped)
     * @param interfaceType
     */
    public SynchronousDispatchProxy(Class<I> interfaceType) {
        // not necessarily a good idea ;
        proxy = (I) Proxy.newProxyInstance(this.getClass().getClassLoader(), new Class[] {interfaceType}, new Invoker()) ;

    }

    /**
     * Same as previous but with a specified ClassLoader
     * @param interfaceType
     * @param loader
     */
    public SynchronousDispatchProxy(Class<I> interfaceType, ClassLoader loader) {
        proxy = (I) Proxy.newProxyInstance(loader, new Class[] {interfaceType}, new Invoker()) ;
    }

     /**
     * Registers a code ready to receive "synchronous" commands
     * @param executant (should implement the interface)
     */
    public  <T extends I> void addExecutant(T executant) {
        list.add(executant) ;
    }

    /**
     * Remove the code from the listeners list
     *
     * @param executant
     */
    public <T extends I> void removeExecutant(T executant){
        list.remove(executant) ;
    }

    /**
     * to be called once to get a dynamically generated proxy that will forward calls to registered agents.
     *
     * @return
     */
    public I getProxy() {
        return proxy ;
    }
}
