package org.lsst.ccs.utilities.taitime;

import java.io.IOException;
import java.net.URL;
import java.time.Instant;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 * This class represents a CCS timestamp, which encapsulates both a TAI and UTC
 * timestamp. If we are running in strict mode, both the TAI and UTC time are
 * obtained by appropriate kernel routine calls, and if the kernel interface
 * library cannot be loaded, or if the the kernel is not correctly configured,
 * then the class will throw an exception. In non-strict mode. if the kernel
 * libraries cannot be loaded the class will fall back to using the built-in
 * Java time, plus a list of leap seconds read from an external URL.
 * <p>
 * The following system properties control how the class works:
 * <table border="1">
 * <tr><th>Property</th><th>Default</th><th>Description</th></tr>
 * <tr><td>org.lsst.ccs.utilities.taitime.useStrict</td><td>false</td><td>Controls
 * whether strict mode is set or not</td></tr>
 * <tr><td>org.lsst.ccs.utilities.taitime.libName</td><td>timeaccess</td><td>The
 * name of the JNI library to load</td></tr>
 * <tr><td>org.lsst.ccs.utilities.taitime.minLeapSeconds</td><td>30</td><td>The
 * minimum number of leap seconds for the kernel to be considered configured
 * correctly</td></tr>
 * </table>
 *
 * @author Farrukh Azfar
 */
public class CCSTimeStamp implements java.io.Serializable, Comparable<CCSTimeStamp> {

    private static final long serialVersionUID = 6264466048267081974L;

    private static final TAITime times;
    private static final LeapSecondReader reader;

    private final Instant taiInstant;
    private final Instant utcInstant;

    private static final Instant INVALID_INSTANT = Instant.ofEpochMilli(0);
    private static final Logger LOG = Logger.getLogger(CCSTimeStamp.class.getName());

    /**
     * Attempts to get the current timestamp. If we are in strict mode, and the
     * kernel library cannot be loaded, or the leap seconds offset does not seem
     * reasonable, an exception will be thrown.
     *
     * @return A CCSTimeStamp for the current time
     * @exception RuntimeException If the TAI time offset is not available.
     */
    public static CCSTimeStamp currentTime() {

        return new CCSTimeStamp();
    }

    /**
     * Attempts to create a CCSTimeStamp from a utc milliseconds value. This is
     * a temporary method to allow migrating from old java time based
     * milliseconds values to CCSTimeStamp.
     *
     * @param milliseconds A standard Java milliseconds values (as returned by
     * System.currentTimeMills())
     * @return A CCSTimeStamp corresponding to the given timestamp.
     */
    public static CCSTimeStamp currentTimeFromMillis(long milliseconds) {

        return new CCSTimeStamp(milliseconds);
    }

    private CCSTimeStamp(long milliseconds) {

        if (TAITime.isStrict()) {
            throw new RuntimeException("Cannot use in strict mode !");
        } else {
            if (milliseconds == 0 || milliseconds == -1) {
                LOG.log(Level.WARNING, "Creating invalid instant version of CCSTimeStamp for milliseconds: {0}", milliseconds);
                utcInstant = INVALID_INSTANT;
                taiInstant = INVALID_INSTANT;
            } else {
                utcInstant = Instant.ofEpochMilli(milliseconds);
                int leapSecond = getReader().getNumberOfLeapSeconds(milliseconds);
                taiInstant = utcInstant.plusSeconds(leapSecond);
            }
        }
    }

    static {
        times = new TAITime();        
        LeapSecondReader leapSecondReader;
        try {
            try {
                leapSecondReader = new LeapSecondReader();
            } catch (IOException x) {
                // Try reading from bootstrap path as a fallback
                URL leapURL = BootstrapResourceUtils.getResourceURL("leap-seconds.list");
                if (leapURL != null) {
                    leapSecondReader = new LeapSecondReader(leapURL);
                    LOG.warning("Unable to read default leap second file, falling back to reading local copy from bootstrap path");
                    long expires = leapSecondReader.getExpiryDate();
                    if (expires != 0 && expires < System.currentTimeMillis()) {
                        LOG.log(Level.SEVERE, "Leap second data read from {0} has expired", leapURL);
                    }
                } else {
                    throw x;
                }
            }
        } catch (IOException x) {
            throw new RuntimeException("Leap second file can not be loaded", x);
        }

        reader = leapSecondReader;
    }
    
    private static LeapSecondReader getReader() {
        if (reader == null) {
            throw new RuntimeException("LeapSecondReader was not initialized");
        }
        return reader;
    }

    private CCSTimeStamp() {
        // If we are in strict mode, and the configuration is not good, an exception will be thrown here
        if (times.isConfigured()) {

            TimeStorage store = times.getTime();
            taiInstant = Instant.ofEpochSecond(store.getTimeSecsTAI(), store.getTimeNanoTAI());
            utcInstant = Instant.ofEpochSecond(store.getTimeSecsUTC(), store.getTimeNanoUTC());

        } else {

            utcInstant = Instant.now();
            int leapSecond = reader.getNumberOfLeapSeconds(utcInstant.toEpochMilli());
            taiInstant = utcInstant.plusSeconds(leapSecond);
        }
    }

    public Instant getTAIInstant() {

        return taiInstant;
    }

    public Instant getUTCInstant() {

        return utcInstant;
    }

    public double getTAIDouble() {

        return taiInstant.getEpochSecond() + taiInstant.getNano() / 1000000000.0;
    }

    public double getUTCDouble() {

        return utcInstant.getEpochSecond() + utcInstant.getNano() / 1000000000.0;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof CCSTimeStamp) {
            CCSTimeStamp ts = (CCSTimeStamp) obj;
            return this.taiInstant.equals(ts.taiInstant) && this.utcInstant.equals(ts.utcInstant);
        }
        return false;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 67 * hash + Objects.hashCode(this.taiInstant);
        hash = 67 * hash + Objects.hashCode(this.utcInstant);
        return hash;
    }

    @Override
    public String toString() {
        return "CCSTimeStamp{utc=" + utcInstant + "}";        
    }
    
    @Override
    public int compareTo(CCSTimeStamp o) {
        return taiInstant.compareTo(o.taiInstant);
    }
    
}
