package org.lsst.ccs.utilities.taitime;

import java.io.IOException;
import java.net.URL;
import java.time.Instant;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 * This class represents a CCS timestamp, which encapsulates both a TAI and UTC
 * timestamp. If we are running in strict mode, both the TAI and UTC time are
 * obtained by appropriate kernel routine calls, and if the kernel interface
 * library cannot be loaded, or if the the kernel is not correctly configured,
 * then the class will throw an exception. In non-strict mode, if the kernel
 * libraries cannot be loaded the class will fall back to using the built-in
 * Java time, plus a list of leap seconds read from an external URL.
 * <p>
 * The following system properties control how the class works:
 * <table border="1">
 * <tr><th>Property</th><th>Default</th><th>Description</th></tr>
 * <tr><td>org.lsst.ccs.utilities.taitime.useStrict</td><td>false</td><td>Controls
 * whether strict mode is set or not</td></tr>
 * <tr><td>org.lsst.ccs.utilities.taitime.libName</td><td>timeaccess</td><td>The
 * name of the JNI library to load</td></tr>
 * <tr><td>org.lsst.ccs.utilities.taitime.minLeapSeconds</td><td>30</td><td>The
 * minimum number of leap seconds for the kernel to be considered configured
 * correctly</td></tr>
 * </table>
 * 
 * @see <a href="https://hpiers.obspm.fr/iers/bul/bulc/ntp/leap-seconds.list">leaps-seconds.list</a>
 *
 * @author Farrukh Azfar
 */
public class CCSTimeStamp implements java.io.Serializable, Comparable<CCSTimeStamp> {

    private static final long serialVersionUID = 6264466048267081974L;

    private static final TAITime times;
    private static volatile LeapSecondReader reader;

    private final Instant taiInstant;
    private final Instant utcInstant;

    private static final Instant INVALID_INSTANT = Instant.ofEpochMilli(0);
    private static final Logger LOG = Logger.getLogger(CCSTimeStamp.class.getName());

    /**
     * Attempts to get the current timestamp. If we are in strict mode, and the
     * kernel library cannot be loaded, or the leap seconds offset does not seem
     * reasonable, an exception will be thrown.
     *
     * @return A CCSTimeStamp for the current time
     * @exception RuntimeException If the TAI time offset is not available.
     */
    public static CCSTimeStamp currentTime() {

        return new CCSTimeStamp();
    }
    
    /**
     * Create an instance of CCSTimeStamp based on the current TAI time, and 
     * the delta between TAI and UTC in effect at the time.
     * @param taiTime The TAI time, as an Instant
     * @param dtai The difference between TAI and UTC in seconds, at the tai instant given. See the definition
     * of dtai in <a href="https://hpiers.obspm.fr/iers/bul/bulc/ntp/leap-seconds.list">leap-seconds.list</a>.
     * @return The corresponding CCSTimeStamp
     */
    public static CCSTimeStamp fromTAI(Instant taiTime, int dtai) {
        return new CCSTimeStamp(taiTime, dtai);
    }

    /**
     * Attempts to create a CCSTimeStamp from a utc milliseconds value. This is
     * a temporary method to allow migrating from old java time based
     * milliseconds values to CCSTimeStamp.
     *
     * @param milliseconds A standard Java milliseconds values (as returned by
     * System.currentTimeMills())
     * @return A CCSTimeStamp corresponding to the given timestamp.
     */
    @Deprecated
    public static CCSTimeStamp currentTimeFromMillis(long milliseconds) {

        return new CCSTimeStamp(milliseconds);
    }

    @Deprecated
    private CCSTimeStamp(long milliseconds) {

        if (TAITime.isStrict()) {
            throw new RuntimeException("Cannot use in strict mode !");
        } else {
            if (milliseconds == 0 || milliseconds == -1) {
                LOG.log(Level.WARNING, "Creating invalid instant version of CCSTimeStamp for milliseconds: {0}", milliseconds);
                utcInstant = INVALID_INSTANT;
                taiInstant = INVALID_INSTANT;
            } else {
                utcInstant = Instant.ofEpochMilli(milliseconds);
                int leapSecond = getReader().getNumberOfLeapSeconds(milliseconds);
                taiInstant = utcInstant.plusSeconds(leapSecond);
            }
        }
    }
    
    private CCSTimeStamp(Instant taiTime, int deltaTAI) {
        taiInstant = taiTime;
        utcInstant = Instant.ofEpochSecond(taiTime.getEpochSecond() - deltaTAI, taiTime.getNano());
    }
    
    public CCSTimeStamp(Instant taiTime, Instant utcTime) {
        taiInstant = taiTime;
        utcInstant = utcTime;
    }

    static {
        times = new TAITime();
        LeapSecondReader leapSecondReader = null;
        // We should only read the leap second file if it is needed.  It is only needed if we are not in strict mode, and
        // the number of leap-seconds obtained from the kernel is not believable, or if we are using the (deprecated)
        // currentTimeFromMillis. We want to avoid calling the synchronized block in createleapSecondReader when we
        // can avoid it (LSSTCCS-2541) so we initialize it here, if it is likely to be used.
        if (!TAITime.isStrict() && !times.isConfigured()) {
            leapSecondReader = createLeapSecondReader();
        } 
        reader = leapSecondReader;
    }

    private static LeapSecondReader createLeapSecondReader() throws RuntimeException {
        try {
            LeapSecondReader leapSecondReader;
            try {
                leapSecondReader = new LeapSecondReader();
            } catch (IOException x) {
                // Try reading from bootstrap path as a fallback
                URL leapURL = BootstrapResourceUtils.getResourceURL("leap-seconds.list");
                if (leapURL != null) {
                    leapSecondReader = new LeapSecondReader(leapURL);
                    LOG.warning("Unable to read default leap second file, falling back to reading local copy from bootstrap path");
                    long expires = leapSecondReader.getExpiryDate();
                    if (expires != 0 && expires < System.currentTimeMillis()) {
                        LOG.log(Level.SEVERE, "Leap second data read from {0} has expired", leapURL);
                    }
                } else {
                    throw x;
                }
            }
            return leapSecondReader;
        } catch (IOException x) {
            throw new RuntimeException("Leap second file can not be loaded", x);
        }
    }
    
    private static LeapSecondReader getReader() {
        if (reader == null) {
            // Only let one thread read the file
            synchronized (times) {
                // In case someone else set it while we were waiting for the lock
                if (reader == null) {
                    reader = createLeapSecondReader();
                }
            }
        }
        return reader;
    }

    private CCSTimeStamp() {
        // If we are in strict mode, and the configuration is not good, an exception will be thrown here
        if (times.isConfigured()) {

            TimeStorage store = times.getTime();
            taiInstant = Instant.ofEpochSecond(store.getTimeSecsTAI(), store.getTimeNanoTAI());
            utcInstant = Instant.ofEpochSecond(store.getTimeSecsUTC(), store.getTimeNanoUTC());

        } else {

            utcInstant = Instant.now();
            int leapSecond = reader.getNumberOfLeapSeconds(utcInstant.toEpochMilli());
            taiInstant = utcInstant.plusSeconds(leapSecond);
        }
    }

    public Instant getTAIInstant() {

        return taiInstant;
    }

    public Instant getUTCInstant() {

        return utcInstant;
    }

    public double getTAIDouble() {

        return taiInstant.getEpochSecond() + taiInstant.getNano() / 1000000000.0;
    }

    public double getUTCDouble() {

        return utcInstant.getEpochSecond() + utcInstant.getNano() / 1000000000.0;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof CCSTimeStamp) {
            CCSTimeStamp ts = (CCSTimeStamp) obj;
            return this.taiInstant.equals(ts.taiInstant) && this.utcInstant.equals(ts.utcInstant);
        }
        return false;
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 67 * hash + Objects.hashCode(this.taiInstant);
        hash = 67 * hash + Objects.hashCode(this.utcInstant);
        return hash;
    }

    @Override
    public String toString() {
        return "CCSTimeStamp{utc=" + utcInstant + "}";        
    }
    
    @Override
    public int compareTo(CCSTimeStamp o) {
        return taiInstant.compareTo(o.taiInstant);
    }
    
}
