package org.lsst.ccs.utilities.taitime;

import java.io.File;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This is an internal class which deals with loading the kernel level routine
 * and accessing timestamps from the kernel.
 *
 * @author Farrukh Azfar
 */
class TAITime {

    private static final Logger logger = Logger.getLogger(TAITime.class.getName());

    // local private boolean 
    private static boolean libFound = true;
    private static UnsatisfiedLinkError loadException;
    private final long leapSeconds;
    private final boolean leapSecondsAreReasonable;

    // Environment variables
    private static final boolean strict = Boolean.getBoolean("org.lsst.ccs.utilities.taitime.useStrict");
    private static final String libName = System.getProperty("org.lsst.ccs.utilities.taitime.libName", "timeaccess");
    private static final int minLeapSeconds = Integer.getInteger("org.lsst.ccs.utilities.taitime.minLeapSeconds", 30);

    static {
        try {
            System.loadLibrary(libName);
        } catch (java.lang.UnsatisfiedLinkError e) {
            libFound = false;
            loadException = e;
            logger.log(Level.WARNING, "c++ shared library {0} not found in java.library.path : " + System.getProperty("java.library.path"), libName);
            String path = System.getProperty("java.library.path");
            StringTokenizer st = new StringTokenizer(path, File.pathSeparator);
            StringBuilder sb = new StringBuilder();
            while (st.hasMoreTokens()) {
                String libPath = st.nextToken() + "lib" + libName + ".so";
                File f = new File(libPath);
                sb.append("File ").append(libPath).append(" (exists: ").append(f.exists()).append(" can read: ").append(f.canRead()).append(")\n");
            }
            logger.log(Level.WARNING, sb.toString());
        }
    }

    TAITime() {
        if (libFound) {
            TimeStorage storage = getTime();
            leapSeconds = storage.getTimeSecsTAI() - storage.getTimeSecsUTC();
            leapSecondsAreReasonable = leapSeconds >= minLeapSeconds;
        } else {
            leapSeconds = 0;
            leapSecondsAreReasonable = false;
        }
    }

    /**
     * Test if the kernel is configured correctly, and throw an exception if we
     * are in strict mode and the the kernel was not configured correctly.
     *
     * @return returns true only if the kernel library was loaded successfully,
     * and the number of leap seconds configured in the kernel seems reasonable.
     * @throws RuntimeException if strict mode is set, and the kernel is not
     * configured correctly
     */
    boolean isConfigured() {
        if (strict) {
            if (!libFound) {
                throw new RuntimeException("Kernel library could not be loaded", loadException);
            }
            if (!leapSecondsAreReasonable) {
                throw new RuntimeException("Kernal leap second offset was too small: " + leapSeconds + "<" + minLeapSeconds);
            }
        }
        return libFound && leapSecondsAreReasonable;
    }

    // Declare a native method timeaccess that receives the java storage class TimeStorage and returns void
    // remember the function timeacces is a C function resident in the library named timeacces
    private native int timeaccess(TimeStorage tstore);

    // for requesting a timestamp ... 
    final TimeStorage getTime() {

        TimeStorage result = new TimeStorage();
        int rc = timeaccess(result);
        if (rc != 0) {
            throw new RuntimeException("Unexpected return code from gettime: " + rc);
        }
        return result;
    }

    static boolean isStrict() {
        return strict;
    }    
}
