package org.lsst.ccs.utilities.taitime;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.Map;
import java.util.TreeMap;
import java.util.logging.Logger;

/**
 * A class which reads a file of leap seconds (from IETF) and can be used to
 * convert between UTC and TAI. This class is only used in non-strict mode, in
 * strict mode we get TAI via PTP from the observatory, so do not need to rely
 * on an external source of leap second data.
 *
 * @author Farrukh Azfar
 */
class LeapSecondReader {

    private final static long EPOCH_OFFSET = 2208988800L;// this is the epoch time since 1900 - what the IETF uses as t0
    private static final Logger LOG = Logger.getLogger(LeapSecondReader.class.getName());

    /**
     * This is a sorted map as read from the leap second file. The keys are a
     * sorted list of leap seconds, and the values are the number of leap
     * seconds in effect following that leap second (valid until the next
     * leapSecond).
     */
    private final TreeMap<Long, Integer> leapMap;
    private long expiryDate;
    private long lastModDate;

    /**
     * Builds a leap second reader using the default IETF source of leap second
     * data.
     *
     */
    LeapSecondReader() throws IOException {
        this(new URL("https://www.ietf.org/timezones/data/leap-seconds.list"));
    }

    /**
     * Build a leap second reader by reading leap seconds from the specified
     * source. The format of the source is documented
     * <a href="https://www.ietf.org/timezones/data/leap-seconds.list">here</a>.
     *
     * @param source The URL from which the list of leap seconds will be read.
     */
    LeapSecondReader(URL source) throws IOException {
        this.leapMap = readLeaps(source);
    }

    /*
     * Reads and parses the leap second data from the URL given to the constructor.
     */
    private TreeMap<Long, Integer> readLeaps(URL source) throws IOException {

        try (BufferedReader in = new BufferedReader(new InputStreamReader(source.openStream()))) {
            TreeMap<Long, Integer> localLeapMap = new TreeMap<>();
            String inputLine;
            while ((inputLine = in.readLine()) != null) {
                //Split on comment delimiter
                String[] split = inputLine.split("#", 2);
                if (split[0].isEmpty()) {
                    if (inputLine.startsWith("#@")) {
                        expiryDate = parseTimestamp(inputLine.substring(2).trim());
                    } else if (inputLine.startsWith("#$")) {
                        lastModDate = parseTimestamp(inputLine.substring(2).trim());
                    }
                    // Ignore other comment lines 
                } else {
                    String[] tokens = split[0].split("\\s+");
                    if (tokens.length != 2) {
                        throw new IOException("Malformed line in leap second file: " + inputLine);
                    }
                    // Parse leap seconds, and convert to Java time conventions
                    long leap = parseTimestamp(tokens[0]);
                    int nSeconds = Integer.parseInt(tokens[1]);
                    localLeapMap.put(leap, nSeconds);
                }
            }
            return localLeapMap;
        }
    }

    private static long parseTimestamp(String token) throws NumberFormatException {
        return 1000 * (Long.parseLong(token) - EPOCH_OFFSET);
    }

    /**
     * Return the current number of leap-seconds in effect
     * @return The number of leap seconds currently in effect.
     */
    int getNumberOfLeapSeconds() {
        return getNumberOfLeapSeconds(System.currentTimeMillis());
    }

    /**
     * Return the number of leap seconds in effect at the given timestamp;
     * @param timestamp A java (UTC) timestamp
     * @return The number of leap seconds in effect at the given timestamp. 
     * If the given timestamp corresponds exactly to a leap second, then that leap
     * second is included in the returned number.
     */
    int getNumberOfLeapSeconds(long timestamp) {
        Map.Entry<Long, Integer> leapSecond = leapMap.floorEntry(timestamp);
        if (leapSecond == null) {
            throw new RuntimeException("Cannot compute leap-second offset for timestamp before beginning of leap second file");
        } 
        return leapSecond.getValue();
    }
    
    /**
     * This method exposes the leap map for testing purposes only.
     * @return 
     */
    TreeMap<Long,Integer> getLeapMap() {
        return leapMap;
    }

    long getExpiryDate() {
        return expiryDate;
    }

    long getLastModificationDate() {
        return lastModDate;
    }
    
    
}
