package org.lsst.ccs.gconsole.services.lock;

import java.time.Instant;
import java.util.*;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPopupMenu;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.plugins.monitor.ConfigView;

/**
 * Lock widget that can be added to any panel, and used to lock and unlock a remote subsystem.
 * Just create an instance, providing the subsystem name to its constructor, and add to any
 * part of your GUI. No wiring is necessary, the only method of this class needed to add this
 * widget to a GUI is its constructor.
 * 
 * Additionally, this class provides public getters and listener registration. Listeners are
 * notified whenever the state of this widget changes.
 * 
 * All access to this class should happen on AWT Event Dispatching Thread.
 * 
 * See {@link ConfigView} for an example of use.
 *
 * @author onoprien
 */
public class Lock extends JLabel {

// -- Fields : -----------------------------------------------------------------
        
    static public final ImageIcon ICON_UNLOCKED = new ImageIcon(Lock.class.getResource("black_unlocked_24.png"), "unlocked black");
    static public final ImageIcon ICON_UNLOCKED_OFFLINE = new ImageIcon(Lock.class.getResource("gray_unlocked_24.png"), "unlocked gray");
    static public final ImageIcon ICON_LOCKED = new ImageIcon(Lock.class.getResource("black_locked_24.png"), "locked black");
    static public final ImageIcon ICON_LOCKED_OFFLINE = new ImageIcon(Lock.class.getResource("gray_locked_24.png"), "locked gray");
    static public final ImageIcon ICON_DETACHED = new ImageIcon(Lock.class.getResource("red_locked_24.png"), "locked red");
    static public final ImageIcon ICON_DETACHED_OFFLINE = new ImageIcon(Lock.class.getResource("light_red_locked_24.png"), "locked light red");
    static public final ImageIcon ICON_ATTACHED = new ImageIcon(Lock.class.getResource("green_locked_24.png"), "locked green");
    static public final ImageIcon ICON_ATTACHED_OFFLINE = new ImageIcon(Lock.class.getResource("light_green_locked_24.png"), "locked green");

    static public final EnumMap<LockService.State, ImageIcon> ICON = new EnumMap<>(LockService.State.class);
    static {
        ICON.put(LockService.State.LOCKED, ICON_LOCKED);
        ICON.put(LockService.State.UNLOCKED, ICON_UNLOCKED);
        ICON.put(LockService.State.DETACHED, ICON_DETACHED);
        ICON.put(LockService.State.ATTACHED, ICON_ATTACHED);
    }
    static public final EnumMap<LockService.State, ImageIcon> ICON_OFFLINE = new EnumMap<>(LockService.State.class);
    static {
        ICON_OFFLINE.put(LockService.State.LOCKED, ICON_LOCKED_OFFLINE);
        ICON_OFFLINE.put(LockService.State.UNLOCKED, ICON_UNLOCKED_OFFLINE);
        ICON_OFFLINE.put(LockService.State.DETACHED, ICON_DETACHED_OFFLINE);
        ICON_OFFLINE.put(LockService.State.ATTACHED, ICON_ATTACHED_OFFLINE);
    }
    
    
    private final String agent;
    
    private boolean online;
    private LockService.State state;
    
    private ArrayList<Listener> listeners;
    
    private final ChangeListener lockerListener = e -> update();
    private final LockService.Listener lockServiceListener = new LockService.Listener() {
        @Override
        public void agentsAdded(Locker... agents) {
            for (Locker locker : agents) {
                if (locker.getName().equals(agent)) {
                    locker.addListener(lockerListener);
                    break;
                }
            }
            update();
        }
        @Override
        public void agentsRemoved(Locker... agents) {
            for (Locker locker : agents) {
                if (locker.getName().equals(agent)) {
                    locker.removeListener(lockerListener);
                    break;
                }
            }
            update();
        }
        @Override
        public void userChanged(String oldUserID, String newUserID) {
            update();
        }
    };


// -- Life cycle : -------------------------------------------------------------

    /**
     * Constructs the widget.
     * @param agentName Subsystem name.
     */
    public Lock(String agentName) {
        agent = agentName;
    }

    @Override
    public void removeNotify() {
        LockService serv = LockService.getService();
        serv.removeListener(lockServiceListener);
        Locker locker = serv.getAgent(agent);
        if (locker != null) {
            locker.removeListener(lockerListener);
        }
        super.removeNotify();
    }

    @Override
    public void addNotify() {
        super.addNotify();
        LockService serv = LockService.getService();
        serv.addListener(lockServiceListener);
        Locker locker = serv.getAgent(agent);
        if (locker != null) {
            locker.addListener(lockerListener);
        }
        update();
    }
    

// -- Getters : ----------------------------------------------------------------

    /**
     * Returns current state for the agent associated with this widget.
     * @return Current state.
     */
    public LockService.State getState() {
        return state;
    }
    
    /**
     * Checks if the agent is online associated with this widget.
     * @return {@code true} if the agent is online.
     */
    public boolean isOnline() {
        return online;
    }
    
    /**
     * Returns icon corresponding to the current state of the specified agent.
     * @param agentName Agent name.
     * @return Icon.
     */
    static public ImageIcon getIcon(String agentName) {
        Locker locker = LockService.getService().getAgent(agentName);
        if (locker == null) {
            return ICON_UNLOCKED_OFFLINE;
        } else {
            LockService.State state = locker.getState();
            return locker.isOnline() ? ICON.get(state) : ICON_OFFLINE.get(state);
        }
    }
    
    
// -- Populate label : ---------------------------------------------------------
    
    private void update() {
        Locker locker = LockService.getService().getAgent(agent);
        if (locker == null) {
            online = false;
            state = LockService.State.UNLOCKED;
            setIcon(ICON_UNLOCKED_OFFLINE);
            setToolTipText("Subsystem offline, unlocked");
            setEnabled(true);
        } else {
            online = locker.isOnline();
            state = locker.getState();
            if (locker.isAdjusting()) {
                setEnabled(false);
                setToolTipText("Waiting for response from Lock Manager");
            } else {
                setEnabled(true);
            }
            switch (state) {
                case UNLOCKED:
                    if (online) {
                        setIcon(ICON_UNLOCKED);
                        setToolTipText("Unlocked");
                    } else {
                        setIcon(ICON_UNLOCKED_OFFLINE);
                        setToolTipText("Unlocked, Offline");
                    }
                    break;
                case LOCKED:
                    AgentLock lock = locker.getLock();
                    StringBuilder sb = new StringBuilder().append("Locked by ").append(lock.getOwner());
                    if (lock instanceof AgentLockInfo lockInfo) {
                        Instant time = lockInfo.getTimeStamp().getUTCInstant();
                        sb.append(" since ").append(Const.DEFAULT_DT_FORMAT.format(time));
                    }
                    sb.append(".").toString();
                    if (online) {
                        setIcon(ICON_LOCKED);
                        setToolTipText(sb.toString());
                    } else {
                        setIcon(ICON_LOCKED_OFFLINE);
                        setToolTipText(sb.append(" Offline.").toString());
                    }
                    break;
                case DETACHED:
                    if (online) {
                        setIcon(ICON_DETACHED);
                        setToolTipText("Locked, Detached");
                    } else {
                        setIcon(ICON_DETACHED_OFFLINE);
                        setToolTipText("Locked, Detached, Offline");
                    }
                    break;
                case ATTACHED:
                    if (online) {
                        setIcon(ICON_ATTACHED);
                        setToolTipText("Locked, level "+ locker.getLevel());
                    } else {
                        setIcon(ICON_ATTACHED_OFFLINE);
                        setToolTipText("Locked, Offline");
                    }
                    break;
            }
        }
        if (listeners != null) {
            listeners.forEach(l -> l.update(this));
        }
    }

    @Override
    public JPopupMenu getComponentPopupMenu() {
        LockService lockService = LockService.getService();
        Locker locker = lockService.getAgent(agent);
        if (locker == null || locker.isAdjusting()) {
            return null;
        } else {
            AgentLock lock = locker.getLock();
            boolean canSetLevel = lock == null || lock.getMaxLevel() > 0;
            JPopupMenu menu = new JPopupMenu();
            switch (locker.getState()) {
                case UNLOCKED:
                    menu.add(LockService.Operation.LEVEL.getAction(agent));
                    menu.add(LockService.Operation.LOCK.getAction(agent));
                    break;
                case LOCKED:
                    menu.add(LockService.Operation.DESTROY.getAction(agent));
                    break;
                case DETACHED:
                    if (canSetLevel) menu.add(LockService.Operation.LEVEL.getAction(agent));
                    menu.add(LockService.Operation.ATTACH.getAction(agent));
                    menu.add(LockService.Operation.UNLOCK.getAction(agent));
                    break;
                case ATTACHED:
                    if (canSetLevel) menu.add(LockService.Operation.LEVEL.getAction(agent));
                    menu.add(LockService.Operation.DETACH.getAction(agent));
                    menu.add(LockService.Operation.UNLOCK.getAction(agent));
                    break;
            }
            return menu;
        }        
    }
    
    
// -- Handling listeners : -----------------------------------------------------
    
    static public interface Listener {
        void update(Lock lock);
    }
    
    public void addListener(Listener listener) {
        if (listeners == null) {
            listeners = new ArrayList<>(1);
        } else {
            listeners.ensureCapacity(listeners.size()+1);
        }
        listeners.add(listener);
    }
    
    public boolean removeListener(Listener listener) {
        boolean out = false;
        Iterator<Listener> it = listeners.iterator();
        while (it.hasNext()) {
            if (listener == it.next()) {
                it.remove();
                out = true;
            }
        }
        if (listeners.isEmpty()) {
            listeners = null;
        } else {
            listeners.trimToSize();
        }
        return out;
    }
    
    public boolean removeAllListeners() {
        boolean out = listeners != null;
        listeners = null;
        return out;
    }

}
