package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.FontMetrics;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.border.Border;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableColumn;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.gconsole.services.lock.Lock;

/**
 * Renderer for monitoring data table cells.
 * Expects an instance of {@link FormattedValue} as the value of the cell.
 *
 * @author onoprien
 */
public class MonitorTableCellRenderer extends DefaultTableCellRenderer {

// -- Fields : -----------------------------------------------------------------
    
    private boolean ensureWidth = false;
    private boolean ensureHeight = false;
    private Border border;
    private boolean extraPadding = true;
    
    
// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the flag that determines whether the table column width should increase if necessary to accommodate new values.
     * The default value is {@code false}.
     * 
     * @param ensureWidth If {@code true}, the table column width will increase if necessary to accommodate new values.
     */
    public void setEnsureWidth(boolean ensureWidth) {
        this.ensureWidth = ensureWidth;
    }

    /**
     * Sets the flag that determines whether the table row height should increase if necessary to accommodate new values.
     * The default value is {@code false}.
     * 
     * @param ensureHeight If {@code true}, the table row height will increase if necessary to accommodate new values.
     */
    public void setEnsureHeight(boolean ensureHeight) {
        this.ensureHeight = ensureHeight;
    }
    
    /**
     * Sets additional padding for the table cells.
     * Positive integers are interpreted as pixels.
     * Absolute value of negative numbers is interpreted as a number of space characters.
     * The specified extra padding is added to the padding provided by {@code DefaultTableCellRenderer}.
     * The default extra padding is 0, -1, 0, -1.
     * 
     * @param top Top padding.
     * @param left Left padding.
     * @param bottom Bottom padding.
     * @param right Right padding.
     */
    public void setPadding(int top, int left, int bottom, int right) {
        if (top == 0 && left == 0 && bottom == 0 && right == 0) {
            extraPadding = false;
            border = null;
        } else {
            extraPadding = true;
            FontMetrics fm = getFontMetrics(getFont());
            int charHeight = fm.getAscent();
            int charWidth = fm.charWidth(' ');
            if (top < 0) top = -top * charHeight;
            if (left < 0) left = -left * charWidth;
            if (bottom < 0) bottom = -bottom * charHeight;
            if (right < 0) right = -right * charWidth;
            border = BorderFactory.createEmptyBorder(top, left, bottom, right);
        }
    }

// -- Rendering : --------------------------------------------------------------
    
    @Override
    public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
                
        if (value instanceof FormattedValue) {
            FormattedValue fv = (FormattedValue) value;
            super.getTableCellRendererComponent(table, fv.getText(), false, false, row, column);
            if (fv.value instanceof AgentLock) {
                setEnsureHeight(true);
                AgentLock lock = (AgentLock) fv.value;
                ImageIcon icon = Lock.getIcon(lock.getAgentName());
                setIcon(icon);
            } else {
                setIcon(null);
            }
            setBackground(fv.getBgColor());
            setForeground(fv.getFgColor());
            setHorizontalAlignment(fv.getHorizontalAlignment());
            setToolTipText(fv.getToolTip());
        } else if (value == null) {
            super.getTableCellRendererComponent(table, " ", false, false, row, column);
            TableColumn c = table.getColumnModel().getColumn(column);
            setBackground(MonitorField.COLOR_OFF);
            setHorizontalAlignment(SwingUtilities.CENTER);
        } else {
            super.getTableCellRendererComponent(table, value.toString(), false, false, row, column);
            setHorizontalAlignment(SwingUtilities.CENTER);
        }
        
        if (extraPadding) {
            if (border == null) {
                setPadding(0, -1, 0, -1); // default, setPadding(...) has not been called before
            }
            Border b = getBorder();
            if (b == null) {
                setBorder(border);
            } else {
                setBorder(BorderFactory.createCompoundBorder(b, border));
            }
        }

        if (ensureWidth) {
            int w = getMinimumSize().width + 2;
            TableColumn c = table.getColumnModel().getColumn(column);
            if (w > c.getMinWidth()) {
                c.setMinWidth(w);
            }
        }

        if (ensureHeight) {
            int h = getMinimumSize().height;
            if (h > table.getRowHeight(row)) {
                table.setRowHeight(row, h);
            }
        }
        
        return this;
    }

    @Override
    public Dimension getMinimumSize() {
        return getPreferredSize();
    }

}
