package org.lsst.ccs.gconsole.plugins.alert;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.time.Instant;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JTextPane;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.ToolTipManager;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultCaret;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyleContext;
import javax.swing.text.StyledDocument;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.MutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.MutedAlertRequest;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RaisedAlertInstance;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.Options;
import org.lsst.ccs.gconsole.agent.command.CommandTask;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import static org.lsst.ccs.gconsole.base.Const.HSPACE;
import static org.lsst.ccs.gconsole.base.Const.VDIM;
import static org.lsst.ccs.gconsole.base.Const.VDIM2;
import static org.lsst.ccs.gconsole.base.Const.VSPACE;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.gconsole.services.lock.LockService;
import org.lsst.ccs.gconsole.services.lock.Locker;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.gconsole.util.swing.CompositeIcon;
import org.lsst.ccs.gconsole.util.tree.TreeUtil;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;

/**
 * Display of CCS alert summary.
 * All methods except {@link #onAlert} should be called on EDT.
 *
 * @author onoprien
 */
public class AlertViewer implements AlertListener {

// -- Fields : -----------------------------------------------------------------
    
    private final LsstAlertPlugin plugin;
    private Descriptor config;
    
    private AlertTree tree;
    private JTextPane infoPanel;
    private JPanel rightPanel;
    private JScrollPane leftPanel;
    private JSplitPane rootPanel;
    
    private Action clearAction, ackAction, muteAction;
    private MuteListDialog muteListDialog; // non-null when open
    
    private JCheckBox freezeBox;
    private JButton muteButton;
    
    boolean historyProp, soundProp, toFrontProp, selectNewProp, ignoreMutedProp, ignoreAckProp;
    
    private StyledDocument document;
    private Style attBase, attBlack, attBlackBold;
    private final EnumMap<AlertState,Style> attPlane = new EnumMap<>(AlertState.class);
    private final EnumMap<AlertState,Style> attBold = new EnumMap<>(AlertState.class);
    
    private StatusMessageListener messageListener;
    private final TreeMap<String,TreeMap<String,MutedAlertRequest>> muteRequests = new TreeMap<>(); // agent -> { regex -> MutedAlertRequest}; all access on EDT
    static private final String MUTE_ALL = ".*";
    
    private LockService.Listener lockListener;
    private final Map<String,Locker> lockers = new HashMap<>();
    
    static final EnumMap<AlertState,Color> COLOR = new EnumMap<>(AlertState.class);
    static {
        COLOR.put(AlertState.NOMINAL, new Color(0, 0xAA, 0));
        COLOR.put(AlertState.WARNING, new Color(0, 0, 0xAA));
        COLOR.put(AlertState.ALARM, new Color(0xAA, 0, 0));
    }
    static private final ImageIcon ACK_ICON = new ImageIcon(AlertViewer.class.getResource("check_16.png"), "acknowledged");
    static private final ImageIcon MUTE_ICON = new ImageIcon(AlertViewer.class.getResource("mute_16.png"), "mute");

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Creates an instance of {@code AlertViewer}.
     * No GUI panels are constructed and no resources are allocated at this point.
     * @param plugin Alert plugin instance that created this viewer.
     * @param configuration
     */
    public AlertViewer(LsstAlertPlugin plugin, Descriptor configuration) {
        this.plugin = plugin;
        config = configuration == null ? new Descriptor() : configuration;
        historyProp = (boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_HISTORY);
        soundProp = (boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_SOUND);
        toFrontProp = (boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_TOFRONT);
        selectNewProp = (boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_SELECT);
        ignoreAckProp = (boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_IGNOREACK);
        ignoreMutedProp = (boolean) plugin.getServices().getProperty(LsstAlertPlugin.OPT_IGNOREMUTED);
    }
    
    /**
     * Wake up and get ready to display alerts.
     * This method is called if an alert event is received while this viewer is asleep,
     * before the event is handled.
     */
    private void start() {
        
        muteAction = new AbstractAction("Mute") {
            @Override public void actionPerformed(ActionEvent e) {
                AlertNode node = tree.getSelectedNode();
                if (node == null || node == tree.getRoot()) return;
                boolean toMute = "Mute".equals(getValue(NAME).toString());
                String agent = node.getAgentName();
                if (toMute) {
                    String regEx;
                    if (node.getLevel() == 1) {
                        regEx = MUTE_ALL;
                    } else if (node instanceof IDNode idNode) {
                        regEx = idNode.getID();
                    } else {
                        regEx = node.getIDPrefix() + MUTE_ALL;
                    }
                    MuteDialog.show(rootPanel, agent, regEx);
                } else {
                    MutedAlertRequest r = node.getMuteRequest();
                    if (r != null) {
                        AlertViewer.sendCommandToMMM("cancelMutedAlertRequest", r.getUniqueId());
                    } else if (node.getLevel() == 1) {
                        AlertViewer.sendCommandToMMM("cancelAllMutedAlertRequestForSubsystem", node.getAgentName());
                    }
                }
            }
        };
                
        Box buttonPanel = Box.createHorizontalBox();
        
        clearAction = new AbstractAction("Clear") {
            @Override public void actionPerformed(ActionEvent e) {
                AlertNode node = tree.getSelectedNode();
                if (node == null) return;
                String agent = node.getAgentName();
                String[] alerts = node.getIDs();
                AlertState maxSeverity = node.getAlertLevel();
                new SwingWorker<Void, Void>() {
                    @Override
                    protected Void doInBackground() throws Exception {
                        plugin.clearAlerts(agent, maxSeverity, alerts);
                        return null;
                    }
                }.execute();
            }
        };
        clearAction.putValue(Action.SHORT_DESCRIPTION, "Clear selected alerts.");
        buttonPanel.add(new JButton(clearAction));
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
        
        muteButton = new JButton("Mute");
        muteButton.setToolTipText("Edit mute requests.");
        muteButton.addActionListener(e -> {
            AlertNode node = tree.getSelectedNode();
            String agent = null;
            String regex = null;
            if (node != null) {
                agent = node.getAgentName();
                MutedAlertRequest r = node.getMuteRequest();
                if (r != null) {
                    regex = r.getAlertIdRegEx();
                }
            }
            muteListDialog = MuteListDialog.show(rootPanel, muteRequests, agent, regex);
            muteListDialog.setVisible(true);
            muteListDialog = null;
        });
        buttonPanel.add(muteButton);
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
                
        ackAction = new AbstractAction("Acknowledge") {
            @Override public void actionPerformed(ActionEvent e) {
                AlertNode node = tree.getSelectedNode();
                if (node == null) return;
                AckDialog d = AckDialog.show(node, rootPanel);
                if (d != null) {
                    String message = d.message;
                    String[] toAck = d.alertsToAck;
                    String[] toNack = d.alertsToNack;
                    String agent = node.getAgentName();
                    new SwingWorker<Void, Void>() {
                        @Override
                        protected Void doInBackground() throws Exception {
                            plugin.ackAlerts(message, agent, toAck, toNack);
                            return null;
                        }
                    }.execute();
                }
            }
        };
        ackAction.putValue(Action.SHORT_DESCRIPTION, "Acknowledge or un-acknowledge selected alerts.");

        freezeBox = new JCheckBox("Freeze");
        freezeBox.setSelected(false);
        freezeBox.addActionListener(e -> {
            if (!freezeBox.isSelected()) {
                tree.getSelectedNode().fillInfoPanel(this);
            }
        });
        freezeBox.setToolTipText("Freeze information panel content");
        buttonPanel.add(freezeBox);
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
        
        JButton settingsButton = new JButton("Settings");
        settingsButton.addActionListener(e -> {
            SettingsPanel sp = new SettingsPanel();
            int out = JOptionPane.showConfirmDialog(settingsButton, sp, "Alert Viewer Settings", JOptionPane.OK_CANCEL_OPTION);
            if (out == JOptionPane.OK_OPTION) {
                sp.save();
            }
        });
        settingsButton.setToolTipText("Modify alert viewer settings");
        buttonPanel.add(settingsButton);
        buttonPanel.add(Box.createRigidArea(new Dimension(2*HSPACE, VSPACE)));
        buttonPanel.add(Box.createHorizontalGlue());
        buttonPanel.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
    
        tree = new AlertTree();
        tree.init();
        tree.addTreeSelectionListener(e -> {
            AlertNode node = tree.getSelectedNode();
            if (node == null) { // nothing selected
                if (!freezeBox.isSelected()) {
                    infoPanel.setText("");
                }
            } else {
                if (!freezeBox.isSelected()) {
                    node.fillInfoPanel(this);
                }
            }
            enableActions();
        });
        tree.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
        tree.addMouseListener(new MouseAdapter() {
            @Override public void mouseClicked(MouseEvent e) {
                if (freezeBox.isSelected()) freezeBox.doClick();
            }
        });
    
        infoPanel = new JTextPane();
        infoPanel.setEditable(false);
        infoPanel.setContentType("text/plain");
        infoPanel.setBorder(BorderFactory.createEmptyBorder(VSPACE, VSPACE, VSPACE, VSPACE));
        try {
            DefaultCaret caret = (DefaultCaret) infoPanel.getCaret();
            caret.setUpdatePolicy(DefaultCaret.NEVER_UPDATE);
        } catch (ClassCastException x) {
        }
        infoPanel.setText("");
        document = infoPanel.getStyledDocument();
        Style style = StyleContext.getDefaultStyleContext().getStyle(StyleContext.DEFAULT_STYLE);
        attBase = document.addStyle("attBase", style);
        attBlack = document.addStyle("attBlack", attBase);
        attBlackBold = document.addStyle("attBlackBold", attBase);
        StyleConstants.setBold(attBlackBold, true);
        for (AlertState state : AlertState.values()) {
            style = document.addStyle(state.name(), attBase);
            StyleConstants.setForeground(style, COLOR.get(state));
            attPlane.put(state, style);
            style = document.addStyle(state.name(), style);
            StyleConstants.setBold(style, true);
            attBold.put(state, style);
        }
        
        rightPanel = new JPanel(new BorderLayout());
        JScrollPane scroll = new JScrollPane(infoPanel);
        rightPanel.add(scroll, BorderLayout.CENTER);
        rightPanel.add(buttonPanel, BorderLayout.SOUTH);
        
        leftPanel = new JScrollPane(tree);
        
        rootPanel = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, leftPanel, rightPanel);
        Dimension minimumSize = new Dimension(100, 50);
        leftPanel.setMinimumSize(minimumSize);
        rightPanel.setMinimumSize(minimumSize);
        rootPanel.setOneTouchExpandable(false);
        rootPanel.setResizeWeight(.3);
        rootPanel.resetToPreferredSizes();
        
        Map<Object, Object> par = new HashMap<>();
        DataPanelDescriptor panDesc = config.getPage();
        if (panDesc != null && panDesc.isOpen()) {
            Map<String, Serializable> data = panDesc.getData();
            if (data != null) {
                par.putAll(data);
            }
        }
        par.put(Panel.TITLE, "Alerts");
        Consumer<JComponent> onClose = c -> stop();
        par.put(Panel.ON_CLOSE, onClose);
        Console.getConsole().getPanelManager().open(rootPanel, par);
        
        messageListener = msg -> {
            try {
                StatusSubsystemData ssd = (StatusSubsystemData) msg;
                String key = ssd.getDataKey();
                if (MutedAlertRequest.PUBLICATION_KEY.equals(key)) {
                    KeyValueData kv = ssd.getSubsystemData();
                    List<MutedAlertRequest> mutes = (List<MutedAlertRequest>) kv.getValue();
                    SwingUtilities.invokeLater(() -> update(new ArrayList<>(mutes)));
                }
            } catch (RuntimeException x) {
            }
        };
        AgentMessagingLayer aml = plugin.getConsole().getMessagingAccess();
        aml.addStatusMessageListener(messageListener, msg -> msg.getOriginAgentInfo().getType() == AgentType.MMM && msg instanceof StatusSubsystemData);
        sendCommandToMMM("publishListOfMutedAlerts");
        
        lockListener = new LockService.Listener() {
            @Override
            public void agentsAdded(Locker... agents) {
                for (Locker a : agents) {
                    lockers.put(a.getName(), a);
                }
                enableActions();
            }
            @Override
            public void agentsRemoved(Locker... agents) {
                for (Locker a : agents) {
                    lockers.remove(a.getName());
                }
                enableActions();
            }
            @Override
            public void agentsUpdated(Locker... agents) {
                agentsAdded(agents);
            }
            @Override
            public void userChanged(String oldUserID, String newUserID) {
                enableActions();
            }
        };
        LockService.getService().addListener(lockListener);
    }

    /** Release resources and go to sleep until the next event is received. */
    void stop() {
        if (rootPanel == null) return;
        plugin.getConsole().getMessagingAccess().removeStatusMessageListener(messageListener);
        LockService.getService().removeListener(lockListener);
        lockListener = null;
        config = save();
        clearAction = null;
        tree = null;
        infoPanel = null;
        rightPanel = null;
        leftPanel = null;
        rootPanel = null;
    }
    
    /** Cleanup before this viewer is permanently destroyed. */
    void destroy() {
        if (rootPanel != null) {
            Console.getConsole().getPanelManager().close(rootPanel);
        }
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    JComponent getGUI() {
        return rootPanel;
    }


// -- Processing alert events --------------------------------------------------
    
    @Override
    public void onAlert(AlertEvent event) {
        if (SwingUtilities.isEventDispatchThread()) {
            update(event);
        } else {
            SwingUtilities.invokeLater(() -> update(event));
        }
    }

    private void update(AlertEvent event) {
        
        String source = event.getSource(); // agent name
        RaisedAlertSummary summary = event.getSummary();
        
        // Process agent disconnects
        
        if (summary == null) {
            if (tree != null) {
                 AlertNode sourceNode = getAgentNode(source);
                if (sourceNode != null) {
                    sourceNode.removeFromParent();
                    tree.getModel().reload();
                    tree.clearSelection();
                }
            }
            return;
        }
        
        // Start viewer if necessary
        
        Set<RaisedAlertHistory> currentAlerts = summary.getAllRaisedAlertHistories();
        if (tree == null) {
            if (currentAlerts.isEmpty()) {
                return;
            } else {
                start();
            }
        }
        
        // Update tree model

        TreePath selPath = tree.getSelectionPath();
        
        AlertNode sourceNode = getAgentNode(source);        
        if (sourceNode == null) { // tree has no alerts from this source agent
            if (currentAlerts.isEmpty()) {
                return; // and the event does not have any alerts either
            }
            sourceNode = new AlertNode(source);
            for (RaisedAlertHistory h : currentAlerts) {
                updateAlert(sourceNode, h);
            }
            tree.getRoot().add(sourceNode);
        } else {
            HashMap<String,RaisedAlertHistory> id2history = new HashMap<>(currentAlerts.size()*2);
            currentAlerts.forEach(h -> id2history.put(h.getLatestAlert().getAlertId(), h));
            for (IDNode idNode : sourceNode.getLeaves()) {
                String id = idNode.getID();
                RaisedAlertHistory history = id2history.remove(id);
                if (history == null) {
                    removeAlert(idNode);
                } else {
                    updateAlert(sourceNode, history);
                }
            }
            for (RaisedAlertHistory h : id2history.values()) {
                updateAlert(sourceNode, h);
            }
        }
        
        // Update tree
        
        ArrayList<TreePath> state = TreeUtil.saveExpansionState(tree);
        tree.getModel().reload();
        TreeUtil.restoreExpansionState(tree, state);
        
        boolean ignore = ignore(event);

        TreePath path;     
        if (selectNewProp && !ignore) {
            String id = "";
            Alert alert = event.getAlert();
            if (alert == null) {
                for (String clearedID : event.getClearedIds()) {
                    try {
                        if (event.isPartialClear(clearedID)) {
                            id = clearedID;
                            break;
                        }
                    } catch (RuntimeException x) {
                    }
                }
            } else {
                id = alert.getAlertId();
            }
            IDNode idNode = id.isEmpty() ? null : tree.findID(sourceNode, id);
            path = new TreePath(idNode == null ? new Object[]{tree.getRoot(), sourceNode} : idNode.getPath());
        } else {
            path = selPath;
        }
        
        if (isValidPath(path)) {
            tree.setSelectionPath(path);
            tree.expandPath(path);
        } else {
            tree.clearSelection();
        }
        
        // Get user's attention
        
        if (!ignore && !soundProp) {
            Toolkit.getDefaultToolkit().beep();
        }
        if (toFrontProp && !ignore) {
            toFront();
        }
        
    }
    
    private void update(List<MutedAlertRequest> mutes) {
        muteRequests.clear();
        mutes.forEach(r -> {
            String agent = r.getSubsystemName();
            TreeMap<String,MutedAlertRequest> mm = muteRequests.computeIfAbsent(agent, a -> new TreeMap<>());
            mm.put(r.getAlertIdRegEx(), r);
        });
        if (tree == null) return;
        for (Enumeration<TreeNode> e = tree.getRoot().breadthFirstEnumeration(); e.hasMoreElements();) {
            tree.getModel().nodeChanged(e.nextElement());
        }
        if (muteListDialog != null) {
            muteListDialog.update();
        }
    }
    
    void toFront() {
        if (rootPanel == null) { // asleep
            AlertService serv = plugin.getConsole().getAgentService(AlertService.class);
            List<AlertEvent> events = new LinkedList<>();
            AlertListener temp = e -> events.add(e);
            serv.addListener(temp); // fill the list of events; not calling update(e) here because the service executes callbacks while holding a lock
            serv.removeListener(temp);
            events.forEach(e -> update(e));
        } else {
            PanelManager panMan = plugin.getConsole().getPanelManager();
            try {
                panMan.set(rootPanel, Panel.SELECTED, true);
            } catch (RuntimeException x) {
            }
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private AlertNode getAgentNode(String agentName) {
        AlertNode root = tree.getRoot();
        for (int i=0; i<root.getChildCount(); i++) {
            AlertNode node = root.getChildAt(i);
            if (node.getUserObject().equals(agentName)) return (AlertNode) node;
        }
        return null;
    }
    
    static public String formatTimeStamp(Instant instant) {
        return Const.DEFAULT_DT_FORMAT.format(instant);
    }
    
    static public String formatTimeStamp(long time) {
        return formatTimeStamp(Instant.ofEpochMilli(time));
    }
    
    private void updateAlert(AlertNode sourceNode, RaisedAlertHistory history) {
        
        String id = history.getLatestAlert().getAlertId();
        String[] ss = id.split("/");
        ArrayList<String> tokens = new ArrayList<>(ss.length);
        for (String s : ss) {
            s = s.trim();
            if (!s.isEmpty()) tokens.add(s);
        }
        if (tokens.isEmpty()) return;
        
        for (int i=0; i<tokens.size()-1; i++) {
            String s = tokens.get(i);
            AlertNode tokenNode = null;
            Enumeration e = sourceNode.children();
            while (e.hasMoreElements()) {
                AlertNode node = (AlertNode) e.nextElement();
                if (!node.isLeaf()) {
                    if (node.getUserObject().equals(s)) {
                        tokenNode = node;
                        break;
                    }
                }
            }
            if (tokenNode == null) {
                tokenNode = new AlertNode(s);
                sourceNode.add(tokenNode);
            }
            sourceNode = tokenNode;
        }
        
        IDNode idNode = null;
        Enumeration e = sourceNode.children();
        while (e.hasMoreElements()) {
            Object o = e.nextElement();
            if (o instanceof IDNode) {
                IDNode node = (IDNode) o;
                if (node.getID().equals(id)) {
                    idNode = node;
                    break;
                }
            }
        }
        
        if (idNode == null) {
            idNode = new IDNode(history);
            sourceNode.add(idNode);
        } else {
            idNode.setHistory(history);
        }
        
    }
    
    private void removeAlert(IDNode idNode) {
        AlertNode node = idNode;
        while (true) {
            AlertNode parent = node.getParent();
            if (parent == null) return;
            if (parent.isRoot() || parent.getChildCount() > 1) {
                node.removeFromParent();
                return;
            } else {
                node = parent;
            }
        }
    }
    
    private boolean isValidPath(TreePath path) {
        if (path == null) return false;
        int n = path.getPathCount();
        if (n == 0) {
            return false;
        } else {
            AlertNode child = (AlertNode) path.getLastPathComponent();
            for (int i=n-2; i>=0; i--) {
                AlertNode parentFromPath = (AlertNode) path.getPathComponent(i);
                AlertNode parentFromChild = child.getParent();
                if (parentFromPath != null && parentFromChild != null && parentFromPath.equals(parentFromChild)) {
                    child = parentFromPath;
                } else {
                    return false;
                }
            }
            return child.equals(tree.getRoot());
        }
    }
    
    private boolean isAcknowledged(AlertEvent event) {
        RaisedAlertSummary sum = event.getSummary();
        if (sum != null) {
            Alert alert = event.getAlert();
            if (alert != null) {
                RaisedAlertHistory hist = sum.getRaisedAlert(alert.getAlertId());
                if (hist != null) {
                    return hist.isAcknowledged();
                }
            }
        }
        return false;
    }
    
    private boolean isMuted(String agent, String alertID) {
        TreeMap<String, MutedAlertRequest> mm = muteRequests.get(agent);
        if (mm != null) {
            for (MutedAlertRequest r : mm.values()) {
                if (r.isValid() && r.matches(alertID)) {
                    return true;
                }
            }
        }
        return false;
    }
    
    private boolean ignore(AlertEvent event) {
        return switch (event.getType()) {
            case ALERT_RAISED -> {
                if (ignoreMutedProp && isMuted(event.getSource(), event.getAlert().getAlertId())) {
                    yield true;
                }
                if (ignoreAckProp && isAcknowledged(event)) {
                    yield true;
                }
                yield false;
            }
            case ALERT_CLEARED -> {
                if (ignoreMutedProp) {
                    String agent = event.getSource();
                    for (String alertID : event.getClearedIds()) {
                        if (!isMuted(agent, alertID)) {
                            yield false;
                        }
                    }
                    yield true;
                 }
                yield false;
            }
            default -> false;
        };
    }
    
    private void enableActions() {
        AlertNode node = tree.getSelectedNode();
        String mmm = getMmmName();
        if (node == null || node == tree.getRoot()) {
            if (clearAction != null) clearAction.setEnabled(false);
            if (ackAction != null) ackAction.setEnabled(false);
            if (muteAction != null) muteAction.setEnabled(false);
        } else {
            String agentName = node.getAgentName();
            Locker locker = lockers.get(agentName);
            if (clearAction != null) {
                if (locker == null) {
                    clearAction.setEnabled(false);
                } else {
                    clearAction.setEnabled(locker.getMaxLevel() != -1);
                }
            }
            if (ackAction != null) {
                ackAction.setEnabled(true);
            }
            if (muteAction != null) {
                if (mmm == null) {
                    muteAction.setEnabled(false);
                } else {
                    Locker mmmLocker = lockers.get(agentName);
                    muteAction.setEnabled(mmmLocker != null && mmmLocker.getMaxLevel() != -1 && (locker == null || locker.getMaxLevel() != -1));
                }
            }
        }
        if (muteButton != null) {
            muteButton.setEnabled(mmm != null);
        }
    }
    

// -- Alert tree utilities : ---------------------------------------------------
    
    private final class AlertTree extends JTree implements HasPopupItems {
        
        private void init() {
            setModel(new AlertTreeModel());
            getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
            setCellRenderer(new AlertTreeRenderer());
            ToolTipManager.sharedInstance().registerComponent(this);
        }
        
        public AlertNode getSelectedNode() {
            return (AlertNode) getLastSelectedPathComponent();
        }
        
        @Override
        public AlertTreeModel getModel() {
            return (AlertTreeModel) super.getModel();
        }
        
        AlertNode getRoot() {
            return getModel().getRoot();
        }
                
        IDNode findID(AlertNode sourceNode, String id) {
            if (sourceNode == null) return null;
            Enumeration e = sourceNode.depthFirstEnumeration();
            while (e.hasMoreElements()) {
                Object node = e.nextElement();
                if (node instanceof IDNode && ((IDNode)node).getID().equals(id))  {
                    return (IDNode)node;
                }
            }
            return null;
        }

        @Override
        public JPopupMenu modifyPopupMenu(JPopupMenu menu, Component cmpnt, Point point) {
            TreePath tp = getPathForLocation(point.x, point.y);
            if (tp != null) {
                setSelectionPath(tp);
                AlertNode node = getSelectedNode();
                if (node != null && node != getRoot()) {
                    menu.insert(new JSeparator(), 0);
                    menu.insert(ackAction, 0);
                    menu.insert(clearAction, 0);
                    muteAction.putValue(Action.NAME, node.getMuteRequest() == null ?  "Mute" : "Unmute");
                    menu.insert(muteAction, 0);
                }
            }
            return menu;
        }
    }
    
    private class AlertTreeModel extends DefaultTreeModel {
        
        AlertTreeModel() {
            super(new AlertNode("CCS Alerts") {
                
                @Override
                ImageIcon getIcon() {
                    return null;
                }

                @Override
                void fillInfoPanel(AlertViewer v) {
                    int activeAlerts = 0;
                    if (children != null) {
                        for (Object child : children) {
                            if (((AlertNode)child).getAlertLevel() != AlertState.NOMINAL) {
                                activeAlerts++;
                            }
                        }
                    }
                    StyledDocument d = v.document;
                    try {
                        d.remove(0, d.getLength());
                        d.insertString(d.getLength(), "CCS Control System\n\n", v.attBlackBold);
                        d.insertString(d.getLength(), "Subsystems with active alerts: ", v.attBlack);
                        d.insertString(d.getLength(), activeAlerts + "\n", v.attBlackBold);
                        d.insertString(d.getLength(), "Overall alert level: ", v.attBlack);
                        d.insertString(d.getLength(), getAlertLevel() +"\n", v.attBold.get(getAlertLevel()));
                        d.insertString(d.getLength(), "Last changed at ", v.attBlack);
                        d.insertString(d.getLength(), Const.DEFAULT_DT_FORMAT.format(getTimeStamp()) + "\n", v.attBlackBold);
                        v.infoPanel.setCaretPosition(0);
                    } catch (BadLocationException x) {
                    }
                }

                @Override
                String getAgentName() {
                    throw new UnsupportedOperationException("Trying to look up source subsystem for the root node.");
                }
                
            });
        }
        
        @Override
        public AlertNode getRoot() {
            return (AlertNode) super.getRoot();
        }
        
    }
    
    class AlertNode extends DefaultMutableTreeNode implements Comparable<AlertNode> {
        
        AlertNode(String userObject) {
            super(userObject);
        }

        @Override
        public String getUserObject() {
            return (String) super.getUserObject();
        }

        @Override
        public int compareTo(AlertNode other) {
            return getUserObject().compareTo(other.getUserObject());
        }
        
        @Override
        public void add(MutableTreeNode newChildNode) {
            AlertNode newChild = (AlertNode) newChildNode;
            Enumeration<TreeNode> e = children();
            int i = 0;
            boolean replace = false;
            while (e.hasMoreElements()) {
                AlertNode child = (AlertNode) e.nextElement();
                int c = newChild.compareTo(child);
                if (c > 0) {
                    i++;
                } else {
                    if (c == 0) replace = true;
                    break;
                }
            }
            if (replace) remove(i);
            insert(newChild, i);
        }

        @Override
        public AlertNode getParent() {
            return (AlertNode) super.getParent();
        }

        @Override
        public AlertNode getChildAt(int index) {
            return (AlertNode) super.getChildAt(index);
        }
        
        public List<IDNode> getLeaves() {
            ArrayList<IDNode> out = new ArrayList<>();
            Enumeration e = this.depthFirstEnumeration();
            while (e.hasMoreElements()) {
                Object node = e.nextElement();
                if (node instanceof IDNode idNode) {
                    out.add(idNode);
                }
            }
            return out;
        }
        
        public String[] getIDs() {
            SortedSet<String> ss = new TreeSet<>();
            getLeaves().stream().map(node -> node.getID()).forEach(id -> ss.add(id));
            return ss.toArray(new String[0]);
        }
        
        public String getIDPrefix() {
            Object[] pp = this.getUserObjectPath();
            return switch (pp.length) {
                case 1,2 -> "";
                default -> {
                    ArrayList<String> ss = new ArrayList<>(pp.length - 2);
                    for (int i=2; i<pp.length; i++) {
                        ss.add(pp[i].toString());
                    }
                    yield String.join("/", ss);
                }
            };
        }
        
        void fillInfoPanel(AlertViewer v) {
            StyledDocument d = v.document;
            try {
                d.remove(0, d.getLength());
                d.insertString(d.getLength(), "Source: ", v.attBlack);
                d.insertString(d.getLength(), getAgentName() +"\n", v.attBlackBold);
                d.insertString(d.getLength(), "Alerts selected: ", v.attBlack);
                d.insertString(d.getLength(), getLeafCount() +"\n", v.attBlackBold);
                d.insertString(d.getLength(), "Highest alert level: ", v.attBlack);
                AlertState highestState = getAlertLevel();
                d.insertString(d.getLength(), highestState +"\n", v.attBold.get(highestState));
                d.insertString(d.getLength(), "Last changed at ", v.attBlack);
                d.insertString(d.getLength(), Const.DEFAULT_DT_FORMAT.format(getTimeStamp()) +"\n", v.attBlackBold);
                v.infoPanel.setCaretPosition(0);
            } catch (BadLocationException x) {
            }
        }

        String getAgentName() {
            int level = getLevel();
            if (level > 1) {
                return getParent().getAgentName();
            } else {
                return getUserObject();
            }
        }
        
        Instant getTimeStamp() {
            Instant out = Instant.EPOCH;
            if (children != null) {
                for (Object child : children) {
                    Instant time = ((AlertNode) child).getTimeStamp();
                    if (time.compareTo(out) > 0) out = time;
                }
            }
            return out;
        }
        
        AlertState getAlertLevel() {
            AlertState alertState = AlertState.NOMINAL;
            Enumeration e = this.children();
            while (e.hasMoreElements()) {
                AlertState childState = ((AlertNode) e.nextElement()).getAlertLevel();
                if (childState.compareTo(alertState) > 0) {
                    alertState = childState;
                    if (alertState.equals(AlertState.ALARM)) break;
                }
            }
            return alertState;
        }
        
        ImageIcon getIcon() {
            return isMuted() ? MUTE_ICON : null;
        }
        
        boolean isMuted() {
            return getMuteRequest() != null;
        }
        
        MutedAlertRequest getMuteRequest() {
            Map<String,MutedAlertRequest> mm = muteRequests.get(getAgentName());
            if (mm == null) return null;
            String regex = getIDPrefix() + MUTE_ALL;
            MutedAlertRequest r = mm.get(regex);
            if (r != null && !r.isValid()) {
                mm.remove(regex);
                r = null;
            }
            return r;
        }
    }
    
    final class IDNode extends AlertNode {
        
        private RaisedAlertHistory history;
        
        IDNode(RaisedAlertHistory history) {
            super(history.getLatestAlert().getAlertId());
            setHistory(history);
        }

        void setHistory(RaisedAlertHistory history) {
            this.history = history;
        }
        
        RaisedAlertHistory getHistory() {
            return history;
        }
        
        String getID() {
            return getUserObject();
        }

        @Override
        public String toString() {
            return getLastIdComponent();
        }
        
        @Override
        public String getIDPrefix() {
            return this.getParent().getIDPrefix();
        }

        String getLastIdComponent() {
            String[] ss = getID().split("/");
            int i = ss.length;
            while (i > 0) {
                String s = ss[--i].trim();
                if (!s.isEmpty()) return s;
            }
            return getID();
        }
        
        @Override
        void fillInfoPanel(AlertViewer v) {
            Alert alert = history.getLatestAlert();
            ArrayList<RaisedAlertInstance> alerts = history.getRaisedAlertInstancesList();
            StyledDocument d = v.document;
            try {
                d.remove(0, d.getLength());
                d.insertString(d.getLength(), "Source: ", v.attBlack);
                d.insertString(d.getLength(), getAgentName() +"\n", v.attBlackBold);
                d.insertString(d.getLength(), "ID: ", v.attBlack);
                d.insertString(d.getLength(), alert.getAlertId() +"\n", v.attBlackBold);
                d.insertString(d.getLength(), "Description: ", v.attBlack);
                d.insertString(d.getLength(), alert.getDescription() +"\n", v.attBlackBold);
                d.insertString(d.getLength(), "Highest level: ", v.attBlack);
                d.insertString(d.getLength(), history.getHighestAlertState() +"\n", v.attBold.get(history.getHighestAlertState()));
                if (history.isAcknowledged()) {
                    d.insertString(d.getLength(), history.getAcknowledgement() +".\n", v.attBlack);
                }
                if (isMuted()) {
                    MutedAlertRequest req = getMuteRequest();
                    if (req == null) {
                        Map<String, MutedAlertRequest> mm = muteRequests.get(getAgentName());
                        if (mm != null) {
                            String id = getID();
                            for (MutedAlertRequest r : mm.values()) {
                                if (r.matches(id)) {
                                    if (req == null || req.getExpirationMillis() < r.getExpirationMillis()) {
                                        req = r;
                                    }
                                }
                            }
                        }
                    }
                    if (req != null) {
                        StringBuilder sb = new StringBuilder("Muted by ");
                        sb.append(req.getUserName()).append(" until ").append(Const.BRIEF_DT_FORMAT.format(Instant.ofEpochMilli(req.getExpirationMillis()))).append(".\n");
                        d.insertString(d.getLength(), sb.toString(), v.attBlack);
                    }
                }
                d.insertString(d.getLength(), "\n", v.attBlack);
                
                if (historyProp) {
                    d.insertString(d.getLength(), "History:\n\n", v.attBlackBold);
                    for (RaisedAlertInstance nextAlert : alerts) {
                            d.insertString(d.getLength(), formatTimeStamp(nextAlert.getCCSTimeStamp().getUTCInstant()) + " " + nextAlert.getAlertState() + "\n", v.attBold.get(nextAlert.getAlertState()));
                            d.insertString(d.getLength(), nextAlert.getCause() + "\n\n", v.attBlack);
                    }
                } else {
                    int n = alerts.size();
                    if (n > 1) {
                        RaisedAlertInstance last = alerts.get(n-1);
                        d.insertString(d.getLength(), "Last raised: " + formatTimeStamp(last.getCCSTimeStamp().getUTCInstant()) + " : ", v.attBlackBold);
                        d.insertString(d.getLength(), last.getAlertState() + "\n", v.attBold.get(history.getLatestAlertState()));
                        d.insertString(d.getLength(), last.getCause() + "\n\n", v.attBlack);
//                        RaisedAlertInstance first = alerts.get(0);
//                        d.insertString(d.getLength(), "First raised: " + formatTimeStamp(first.getCCSTimeStamp().getUTCInstant()) + " : ", v.attBlackBold);
//                        d.insertString(d.getLength(), first.getAlertState() + "\n", v.attBold.get(first.getAlertState()));
//                        d.insertString(d.getLength(), first.getCause() + "\n", v.attBlack);
                    } else if (n > 0) {
                        d.insertString(d.getLength(), "Raised: " + formatTimeStamp(history.getLatestAlertCCSTimeStamp().getUTCInstant()) + " : ", v.attBlackBold);
                        d.insertString(d.getLength(), history.getLatestAlertState() + "\n", v.attBold.get(history.getLatestAlertState()));
                        d.insertString(d.getLength(), alerts.get(0).getCause() + "\n", v.attBlack);
                    }
                }
                v.infoPanel.setCaretPosition(0);
            } catch (BadLocationException x) {
            }
        }
        
        @Override
        Instant getTimeStamp() {
            return history.getLatestAlertCCSTimeStamp().getUTCInstant();
        }
        
        @Override
        AlertState getAlertLevel() {
            return history.getHighestAlertState();
        }

        @Override
        ImageIcon getIcon() {
            ImageIcon out = super.getIcon(); // muted or null
            if (history != null && history.isAcknowledged()) {
                out = CompositeIcon.get(out, ACK_ICON);
            }
            return out;
        }

        @Override
        public List<IDNode> getLeaves() {
            return Collections.singletonList(this);
        }

        @Override
        public String[] getIDs() {
            return new String[] {getID()};
        }
        
        @Override
        MutedAlertRequest getMuteRequest() {
            Map<String,MutedAlertRequest> mm = muteRequests.get(getAgentName());
            if (mm == null) return null;
            String regex = getID();
            MutedAlertRequest r = mm.get(regex);
            if (r != null && !r.isValid()) {
                mm.remove(regex);
                r = null;
            }
            return r;
        }

        @Override
        boolean isMuted() {
            Map<String,MutedAlertRequest> mm = muteRequests.get(getAgentName());
            if (mm == null) return false;
            String id = getID();
            for (MutedAlertRequest r : mm.values()) {
                if (r.matches(id)) return true;
            }
            return false;
        }
        
    }
    
    static private class AlertTreeRenderer extends DefaultTreeCellRenderer {

        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean sel, boolean expanded, boolean leaf, int row, boolean hasFocus) {
            super.getTreeCellRendererComponent(tree, value, sel, expanded, leaf, row, hasFocus);
            try {
                AlertNode node = (AlertNode) value;
                AlertState state = node.getAlertLevel();
                if (!sel) {
                    setForeground(COLOR.get(state));
                }
                if (leaf) {
                    setToolTipText(((IDNode)node).getHistory().getLatestAlert().getDescription());
                } else {
                    setToolTipText(null);
                }
            } catch (ClassCastException | NullPointerException x) {
            }
            try {
                setIcon(((AlertNode)value).getIcon());
            } catch (RuntimeException x) {
                setIcon(null);
            }
            setHorizontalTextPosition(LEADING);
            return this;
        }
        
    }
    
    private class SettingsPanel extends JPanel {
        
        private final JCheckBox historyBox;
        private final JCheckBox muteBox;
        private final JCheckBox toFrontBox;
        private final JCheckBox selectNewBox;
        private final JCheckBox ignoreMutedBox;
        private final JCheckBox ignoreAckBox;
        
        SettingsPanel() {
            setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
            Box root = Box.createVerticalBox();
            add(root);
 
            historyBox = new JCheckBox("Show alert history in the information panel.");
            historyBox.setSelected(historyProp);
            historyBox.setToolTipText("Display recent history of raised alerts for the selected ID, in addition to the current status.");
            historyBox.addActionListener(e -> {
                historyProp = historyBox.isSelected();
                AlertNode node = tree.getSelectedNode();
                if (node != null) {
                    node.fillInfoPanel(AlertViewer.this);
                }
            });
            root.add(historyBox);
            root.add(Box.createRigidArea(VDIM));
 
            muteBox = new JCheckBox("Sound off");
            muteBox.setSelected(soundProp);
            muteBox.setToolTipText("Disable all audio alerts.");
            root.add(muteBox);
            root.add(Box.createRigidArea(VDIM2));
            
            Box newAlertBox = Box.createVerticalBox();
            newAlertBox.setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createEtchedBorder(), 
                    BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
            root.add(newAlertBox);
            
            newAlertBox.add(new JLabel("<html>What happens when a new alert is received from one of the subsystems?"));
            newAlertBox.add(Box.createRigidArea(new Dimension(HSPACE, VSPACE)));

            toFrontBox = new JCheckBox("Bring the alert viewer to front.");
            toFrontBox.setSelected(toFrontProp);
            toFrontBox.setToolTipText("Display the alert viewer and bring it to front whenever a new alert is received.");
            toFrontBox.addActionListener(e -> refresh());
            newAlertBox.add(toFrontBox);
            newAlertBox.add(Box.createRigidArea(VDIM));

            selectNewBox = new JCheckBox("Select and display the new alert.");
            selectNewBox.setSelected(selectNewProp);
            selectNewBox.setToolTipText("Automatically select and display any newly raised alert.");
            selectNewBox.addActionListener(e -> refresh());
            newAlertBox.add(selectNewBox);
            newAlertBox.add(Box.createRigidArea(VDIM));

            ignoreMutedBox = new JCheckBox("Ignore muted alerts.");
            ignoreMutedBox.setSelected(ignoreMutedProp);
            ignoreMutedBox.setToolTipText("Do not display the viewer or switch to the new alert if it has been muted by MMM.");
            newAlertBox.add(ignoreMutedBox);
            newAlertBox.add(Box.createRigidArea(VDIM));

            ignoreAckBox = new JCheckBox("Ignore acknowledged alerts.");
            ignoreAckBox.setSelected(ignoreAckProp);
            ignoreAckBox.setToolTipText("Do not display the viewer or switch to the new alert if it has been previously acknowledged, and will remain acknowledged.");
            newAlertBox.add(ignoreAckBox);
            
            root.add(Box.createVerticalGlue());
            refresh();
        }
        
        void save() {
            plugin.getServices().setProperty(LsstAlertPlugin.OPT_HISTORY, historyProp = historyBox.isSelected());
            plugin.getServices().setProperty(LsstAlertPlugin.OPT_SOUND, soundProp = muteBox.isSelected());
            plugin.getServices().setProperty(LsstAlertPlugin.OPT_SELECT, selectNewProp = selectNewBox.isSelected());
            plugin.getServices().setProperty(LsstAlertPlugin.OPT_TOFRONT, toFrontProp = toFrontBox.isSelected());
            plugin.getServices().setProperty(LsstAlertPlugin.OPT_IGNOREMUTED, ignoreMutedProp = ignoreMutedBox.isSelected());
            plugin.getServices().setProperty(LsstAlertPlugin.OPT_IGNOREACK, ignoreAckProp = ignoreAckBox.isSelected());
        }
        
        private void refresh() {
            boolean attention = toFrontBox.isSelected() ||  selectNewBox.isSelected();
            ignoreMutedBox.setEnabled(attention);
            ignoreAckBox.setEnabled(attention);
        }
    }
    
    
// -- Static utilities : -------------------------------------------------------
    
    /**
     * Returns the name of MMM subsystem.
     * @return MMM name if present on the buses, {@code null} otherwise.
     */
    static private String getMmmName() {
        String mmm = null;
        AgentMessagingLayer aml = Console.getConsole().getMessagingAccess();
        if (aml != null) {
            for (AgentInfo ai : aml.getAgentPresenceManager().listConnectedAgents()) {
                if (ai.getType() == AgentType.MMM) {
                    mmm = ai.getName();
                    break;
                }
            }
        }
        return mmm;
    }
    
    /**
     * Sends raw command to MMM on swing worker thread and returns immediately.
     * Should be called on EDT.
     * 
     * @param command Command name.
     * @param args Command arguments as objects that should be passed directly to the command implementation.
     */
    static void sendCommandToMMM(String command, Object... args) {
        new SwingWorker<CommandTask, Void>() {
            @Override
            protected CommandTask doInBackground() throws Exception {
                String mmmName = getMmmName();
                if (mmmName == null) {
                    return null;
                } else {
                    CommandService serv = CommandService.getService();
                    if (args.length == 0) {
                        return serv.sendRaw(mmmName, command);
                    } else {
                        Object[] aa = Arrays.copyOf(args, args.length + 1);
                        aa[args.length] = new Options().withOption("w");
                        return serv.sendRaw(mmmName, command, aa);
                    }
                    
                }
            }
        }.execute();
    }
    
    
// -- Saving session : ---------------------------------------------------------
    
    static public class Descriptor extends ComponentDescriptor {
        
        private DataPanelDescriptor page;

        public DataPanelDescriptor getPage() {
            return page;
        }

        public void setPage(DataPanelDescriptor page) {
            this.page = page;
        }
      
    }
    
    Descriptor save() {
        if (rootPanel != null) {
            config.setPage(DataPanelDescriptor.get(rootPanel));
        }
        return config;
    }
    
}
