package org.lsst.ccs.gconsole.base;

import java.awt.AWTEvent;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.AWTEventListener;
import java.awt.event.ActionListener;
import java.util.EnumSet;
import java.util.concurrent.TimeUnit;
import static java.util.concurrent.TimeUnit.DAYS;
import static java.util.concurrent.TimeUnit.HOURS;
import static java.util.concurrent.TimeUnit.MINUTES;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import org.freehep.jas.services.PreferencesTopic;
import org.lsst.ccs.gconsole.util.ThreadUtil;
import org.lsst.ccs.gconsole.util.swing.BoxedComboBox;
import org.lsst.ccs.gconsole.util.swing.BoxedSpinner;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;

/**
 * Auxiliary class that performs actions on idle consoles.
 *
 * @author onoprien
 */
class SelfDestructor {
    
    private enum SafeAct {Detach, Block}
    private enum Unit {days, hours, minutes}

// -- Fields : -----------------------------------------------------------------
    
    static private final String PROP_BEFORE = "org.lsst.ccs.gui.self-destruct.before"; // seconds idle before shutdown warning is displayed
    static private final String PROP_AFTER = "org.lsst.ccs.gui.self-destruct.after";  // seconds after warning to shutdown
    static private final String PROP_SAFE = "org.lsst.ccs.gui.safe"; // "(Detach|Block) N", action/seconds idle
    static private final int DEFAULT_BEFORE = 2 * 3600;
    static private final int DEFAULT_AFTER = 2 * 3600;
    static private final String DEFAULT_SAFE = "Block 1800";
    
    static private int beforeShutdown, afterShutdown; // seconds
    static private EnumSet<SafeAct> safeAct;
    static private int beforeSafe; // seconds
    static private boolean inSafe;

    static private long lastTouch; // time of last user action
    static private PeriodicTask task; // once a minute task that checks time since last user action
    static private JLabel timeLeftLabel; // non-null when any countdown is in progress
    
    static private final AWTEventListener inputListener = e -> {
        lastTouch = System.currentTimeMillis();
    };
    
    static private boolean initialized;
        

// -- Life cycle : -------------------------------------------------------------
    
    /** Restores everything to "as if just started" state. */
    static void restart() {
        ThreadUtil.invokeLater(() -> {
            
            Console c = Console.getConsole();
            if (!initialized) {
                initialized = true;
                c.addProperty(PROP_BEFORE, DEFAULT_BEFORE);
                c.addProperty(PROP_AFTER, DEFAULT_AFTER);
                c.addProperty(PROP_SAFE, DEFAULT_SAFE);
                c.getConsoleLookup().add(new Pref());
            }
            
            // Read serttings
            
            beforeShutdown = (int) c.getProperty(PROP_BEFORE);
            afterShutdown = (int) c.getProperty(PROP_AFTER);
            String s = (String) c.getProperty(PROP_SAFE);
            try {
                String[] ss = s.split("\\s");
                int n = ss.length-1;
                safeAct = EnumSet.noneOf(SafeAct.class);
                beforeSafe = Integer.parseInt(ss[n]);
                for (int i=0; i<n; i++) {
                    safeAct.add(SafeAct.valueOf(ss[i]));
                }
            } catch (ArrayIndexOutOfBoundsException | IllegalArgumentException x) {
                safeAct = EnumSet.of(SafeAct.Block);
                beforeSafe = 1800;
            }
            
            // Restart task

            Toolkit.getDefaultToolkit().removeAWTEventListener(inputListener);
            if (task != null) {
                task.cancel(true);
                task = null;
            }
            timeLeftLabel = null;
            if (beforeShutdown > 0 || beforeSafe > 0) {
                Toolkit.getDefaultToolkit().addAWTEventListener(inputListener, AWTEvent.KEY_EVENT_MASK + AWTEvent.MOUSE_EVENT_MASK /* + AWTEvent.MOUSE_MOTION_EVENT_MASK*/);
                lastTouch = System.currentTimeMillis();
                task = Console.getConsole().getScheduler().scheduleAtFixedRate(SelfDestructor::check, 1L, 1L, TimeUnit.MINUTES);
            }
        });
    }
    

// -- Local methods : ----------------------------------------------------------
    
    /** Checks time since last user activity and triggers actions. */
    static private void check() {
        try {
            SwingUtilities.invokeLater(() -> {
                Console c = Console.getConsole();
                if (timeLeftLabel != null) return; // already counting down
                long sinceLastTouch = System.currentTimeMillis() - lastTouch;
                if (beforeShutdown > 0 && sinceLastTouch > (beforeShutdown * 1000L)) {
                    try {
                        startShutdown();
                    } catch (Throwable x) {
                        emergencyShutdown(x);
                    }
                } else if (!inSafe && beforeSafe > 0 && sinceLastTouch > (beforeSafe * 1000L)) {
                    startSafeMode();
                }
            });
        } catch (Throwable x) {
            emergencyShutdown(x);
        }
    }
    
    /** Initiates shutdown countdown, and then kills the console. */
    static private void startShutdown() {
        if (task != null) {
            task.cancel(true);
            task = null;
        }
        final String killLabel = "Shut down";
        final String cancelLabel = "Keep alive";
        String s = countdown(afterShutdown, "Idle CCS console", "<center>This console will be shut down due to inactivity in", cancelLabel, killLabel, cancelLabel);
        if (s == null) s = killLabel;
        switch (s) {
            case killLabel:
                if (task != null) task.cancel(true);
                try {
                    Console.getConsole().shutdownAgent();
                } catch (Exception x) {}
                break;
            default:
                restart();
                break;
        }
    }
    
    /** Kills the console immediately no matter what. */
    static public void emergencyShutdown(Throwable x) {
        try {
            Runnable r;
            if (x == null) {
                Thread.dumpStack();
                r = () -> Console.getConsole().getLogger().error("Emergency shutdown");
            } else {
                r = () -> Console.getConsole().getLogger().error("Emergency shutdown", x);
            }
            Thread t = new Thread(r, "Emergency shutdown logger");
            t.setDaemon(true);
            t.start();
            t.join(2000);
        } catch (Throwable xx) {
        } finally {
            System.exit(1);
        }
    }
    
    static private void startSafeMode() {
        if (inSafe) return;
        StringBuilder sb = new StringBuilder();
        sb.append("This console has been idle for a while. To prevent accidental actions, it will enter safe mode in ");
        final String keepLabel = "Cancel";
        String s = countdown(60, "Entering safe mode", sb.toString(), keepLabel, keepLabel);
        if (s == null) {
            InputBlocker.setBlockAll(true);
            inSafe = true;
        }
    }
    
    static void setSafe(boolean safe) {
        inSafe = safe;
    }
    
    /**
     * Displays countdown dialog, and returns its outcome.
     * 
     * @return Selected option.
     *         If the dialog is closed, the default option is returned.
     *         If the countdown ends without any user selection, {@code null} is returned.
     */
    static private String countdown(int seconds, String title, String message, String defaultOption, String... options) {
        if (seconds <= 0) return null;
        if (options.length == 0) {
            options = new String[] {defaultOption};
        }
        long start = System.currentTimeMillis();
        timeLeftLabel = new JLabel();
        JOptionPane pane = new JOptionPane(timeLeftLabel, JOptionPane.INFORMATION_MESSAGE, JOptionPane.DEFAULT_OPTION, null, options, defaultOption);
        JDialog dialog = pane.createDialog(Console.getConsole().getWindow(), title);
        dialog.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
        ActionListener act = e -> {
            if (timeLeftLabel == null) return;
            int remainSeconds = seconds - (int) ((System.currentTimeMillis() - start)/1000L);
            if (remainSeconds < 0) {
                dialog.dispose();
                pane.setValue("");
            }
            StringBuilder sb = new StringBuilder("<html>");
            sb.append(message).append("<p>");
            int hours = remainSeconds / 3600;
            if (hours > 0) sb.append(hours).append(" hours ");
            remainSeconds %= 3600;
            int minutes = remainSeconds / 60;
            if (minutes > 0) sb.append(minutes).append(" minutes ");
            remainSeconds %= 60;
            sb.append(remainSeconds).append(" seconds.");
            timeLeftLabel.setText(sb.toString());
        };
        act.actionPerformed(null);
        Timer timer = new Timer(1000, act);
        timer.start();
        dialog.pack();
        dialog.setVisible(true);
        timer.stop();
        timeLeftLabel = null;
        String out = (String) pane.getValue();
        if (out == null) {
            return defaultOption;
        } else if (out.equals("")) {
            return null;
        } else {
            return out;
        }
    }
    
    
// -- Preferences : ------------------------------------------------------------
    
    static private class Pref implements PreferencesTopic {

        @Override
        public String[] path() {
            return new String[] {"LSST"};
        }

        @Override
        public JComponent component() {
            PrefGUI panel = new PrefGUI();
            panel.set();
            return panel;
        }

        @Override
        public boolean apply(JComponent jc) {
            try {
                PrefGUI panel = (PrefGUI) jc;
                panel.get();
                restart();
                return true;
            } catch (RuntimeException x) {
                return false;
            }
        }
        
    }
    
    static private final class PrefGUI extends JPanel {
        
        private final JCheckBox killBox, warnBox;
        private final JSpinner killSpinner, warnSpinner;
        private final JComboBox<TimeUnit> killCombo, warnCombo;
        private final JLabel warnLabel;
        
        private final JCheckBox safeBox;
        private final JComboBox<TimeUnit> safeTimeCombo;
        private final JLabel safeLabel;
        private final JSpinner safeSpinner;
        
        PrefGUI() {
            
            this.setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            setBorder(BorderFactory.createTitledBorder("Idle console shutdown"));
            
            Box line = Box.createHorizontalBox();
            add(line);
            line.add(Box.createRigidArea(Const.HDIM));
            killBox = new JCheckBox("Shut down if idle for ");
            line.add(killBox);
            killBox.addActionListener(e -> enableComponents());
            killSpinner = new BoxedSpinner(new SpinnerNumberModel(1, 1, 100, 1));
            line.add(killSpinner);
            line.add(Box.createRigidArea(Const.HDIM));
            killCombo = new BoxedComboBox<>(new TimeUnit[] {DAYS, HOURS, MINUTES});
            line.add(killCombo);
            line.add(Box.createRigidArea(Const.HDIM));
            line.add(Box.createHorizontalGlue());
            
            line = Box.createHorizontalBox();
            add(line);
            line.add(Box.createRigidArea(new Dimension(3*Const.HSPACE, 0)));
            warnBox = new JCheckBox("Warn ");
            line.add(warnBox);
            warnSpinner = new BoxedSpinner(new SpinnerNumberModel(1, 1, 100, 1));
            warnCombo = new BoxedComboBox<>(new TimeUnit[] {DAYS, HOURS, MINUTES});
            warnBox.addActionListener(e -> {
                boolean warningEnabled = warnBox.isSelected();
                warnSpinner.setEnabled(warningEnabled);
                warnCombo.setEnabled(warningEnabled);
            });
            line.add(warnSpinner);
            line.add(Box.createRigidArea(Const.HDIM));
            line.add(warnCombo);
            warnLabel = new JLabel(" before shutdown.");
            line.add(warnLabel);
            line.add(Box.createRigidArea(Const.HDIM));
            line.add(Box.createHorizontalGlue());
            
            add(Box.createRigidArea(new Dimension(0, 3*Const.VSPACE)));

            line = Box.createHorizontalBox();
            add(line);
            line.add(Box.createRigidArea(Const.HDIM));
            safeBox = new JCheckBox("Enter safe mode");
            line.add(safeBox);
            safeBox.addActionListener(e -> enableComponents());
            line.add(Box.createRigidArea(Const.HDIM));
            line.add(Box.createHorizontalGlue());

            line = Box.createHorizontalBox();
            add(line);
            line.add(Box.createRigidArea(new Dimension(3*Const.HSPACE, 0)));
            safeLabel = new JLabel(" if idle for ");
            line.add(safeLabel);
            safeSpinner = new BoxedSpinner(new SpinnerNumberModel(1, 1, 100, 1));
            line.add(safeSpinner);
            line.add(Box.createRigidArea(Const.HDIM));
            safeTimeCombo = new BoxedComboBox<>(new TimeUnit[] {DAYS, HOURS, MINUTES});
            line.add(safeTimeCombo);
            line.add(Box.createRigidArea(Const.HDIM));
            line.add(Box.createHorizontalGlue());
            
            add(Box.createVerticalGlue());
        }
        
        void set() {
            set(beforeShutdown, killBox, killSpinner, killCombo);
            set(afterShutdown, warnBox, warnSpinner, warnCombo);
            set(beforeSafe, safeBox, safeSpinner, safeTimeCombo);
            enableComponents();
        }
        
        void get() {
            Console c = Console.getConsole();
            c.setProperty(PROP_BEFORE, get(killBox, killSpinner, killCombo));
            c.setProperty(PROP_AFTER, get(warnBox, warnSpinner, warnCombo));
            c.setProperty(PROP_SAFE, Integer.toString(get(safeBox, safeSpinner, safeTimeCombo)));
        }
        
        private void enableComponents() {
            if (killBox.isSelected()) {
                killSpinner.setEnabled(true);
                killCombo.setEnabled(true);
                warnBox.setEnabled(true);
                boolean warningEnabled = warnBox.isSelected();
                warnSpinner.setEnabled(warningEnabled);
                warnCombo.setEnabled(warningEnabled);
                warnLabel.setEnabled(true);
            } else {
                killSpinner.setEnabled(false);
                killCombo.setEnabled(false);
                warnBox.setEnabled(false);
                warnSpinner.setEnabled(false);
                warnCombo.setEnabled(false);
                warnLabel.setEnabled(false);
            }
            boolean enable = safeBox.isSelected();
            safeLabel.setEnabled(enable);
            safeTimeCombo.setEnabled(enable);
            safeSpinner.setEnabled(enable);
        }
        
        private void set(int seconds, JCheckBox box, JSpinner spinner, JComboBox combo) {
            boolean enabled = seconds > 0;
            box.setSelected(enabled);
            seconds = Math.abs(seconds);
            seconds /= 60;
            if (seconds % 60 == 0) {
                seconds /= 60;
                if (seconds % 24 == 0) {
                    seconds /= 24;
                    combo.setSelectedItem(TimeUnit.DAYS);
                } else {
                    combo.setSelectedItem(TimeUnit.HOURS);
                }
            } else {
                combo.setSelectedItem(TimeUnit.MINUTES);
            }
            spinner.setValue(seconds);
        }
        
        private int get(JCheckBox box, JSpinner spinner, JComboBox<TimeUnit> combo) {
            int seconds = (int) spinner.getValue();
            TimeUnit unit = (TimeUnit) combo.getSelectedItem();
            seconds = (int) unit.toSeconds(seconds);
            if (!box.isSelected()) seconds = -seconds;
            return seconds;
        }
        
    }
    
}
