package org.lsst.ccs.gconsole.plugins.trending.dataselection;

import java.awt.Color;
import static java.awt.Component.LEFT_ALIGNMENT;
import java.awt.Dimension;
import java.awt.Window;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import static org.lsst.ccs.gconsole.base.Const.HSPACE;
import static org.lsst.ccs.gconsole.base.Const.VSPACE;
import org.lsst.ccs.gconsole.util.swing.Utilities;

/**
 * GUI panel for entering a custom data type.
 *
 * @author onoprien
 */
public final class DataTypeDialog extends JPanel {

// -- Fields : -----------------------------------------------------------------
    
    private final JPanel panel;
    private final JLabel label;
    private final JRadioButton useRawRadio, useStatRadio;
    private final JTextField nBinsField, nameField;
    private final JCheckBox nBinsCheck, saveCheck;
    
    private DataType outType;
    private final JButton okButton;
    
    private boolean defaultName = true; // FIXME: custom name protection


// -- Life cycle : -------------------------------------------------------------
    
    private DataTypeDialog(DataType seed) {
        
        if (seed == null) seed = DataType.DEFAULT;
        
        okButton = new JButton("  OK  ");
        setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
        
        // Settings
        
        panel = new JPanel();    
        panel.setLayout(new BoxLayout(panel, BoxLayout.Y_AXIS));
        panel.setAlignmentX(LEFT_ALIGNMENT);
        panel.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createTitledBorder(" DATA "),
                BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));


        Box row = Box.createHorizontalBox();
        row.setAlignmentX(LEFT_ALIGNMENT);
        row.add(new JLabel("Data source: "));
        ButtonGroup group = new ButtonGroup();
        useRawRadio = new JRadioButton("Raw");
        group.add(useRawRadio);
        row.add(useRawRadio);
        useStatRadio = new JRadioButton("Statistical");
        group.add(useStatRadio);
        row.add(useStatRadio);
        row.add(Box.createHorizontalGlue());
        panel.add(row);

        row = Box.createHorizontalBox();
        row.setAlignmentX(LEFT_ALIGNMENT);
        nBinsCheck = new JCheckBox("Plot approximately");
        nBinsField = new JTextField(3);

        row.add(nBinsCheck);
        nBinsField.setMaximumSize(new Dimension(nBinsField.getPreferredSize().width, nBinsField.getPreferredSize().height));
        nBinsField.setInputVerifier(Utilities.BLACK_VERIFIER);
        row.add(nBinsField);
        label = new JLabel(" data points.");
        row.add(label);
        row.add(Box.createHorizontalGlue());
        panel.add(row);
        
        panel.add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        add(panel);
       
        add(Box.createRigidArea(new Dimension(0, 2*VSPACE)));
        
        // Remember As panel
       
        Box saveBox = Box.createVerticalBox();
        saveBox.setAlignmentX(LEFT_ALIGNMENT);
        saveBox.setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createTitledBorder(" Remember As "), 
                    BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
        nameField = new JTextField();
        nameField.setColumns(30);
        saveBox.add(nameField);
        saveBox.add(Box.createRigidArea(new Dimension(0, VSPACE)));
        saveCheck = new JCheckBox("Remember between sessions");
        saveBox.add(saveCheck);
        add(saveBox);
        
        // Seed
        
        if (seed.getRaw()) {
            useRawRadio.setSelected(true);
            nBinsCheck.setEnabled(false);
            nBinsField.setEnabled(false);
            nBinsField.setText("");
            label.setEnabled(false);
        } else {
            useStatRadio.setSelected(true);
            nBinsCheck.setEnabled(true);
            label.setEnabled(true);
            int n = seed.getBins();
            if (n == 0) {
                nBinsCheck.setSelected(false);
                nBinsField.setEnabled(false);
                nBinsField.setText("");
            } else {
                nBinsCheck.setSelected(true);
                nBinsField.setEnabled(true);
                nBinsField.setText(Integer.toString(n));
            }
            saveCheck.setSelected(seed.isPersistent());
            adjustName();
            nameField.setText(seed.getName());
        }

        // Wire for events
        
        nBinsField.addCaretListener(e -> {
            String text = nBinsField.getText().trim();
            boolean valid = true;
            if (!text.isEmpty()) {
                try {
                    int value = Integer.parseInt(text);
                    valid = value > 0;
                } catch (NumberFormatException x) {
                    valid = false;
                }
            }
            nBinsField.setForeground(valid ? Color.BLACK : Color.RED);
            adjustName();
        });
        useRawRadio.addActionListener(e -> {
            nBinsCheck.setEnabled(false);
            nBinsCheck.setSelected(false);
            nBinsField.setEnabled(false);
            nBinsField.setText("");
            label.setEnabled(false);
            nameField.setEnabled(false);
            saveCheck.setEnabled(false);
            adjustName();
        });
        useStatRadio.addActionListener(e -> {
            nBinsCheck.setEnabled(true);
            nBinsField.setEnabled(false);
            nBinsField.setText("30");
            label.setEnabled(true);
            adjustName();
        });
        nBinsCheck.addActionListener(e -> {
            if (nBinsCheck.isSelected()) {
                nBinsField.setEnabled(true);
                nBinsField.grabFocus();
            } else {
                nBinsField.setEnabled(false);
            }
            adjustName();
        });
        
    }
    
    /**
     * Opens trending time window creation/editing dialog.
     * 
     * @param parent Parent component.
     * @param seed Instance to be used to set initial values in the dialog, or {@code null} if default initial values should be used.
     * @return New {@code DataType} instance, or {@code null} if the dialog is canceled.
     */
    static public DataType defineDataType(JComponent parent, DataType seed) {
        final DataTypeDialog tsp = new DataTypeDialog(seed);
        tsp.okButton.addActionListener(e -> {
            tsp.outType = tsp.getType();
            Window w = SwingUtilities.getWindowAncestor(tsp);
            if (w != null) w.setVisible(false);
        });
        JButton cancelButton = new JButton("Cancel");
        cancelButton.addActionListener(e -> {
            Window w = SwingUtilities.getWindowAncestor(tsp);
            if (w != null) w.setVisible(false);
        });
        JOptionPane.showOptionDialog(parent, tsp, "Select data type", JOptionPane.OK_CANCEL_OPTION, JOptionPane.PLAIN_MESSAGE, null, new Object[] {tsp.okButton, cancelButton}, null);
        return tsp.outType;
    }

    
// -- Local methods : ----------------------------------------------------------
    
    private DataType getType() {
        String name = nameField.getText().trim();
        if (name.isEmpty()) name = null;
        boolean persistent = saveCheck.isSelected();
        boolean raw = useRawRadio.isSelected();
        if (raw) {
            return DataType.RAW;
        } else if (nBinsCheck.isSelected()) {
                try {
                    int bins = Integer.parseInt(nBinsField.getText());
                    if (bins < 1) throw new NumberFormatException();
                    return new DataType(name, raw, bins, persistent);
                } catch (NumberFormatException x) {
                    return null;
                }
        } else {
            return DataType.DEFAULT;
        }
    }
    
    private void adjustName() {
        DataType type = getType();
        if (type == null) {
            nameField.setText("");
            okButton.setEnabled(false);
        } else {
            if (type == DataType.RAW || type == DataType.DEFAULT) {
                nameField.setEnabled(false);
                saveCheck.setEnabled(false);
                saveCheck.setSelected(false);
            } else {
                type.setName(null);
                nameField.setEnabled(true);
                saveCheck.setEnabled(true);
            }
            nameField.setText(type.getName());
            okButton.setEnabled(true);
        }
    }

    
// -- Testing : ----------------------------------------------------------------
    
    static public void main(String... args) {
        
        SwingUtilities.invokeLater(() -> {

            DataType tw = null;
            while (true) {
                tw = DataTypeDialog.defineDataType(null, tw);
                System.out.println(tw);
                if (tw == null) {
                    break;
                }
            }
        });

                
    }
    
}
