package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.Color;
import java.awt.Dialog;
import java.awt.event.ActionEvent;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.lang.reflect.Type;
import java.time.Duration;
import java.util.*;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.states.ConfigurationState;
import org.lsst.ccs.command.Options;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.gconsole.agent.command.CommandHandle;
import org.lsst.ccs.gconsole.agent.command.CommandSender;
import org.lsst.ccs.gconsole.agent.command.CommandTask;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusEvent;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.gconsole.util.swing.BoxedComboBox;
import org.lsst.ccs.gconsole.util.swing.BoxedFormattedTextField;
import org.lsst.ccs.gconsole.util.swing.BoxedTextField;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentLoginService;
import org.lsst.ccs.services.AgentLoginService.AgentLoginUpdateListener;
import org.lsst.ccs.utilities.conv.InputConversionEngine;
import org.lsst.ccs.utilities.conv.TypeConversionException;

/**
 *
 * @author onoprien
 */
class UserManager {
    
// -- Fields : -----------------------------------------------------------------
    
    private Action menuAction;
    private CommandSender sender;
    public Map<String,Map<String,String>> config; // public to enable type reflection
    
    
// -- Life cycle : -------------------------------------------------------------
    
    void initialize(LsstCommandBrowserPlugin plugin) {
        menuAction = new AbstractAction("Manage users...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                sender = CommandService.getService().getSender();
                CommandHandle handle = new CommandHandle() {
                    @Override
                    public void onSuccess(Object result, CommandTask source) {
                        try {
                            Type type = UserManager.class.getField("config").getGenericType();
                            config = (Map<String,Map<String,String>>) InputConversionEngine.convertArgToType((String) result, type);
                            
                            // FIXME: TRANSIT : remove duplicates
                            Set<String> groups = new HashSet<>();
                            config.keySet().forEach(key -> {
                                if (key.startsWith("@")) groups.add(key.substring(1));
                            });
                            groups.forEach(key -> config.remove(key));
                            // FIXME: TRANSIT
                            
                            GUI gui = new GUI();
                            gui.pack();
                            gui.setLocationRelativeTo(gui.getOwner());
                            gui.setVisible(true);
                        } catch (ClassCastException | NoSuchFieldException | TypeConversionException x) {
                            Console.getConsole().error("Unable to fetch current settings from the lock manager", x);
                        }
                    }
                };
                sender.send(handle, Duration.ofSeconds(3), "lockmanager getConfigurationParameterValue / maxLevelMap");
            }
        };
        AgentLoginService loginServ = Console.getConsole().getAgentService(AgentLoginService.class);
        loginServ.addAgentLoginUpdateListener(new AgentLoginUpdateListener() {
            @Override
            public void onAgentLoginUpdate(String oldUserId, String newUserId) {
                SwingUtilities.invokeLater(() -> {
                    menuAction.setEnabled(Console.getConsole().getAgentService(AgentLockService.class).getMaxLevel(newUserId, AgentLockService.LOCK_MANAGER_SUBSYSTEM_NAME) >= Command.ENGINEERING_ROUTINE);
                });
            }
        });
        menuAction.setEnabled(Console.getConsole().getAgentService(AgentLockService.class).getMaxLevel(loginServ.getUserId(), AgentLockService.LOCK_MANAGER_SUBSYSTEM_NAME) >= Command.ENGINEERING_ROUTINE);
        plugin.getServices().addMenu(menuAction, "400: CCS Tools :-1:5", LsstCommandBrowserPlugin.TOOL_NAME +":10");
    }
    
    
// -- GUI : --------------------------------------------------------------------
    
    private class GUI extends JDialog {
        
        JComboBox userOrGroupBox;
        BoxedFormattedTextField nameField;
        JLabel userStatusLabel;
        JButton deleteButton;
        
        JTextField settingsField;
        
        Box tablePanel = Box.createHorizontalBox();
        List<SettingsEntry> entries = new LinkedList<>();
        
        Box groupsPanel = Box.createVerticalBox();
        List<JCheckBox> groupBoxes = new ArrayList<>();
        
        JButton saveConfigButton, resetButton, closeButton, submitButton;
        
        final AgentStatusListener statusListener;
        
        String currentUser, currentSettings;
        
        GUI() {
            
            super(Console.getConsole().getWindow(), "Manage authorized users", Dialog.ModalityType.APPLICATION_MODAL);
            setDefaultCloseOperation(DISPOSE_ON_CLOSE);
            
            entries.add(new SettingsEntry("", ""));
            
            Box root = Box.createVerticalBox();
            root.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
            add(root);
            
            Box line = Box.createHorizontalBox();
            userOrGroupBox = new BoxedComboBox(new String[] {"User", "Group"});
            userOrGroupBox.addActionListener(e -> {
                nameField.setText("");
                onUserSelection("");
            });
            line.add(userOrGroupBox);
            line.add(Box.createRigidArea(Const.HDIM));
            nameField = new BoxedFormattedTextField();
            nameField.setFilter("[-a-zA-Z_0-9]*");
            JPopupMenu popupMenu = new JPopupMenu();
            nameField.getDocument().addDocumentListener(new DocumentListener() {
                private void updatePopup() {
                    String input = nameField.getText().toLowerCase();
                    popupMenu.setVisible(false);
                    popupMenu.removeAll();
                    List<String> choices = userCandidates(input);
                    if (!choices.isEmpty()) {
                        for (String choice : choices) {
                                JMenuItem item = new JMenuItem(choice);
                                item.addActionListener(ev -> {
                                    nameField.setText(choice);
                                    popupMenu.setVisible(false);
                                });
                                popupMenu.add(item);
                        }
                        if (popupMenu.getComponentCount() > 0) {
                            popupMenu.show(nameField, 0, nameField.getHeight());
                            nameField.requestFocusInWindow();
                        }
                    }
                }
                @Override
                public void insertUpdate(DocumentEvent e) { updatePopup(); }
                @Override
                public void removeUpdate(DocumentEvent e) { updatePopup(); }
                @Override
                public void changedUpdate(DocumentEvent e) {}
            });
            nameField.addFocusListener(new FocusAdapter() {
                @Override
                public void focusLost(FocusEvent e) {
                    if (!popupMenu.isVisible()) {
                        nameField.getText();
                        String user = nameField.getText().strip();
                        if (!user.equals(currentUser)) {
                            onUserSelection(user);
                        }
                    }
                }
            });
            nameField.addActionListener(e -> onUserSelection(nameField.getText()));
            line.add(nameField);
            line.add(Box.createRigidArea(Const.HDIM));
            userStatusLabel = new JLabel("      ");
            line.add(userStatusLabel);
            line.add(Box.createRigidArea(Const.HDIM));
            deleteButton = new JButton("Delete");
            deleteButton.addActionListener(e -> {
                settingsField.setText("");
                updateFromSettings();
                submit();
            });
            line.add(deleteButton);
            line.add(Box.createHorizontalGlue());
            root.add(line);
            
            settingsField = new BoxedTextField(80);
            settingsField.addActionListener(e -> updateFromSettings());
            settingsField.addFocusListener(new FocusAdapter() {
                @Override
                public void focusLost(FocusEvent e) {
                    String settings = settingsField.getText().strip();
                    if (!settings.equals(currentSettings)) {
                        updateFromSettings();
                    }
                }
            });
            root.add(settingsField);
            
            line = Box.createHorizontalBox();
            populateTablePanel();
            tablePanel.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
            line.add(new JScrollPane(tablePanel));
            line.add(Box.createRigidArea(Const.HDIM));
            updateGroupsPanel();
            groupsPanel.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
            line.add(new JScrollPane(groupsPanel));
            root.add(line);
            
            line = Box.createHorizontalBox();
            saveConfigButton = new JButton("Save Configuration");
            saveConfigButton.setToolTipText("Tell lock manager to permanently save current configuration");
            saveConfigButton.setEnabled(Console.getConsole().getStatusAggregator().getAgentState(AgentLockService.LOCK_MANAGER_SUBSYSTEM_NAME).isInState(ConfigurationState.DIRTY));
            saveConfigButton.addActionListener(e -> CommandService.getService().send(AgentLockService.LOCK_MANAGER_SUBSYSTEM_NAME + " saveCategories -w General"));
            line.add(saveConfigButton);
            line.add(Box.createHorizontalGlue());
            line.add(Box.createRigidArea(Const.HDIM));
            resetButton = new JButton("Reset");
            resetButton.setToolTipText("Discard unsubmitted changes for the current user.");
            resetButton.addActionListener(e -> onUserSelection(nameField.getText()));
            line.add(resetButton);
            line.add(Box.createRigidArea(Const.HDIM));
            closeButton = new JButton("Close");
            closeButton.setToolTipText("Close user manager");
            closeButton.addActionListener(e -> dispose());
            line.add(closeButton);
            line.add(Box.createRigidArea(Const.HDIM));
            submitButton = new JButton("Submit");
            submitButton.setToolTipText("Submit changes for the current user to lock manager");
            submitButton.addActionListener(e -> submit());
            line.add(submitButton);
            root.add(line);
            
            String statePath = AgentLockService.LOCK_MANAGER_SUBSYSTEM_NAME + AgentChannel.MARK_STATE + "ConfigurationState";
            String configPath = AgentLockService.LOCK_MANAGER_SUBSYSTEM_NAME + AgentChannel.MARK_CONFIG + "maxLevelMap";
            statusListener = new AgentStatusListener() {
                @Override
                public void statusChanged(AgentStatusEvent e) {
                    if (e.getAgentNames().contains(AgentLockService.LOCK_MANAGER_SUBSYSTEM_NAME)) {
                        for (AgentChannel ch : e.getStatusChanges().keySet()) {
                            String p = ch.getPath();
                            if (p.equals(statePath)) {
                                SwingUtilities.invokeLater(() -> saveConfigButton.setEnabled(ConfigurationState.DIRTY.name().equals(ch.get())));
                            } else if (p.equals(configPath)) {
                                ConfigurationParameterInfo par = ch.get();
                                try {
                                    Map<String, Map<String, String>> updated = (Map<String, Map<String, String>>) par.getCurrentValueObject();
                                    SwingUtilities.invokeLater(() -> {
                                        config = updated;
                                        updateGroupsPanel();
                                        String user = nameField.getText().strip();
                                        onUserSelection(user);
                                    });
                                } catch (RuntimeException x) {
                                    Console.getConsole().getLogger().warn("Error updating User Manager", x);
                                }
                            }
                        }
                    }
                }
            };
            Console.getConsole().getStatusAggregator().addListener(statusListener, Collections.singletonList(AgentLockService.LOCK_MANAGER_SUBSYSTEM_NAME), List.of(statePath, configPath));
        }

        @Override
        public void dispose() {
            Console.getConsole().getStatusAggregator().removeListener(statusListener);
            super.dispose();
        }
        
        private void updateGroupsPanel() {
            SortedSet<String> groups = new TreeSet<>();
            if (config != null) {
                for (String k : config.keySet()) {
                    if (k.startsWith("@")) groups.add(k.substring(1));
                }
            }
            groupBoxes.clear();
            groups.forEach(g -> {
                JCheckBox cb = new JCheckBox(g);
                cb.addActionListener(e -> updateToSettings());
                groupBoxes.add(cb);
            });
            groupsPanel.removeAll();
            groupsPanel.add(new JLabel("<html><b>Groups"));
            groupsPanel.add(Box.createRigidArea(Const.VDIM));
            groupBoxes.forEach(cb -> {
                groupsPanel.add(cb);
                groupsPanel.add(Box.createRigidArea(Const.VDIM));
            });
        }
        
        private void populateTablePanel() {
            tablePanel.removeAll();
            Box agentsBox = Box.createVerticalBox();
            agentsBox.add(new JLabel("<html><b>Subsystem"));
            agentsBox.add(Box.createRigidArea(Const.VDIM));
            tablePanel.add(agentsBox);
            tablePanel.add(Box.createRigidArea(Const.HDIM));
            Box levelsBox = Box.createVerticalBox();
            levelsBox.add(new JLabel("<html><b>Maximum level"));
            levelsBox.add(Box.createRigidArea(Const.VDIM));
            tablePanel.add(levelsBox);
            for (SettingsEntry e : entries) {
                agentsBox.add(e.agentField);
                levelsBox.add(e.levelField);
            }
            agentsBox.add(Box.createVerticalGlue());
            levelsBox.add(Box.createVerticalGlue());
            tablePanel.revalidate();
            tablePanel.repaint();
        }
        
        private boolean updateFromSettings() {
            String s = settingsField.getText().strip();
//            if (s.equals(currentSettings)) return true;
            currentSettings = s;
            try {
                Map<String,String> m = AgentLockService.User.levelsFromString(s);
                entries.clear();
                m.forEach((k,v) -> {
                    if (!v.isBlank()) {
                        entries.add(new SettingsEntry(k, v));
                    }
                });
                entries.add(new SettingsEntry("",""));
                populateTablePanel();
                groupBoxes.forEach(b -> b.setSelected(m.getOrDefault(b.getText(), "x").isBlank()));
                settingsField.setForeground(null);
                return true;
            } catch (IllegalArgumentException x) {
                settingsField.setForeground(Color.RED);
                return false;
            }
        }
        
        private boolean updateToSettings() {
            boolean out = true;
            int empty = 0;
            int invalid = 0;
            Map<String,String> m = new TreeMap<>();
            for (SettingsEntry e : entries) {
                if (e.isEmpty()) {
                    empty++;
                } else if (!e.isValid()) {
                    invalid++;
                    out = false;
                } else {
                    m.put(e.getAgent(), e.getLevel());
                }
            }
            if (empty == 0 && invalid == 0) {
                entries.add(new SettingsEntry("",""));
                populateTablePanel();
            } else if (empty > 1) {
                Iterator<SettingsEntry> it = entries.iterator();
                while (it.hasNext() && empty > 1) {
                    if (it.next().isEmpty()) {
                        it.remove();
                        empty--;
                    }
                }
                populateTablePanel();
            }
            groupBoxes.forEach(b -> {
                if (b.isSelected()) m.put(b.getText(), "");
            });
            if (out) {
                String s = AgentLockService.User.levelsToString(m, false);
                settingsField.setText(s);
                settingsField.setForeground(null);
            }
            return out;
        }
        
        private void onUserSelection(String user) {
//            if (user == null || user.isBlank() || user.equals(currentUser)) return;
//            if (user == null || user.isBlank()) return;
            currentUser = user;
            if (isGroup()) user = "@"+ user;
            Map<String,String> m = config.get(user);
            if (m == null) {
                userStatusLabel.setText("New");
                deleteButton.setEnabled(false);
                settingsField.setText("");
            } else {
                userStatusLabel.setText("   ");
                deleteButton.setEnabled(true);
                settingsField.setText(AgentLockService.User.levelsToString(m, false));
            }
            updateFromSettings();
        }
        
        private void submit() {
            String user = nameField.getText().strip();
            if (isGroup()) user = "@" + user;
            try {
                Map<String, String> settings = AgentLockService.User.levelsFromString(settingsField.getText());
                Map<String, String> old = config.getOrDefault(user, Collections.emptyMap());
                if (!settings.equals(old)) {
                    TreeMap<String, Map<String, String>> updated = new TreeMap<>(config);
                    if (settings.isEmpty()) {
                        updated.remove(user);
                    } else {
                        updated.put(user, settings);
                    }
                    CommandService.getService().sendRaw(AgentLockService.LOCK_MANAGER_SUBSYSTEM_NAME, "change", "/", "maxLevelMap", updated, new Options().withOption("withLock"));
                }
            } catch (IllegalArgumentException x) {
            }
        }
        
        private boolean isGroup() {
            return userOrGroupBox.getSelectedIndex() == 1;
        }
        
        private List<String> userCandidates(String input) {
            if (config == null || input.isBlank()) return Collections.emptyList();
            input = input.strip();
            int maxCandidates = 10;
            ArrayList<String> out = new ArrayList(maxCandidates);
            int n = 0;
            if (isGroup()) input = "@"+ input;
            for (String key : config.keySet()) {
                if (key.startsWith(input)) {
                    if (isGroup()) key = key.substring(1);
                    out.add(key);
                    if (++n > maxCandidates) break;
                }
            }
            return out;
        }
        
        private class SettingsEntry {
            static final Command.Level levelParser = new Command.Level().with(Command.Level.OnTooHigh.ON_TOO_HIGH_DEFAULT).withDefault(-1);
            final BoxedFormattedTextField agentField ;
            final BoxedFormattedTextField levelField;
            SettingsEntry(String key, String value) {
                agentField = new BoxedFormattedTextField();
                agentField.setFilter("[-a-zA-Z_0-9*]*");
                agentField.setText(key.strip());
                levelField = new BoxedFormattedTextField();
                levelField.setFilter("[_0-9ENGIROUTADVCDXPMLengiroutadvcdxpml]*");
                value = value.strip();
                if (!value.isEmpty()) {
                    try {
                        int i = Integer.parseInt(value);
                        if (i > Command.MAX) value = Integer.toString(Command.MAX);
                    } catch (NumberFormatException x) {
                        value = value.toUpperCase();
                    }
                }
                
                levelField.setText(value);
                agentField.addActionListener(e -> updateToSettings());
                agentField.addFocusListener(new FocusAdapter() {
                    @Override
                    public void focusLost(FocusEvent e) {
                        updateToSettings();
                    }
                    
                });
                levelField.addActionListener(e -> updateToSettings());
                levelField.addFocusListener(new FocusAdapter() {
                    @Override
                    public void focusLost(FocusEvent e) {
                        updateToSettings();
                    }
                    
                });
                levelField.addKeyListener(new KeyAdapter() {
                    @Override
                    public void keyReleased(KeyEvent e) {
                        getLevel();
                    }
                });
            }
            String getAgent() {
                return agentField.getText().strip();
            }
            String getLevel() {
                        String s = levelField.getText().strip().toUpperCase();
                        int level = levelParser.getValue(s);
                        if (level == -1) {
                            levelField.setForeground(Color.red);
                            return "";
                        } else {
                            levelField.setForeground(null);
                            return s;
                        }
            }
            boolean isValid() {
                String agent = getAgent();
                String level = getLevel();
                return !(agent.isBlank() || level.isBlank() || agent.length() < 1 || agent.length() > 80);
            }
            boolean isEmpty() {
                return agentField.getText().isBlank() && levelField.getText().isBlank();
            }
        }
        
    }
    
}
