package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.io.Serializable;
import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.gconsole.agent.command.CommandCode;
import org.lsst.ccs.gconsole.agent.command.CommandHandle;
import org.lsst.ccs.gconsole.agent.command.CommandSender;
import org.lsst.ccs.gconsole.agent.command.CommandTask;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.gconsole.services.lock.Locker;
import org.lsst.ccs.gconsole.services.lock.LockService;
import org.lsst.ccs.gconsole.services.persist.Savable;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;

/**
 * Graphics panel that contains all other panels for a specific subsystem.
 * Used by both {@link BrowserFull} and {@link BrowserOneSubsystem}.
 * Contains {@code ComponentTreePanel}, {@code CommandListPanel}, {@code ArgInputPanel},
 * "Send" button and the results panel.
 *
 * @author onoprien
 */
public final class AgentPanel extends JSplitPane implements Savable, ChangeListener {

// -- Fields : -----------------------------------------------------------------
    
    static private final String ID_KEY = "id";
    
    private Locker agent;
    
    final ComponentTree componentTree;
    private final CommandListPanel commandListPanel;
    private final ArgInputPanel argInputPanel;
    private final OutputPanel resultPane;
    
    private final CommandSender sender;
    private final JCheckBox hideLockedBox, hideSystemBox, withLockBox;
    private final JTextField timeoutField;
    private final JProgressBar progressBar;
    
    Descriptor descriptor;
    
    private CommandTask lastCommand;
    private long lastCommandStart;

// -- Life cycle : -------------------------------------------------------------
    
    public AgentPanel() {
        
        super(VERTICAL_SPLIT);
        setContinuousLayout(true);
        descriptor = new Descriptor();
        
        sender = Console.getConsole().getSingleton(CommandService.class).getSender();
        sender.setCommandHandle(new BrowserCommandHandle());
        
        // Results panel:

        JPanel topPane = new JPanel(new GridLayout(1, 3));
        setTopComponent(topPane);
        resultPane = new OutputPanel();
        setBottomComponent(new JScrollPane(resultPane));
        
        // Components tree panel:
        
        componentTree = new ComponentTree(this);
        topPane.add(new JScrollPane(componentTree));
        
        // Commands list panel:

        commandListPanel = new CommandListPanel(this);
        topPane.add(new JScrollPane(commandListPanel));
        componentTree.addTreeSelectionListener(commandListPanel);
        
        // Command description and argument input panel:

        JPanel topPaneRightPanel = new JPanel(new BorderLayout());
        topPane.add(topPaneRightPanel);
        
        argInputPanel = new ArgInputPanel(this);
        topPaneRightPanel.add(argInputPanel, BorderLayout.CENTER);
        commandListPanel.addListSelectionListener(argInputPanel);
        
        // Settings and send button panel:
        
        JPanel controlPane = new JPanel(new GridBagLayout());
        topPaneRightPanel.add(controlPane, BorderLayout.SOUTH);
        controlPane.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, 0, Const.HSPACE));
        
        hideLockedBox = new JCheckBox("Hide locked ");
        GridBagConstraints c = new GridBagConstraints();
        c.gridx = 0;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        controlPane.add(hideLockedBox, c);
        hideLockedBox.setToolTipText("Hide commands unavailable due to agent lock state");
        hideLockedBox.setSelected(descriptor.isHideLocked());
        hideLockedBox.addActionListener(e -> {
            descriptor.setHideLocked(((JCheckBox)e.getSource()).isSelected());
            componentTree.updateData(agent);
        });
        
        hideSystemBox = new JCheckBox("Hide system ");
        c.gridx = 0;
        c.gridy = 1;
        controlPane.add(hideSystemBox, c);
        hideSystemBox.setToolTipText("Hide system commands");
        hideSystemBox.setSelected(descriptor.isHideSystem());
        hideSystemBox.addActionListener(e -> {
            descriptor.setHideSystem(((JCheckBox)e.getSource()).isSelected());
            componentTree.updateData(agent);
        });
        
        withLockBox = new JCheckBox("Auto-lock");
        c.gridx = 1;
        c.gridy = 0;
        controlPane.add(withLockBox, c);
        withLockBox.setToolTipText("Automatically lock/unlock where possible");
        withLockBox.setSelected(descriptor.isWithLock());
        withLockBox.addActionListener(e -> {
            descriptor.setWithLock(((JCheckBox) e.getSource()).isSelected());
            componentTree.updateData(agent);
        });
        
        JButton sendCmdButton = new JButton(new ImageIcon(this.getClass().getResource("ic_send_black_24dp.png")));
        argInputPanel.setCommandButton(sendCmdButton);
        c = new GridBagConstraints();
        c.gridx = 2;
        c.gridy = 0;
        c.gridheight = 2;
        c.weightx = 1.0;
        c.weighty = 1.0;
        c.anchor = GridBagConstraints.EAST;
        controlPane.add(sendCmdButton, c);
        sendCmdButton.setText("Send");
        sendCmdButton.setEnabled(false);
        sendCmdButton.addActionListener(e -> sendCommand());
        
        Box timeoutPanel = Box.createHorizontalBox();
        c = new GridBagConstraints();
        c.gridx = 0;
        c.gridy = 2;
        c.gridwidth = 3;
        c.weightx = 1.;
        c.weighty = 0.;
        c.insets = new Insets(Const.VSPACE, 0, Const.VSPACE, 0);
        c.anchor = GridBagConstraints.WEST;
        c.fill = GridBagConstraints.NONE;
        controlPane.add(timeoutPanel, c);
        timeoutPanel.add(new JLabel("Timeout:"));
        timeoutPanel.add(Box.createRigidArea(Const.HDIM));
        timeoutField = new JTextField(Integer.toString(descriptor.getTimeout()), 4);
        timeoutPanel.add(timeoutField);
        timeoutField.addActionListener(e -> setTimeout());
        timeoutField.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent e) {
                setTimeout();
            }
        });
        timeoutPanel.add(new JLabel(" seconds."));
        timeoutPanel.add(Box.createHorizontalGlue());
        
        progressBar = new JProgressBar(0, descriptor.getTimeout());
        c = new GridBagConstraints();
        c.gridx = 0;
        c.gridy = 3;
        c.gridwidth = 3;
        c.weightx = 1.;
        c.weighty = 0.;
        c.anchor = GridBagConstraints.CENTER;
        c.fill = GridBagConstraints.HORIZONTAL;
        controlPane.add(progressBar, c);
    }

    
// -- Receiving external events : ----------------------------------------------
    
    void setAgent(Locker agent) {
        if (this.agent != null) {
            this.agent.removeListener(this);
        }
        this.agent = agent;
        withLockBox.setToolTipText("Automatically lock/unlock where possible");
        withLockBox.setSelected(descriptor.isWithLock());
        withLockBox.setEnabled(true);
        if (agent != null) {
            agent.addListener(this);
        }
        componentTree.updateData(agent);
    }

    /**
     * Called in response to changes in agent data.
     * @param e Event
     */
    @Override
    public void stateChanged(ChangeEvent e) {
        if (agent == null || !agent.isAdjusting()) {
            componentTree.updateData(agent);
        }
    }
    
    
// -- Utility methods : --------------------------------------------------------
    
    /**
     * Returns {@code true} if the specified command is currently unavailable for use based on lock and level settings.
     * 
     * @param command Command to check.
     * @return {@code true} If the command is locked or {@code agent} is {@code null}.
     */
    public boolean isCommandLocked(DictionaryCommand command) {
        if (agent == null) {
            return true;
        } else {
            int commandLevel = command.getLevel();
            Command.CommandType commandType = command.getType();
            if (commandLevel == 0 && commandType == Command.CommandType.QUERY) {
                return false;
            } else if (descriptor != null && descriptor.isWithLock()
                    && command.getSupportedOptions().stream().map(p -> p.getName()).anyMatch(s -> s.equals(AgentCommandDictionaryService.withLockOption.getName()))) {
                return commandLevel > agent.getMaxLevel();
            } else {
                return agent.getState() != LockService.State.ATTACHED || command.getLevel() > agent.getLevel();
            }
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------

    private void sendCommand() {
        String destination = componentTree.getSelectedPathAsString();
        DictionaryCommand cmd = commandListPanel.getSelectedValue();
        if (destination != null && cmd != null) {
            
            String arguments = argInputPanel.getValuesAsString();
            String commandName = cmd.getCommandName();
            long id = resultPane.addCommand(destination +" "+ commandName +" "+ arguments);
            
            progressBar.setMaximum(descriptor.getTimeout() * 1000);
            progressBar.setValue(0);
            progressBar.setString(null);
            
            lastCommandStart = System.currentTimeMillis();
            lastCommand = sender.send(destination, commandName, arguments);
            lastCommand.putProperty(ID_KEY, id);  // FIXME: need 
            
            Progress prog = new Progress(lastCommand);
            PeriodicTask task = new PeriodicTask(Console.getConsole().getScheduler(), prog, true, "Com browser progress bar", Level.SEVERE, 500, TimeUnit.MILLISECONDS);
            prog.setPeriodicTask(task);
            task.start(500, TimeUnit.MILLISECONDS);
        }
    }
    
    private void setTimeout() {
        try {
            int t = Integer.parseInt(timeoutField.getText());
            if (t > 0 && t < 9999) {
                descriptor.setTimeout(t);
                sender.setTimeout(Duration.ofSeconds(t));
            } else {
                timeoutField.setText(Integer.toString(descriptor.getTimeout()));
            }
        } catch (NumberFormatException x) {
            timeoutField.setText(Integer.toString(descriptor.getTimeout()));
        }
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private class BrowserCommandHandle implements CommandHandle {

        @Override
        public void onAck(CommandAck ack, CommandTask source) {
            if (source == lastCommand) {
                progressBar.setStringPainted(true);
                progressBar.setBackground(Color.GREEN);
                progressBar.setString("ACK received");
                Duration customTimeout = ack.getTimeout();
                if (customTimeout != null) {
                    long current = System.currentTimeMillis();
                    int total = (int) (current - lastCommandStart + customTimeout.toMillis());
                    progressBar.setMaximum(total);
                    int progress = (int) (current - lastCommandStart);
                    progressBar.setValue(progress);
                }
            }
            long id = (Long) source.getProperty(ID_KEY);
            resultPane.addAck(id);
        }

        @Override
        public void onResult(CommandCode code, Object result, CommandTask source) {
            resetProgressBar(source);
            long id = (Long) source.getProperty(ID_KEY);
            resultPane.addResponse(id, code, result);
        }
        
        private void resetProgressBar(CommandTask source) {
            if (source == lastCommand) {
                progressBar.setValue(0);
                progressBar.setStringPainted(false);
                progressBar.setBackground(null);
                progressBar.setString(null);
                lastCommand = null;
            }
        }
        
    }
    
    private class Progress implements Runnable {
        
        CommandTask commandTask;
        PeriodicTask periodicTask;
        
        Progress(CommandTask commandTask) {
            this.commandTask = commandTask;
        }
        
        void setPeriodicTask(PeriodicTask periodicTask) {
            this.periodicTask = periodicTask;
        }

        @Override
        public void run() {
            SwingUtilities.invokeLater(() -> {
                if (lastCommand == commandTask) {
                    int progress = (int) (System.currentTimeMillis() - lastCommandStart);
                    progressBar.setValue(progress);
                } else {
                    periodicTask.cancel(false);
                }
            });
        }
    }
    
// -- Saving/restoring : -------------------------------------------------------    
    
    @Override
    public void restore(Serializable descriptor) {
        
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
        } else if (descriptor == null) {
            this.descriptor = new Descriptor();
        }
        
        hideLockedBox.setSelected(this.descriptor.isHideLocked());
        hideSystemBox.setSelected(this.descriptor.isHideSystem());
        int timeout = this.descriptor.getTimeout();
        timeoutField.setText(Integer.toString(timeout));
        sender.setTimeout(Duration.ofSeconds(timeout));
        
        resultPane.restore(this.descriptor.getOutputPanel());
    }

    @Override
    public Descriptor save() {
        Descriptor d = descriptor.clone();
        d.setOutputPanel(resultPane.save());
        return d;
    }
    
    static public class Descriptor implements Serializable, Cloneable {

        private boolean hideLocked;
        private boolean hideSystem = true;
        private boolean withLock;
        private int timeout = 10;
        private OutputPanel.Descriptor outputPanel;

        public boolean isWithLock() {
            return withLock;
        }

        public void setWithLock(boolean withLock) {
            this.withLock = withLock;
        }

        public boolean isHideLocked() {
            return hideLocked;
        }

        public void setHideLocked(boolean hideLocked) {
            this.hideLocked = hideLocked;
        }

        public boolean isHideSystem() {
            return hideSystem;
        }

        public void setHideSystem(boolean hideSystem) {
            this.hideSystem = hideSystem;
        }

        public int getTimeout() {
            return timeout;
        }

        public void setTimeout(int timeout) {
            this.timeout = timeout;
        }

        public OutputPanel.Descriptor getOutputPanel() {
            return outputPanel;
        }

        public void setOutputPanel(OutputPanel.Descriptor outputPanel) {
            this.outputPanel = outputPanel;
        }
        
        @Override
        public Descriptor clone() {
            try {
                Descriptor d = (Descriptor) super.clone();
                if (d.outputPanel != null) {
                    d.outputPanel = d.outputPanel.clone();
                }
                return d;
            } catch (CloneNotSupportedException x) {
                throw new RuntimeException(); // never
            }
        }
        
    }
    
}
