package org.lsst.ccs.gconsole.plugins.trending.timeselection;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;
import javax.swing.table.AbstractTableModel;
import org.freehep.application.PropertyUtilities;
import org.freehep.application.studio.Studio;

/**
 * Maintains a list of time window presets known to the application.
 *
 * @author onoprien
 */
public class PresetList extends AbstractTableModel {

// -- Private parts : ----------------------------------------------------------
    
    private final String KEY = "org.lsst.trending.timewindows";
    private final Studio app;
    
    private ArrayList<TimeWindow> data; // sorted by name

// -- Construction and initialization : ----------------------------------------
    
    public PresetList(Studio application) {
        app = application;
        restore();
    }
    

// -- Getters and setters : ----------------------------------------------------
    
    /** Returns a list of {@code maxSize} most recently used time windows ordered by name. */
    public List<TimeWindow> getRecent(int maxSize) {
        List<TimeWindow> out = new ArrayList<>(data);
        Collections.sort(out, TimeWindow.compareByTime());
        if (maxSize < out.size()) out = out.subList(0, maxSize);
        Collections.sort(out, TimeWindow.compareByName());
        return out;
    }
    
    /** Returns preset with the specified index. */
    public TimeWindow get(int index) {
        return data.get(index);
    }
    
    /** Returns the index of the specified preset in this list, or -1 if this list does not contain the preset. */
    public int indexOf(TimeWindow timeWindow) {
        return data.indexOf(timeWindow);
    }
    
    /**
     * Returns time window with the specified name.
     * @param name Name of the time window.
     * @return Time window, or {@code null} if this preset list does not contain a time window with the specified name.
     */
    public TimeWindow get(String name) {
        for (TimeWindow tw : data) {
            if (name.equals(tw.getName())) return tw;
        }
        return null;
    }
    
    
// -- Modifying the list : -----------------------------------------------------
    
    /**
     * Adds time window to this list if necessary, removing any identically named elements.
     * Unnamed windows are not added, but used to find and return windows with the same settings.
     * If this list already contains either identical or otherwise identical but persistent window,
     * the list remains unchanged and the existing window is returned. 
     * If this list contains identically named but different window, the existing window is removed.
     */
    public TimeWindow insert(TimeWindow timeWindow) {
        if (timeWindow.isReplaceableBy(TimeWindow.LAST_HOUR)) return TimeWindow.LAST_HOUR;
        if (timeWindow.isReplaceableBy(TimeWindow.LAST_6_HOURS)) return TimeWindow.LAST_6_HOURS;
        if (timeWindow.isReplaceableBy(TimeWindow.LAST_24_HOURS)) return TimeWindow.LAST_24_HOURS;
        String name = timeWindow.getName();
        if (name.isEmpty()) {
            for (TimeWindow tw : data) {
                if (timeWindow.isReplaceableBy(tw)) {
                    return tw.touch();
                }
            }
            return timeWindow;
        } else {
            for (int i=0; i<data.size(); i++) {
                TimeWindow e = data.get(i);
                int out = name.compareTo(e.getName());
                if (out == 0) {
                    if (timeWindow.isReplaceableBy(e)) {
                        return e.touch();
                    } else {
                        data.set(i, timeWindow);
                        if (e.isPersistent() || timeWindow.isPersistent()) {
                            save();
                        }
                        fireTableRowsUpdated(i, i);
                        return timeWindow;
                    }
                } else if (out < 0) { // no identically named window
                    data.add(i, timeWindow);
                    if (timeWindow.isPersistent()) save();
                    fireTableRowsInserted(i, i);
                    return timeWindow;
                }
            }
            int row = data.size();
            data.add(timeWindow);
            if (timeWindow.isPersistent()) save();
            timeWindow.touch();
            fireTableRowsInserted(row, row);
            return timeWindow;
        }
    }
    
    /**
     * Removes the specified window from this list.
     * Updates application properties if necessary.
     */
    public void delete(TimeWindow timeWindow) {
        if (data.remove(timeWindow) && timeWindow.isPersistent()) {
            save();
        }
        fireTableStructureChanged();
    }
    

// -- Implementing TableModel : ------------------------------------------------

    @Override
    public int getRowCount() {
        return data.size();
    }

    @Override
    public int getColumnCount() {
        return 4;
    }

    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
        switch (columnIndex) {
            case 0: return data.get(rowIndex).getName();
            case 1: return data.get(rowIndex).getLowerEdgeString();
            case 2: return data.get(rowIndex).getUpperEdgeString();
            case 3: return data.get(rowIndex).isPersistent();
            default: throw new IllegalArgumentException();
        }
    }

    @Override
    public Class<?> getColumnClass(int columnIndex) {
        return String.class;
    }

    @Override
    public String getColumnName(int column) {
        switch (column) {
            case 0: return "Name";
            case 1: return "Start";
            case 2: return "End";
            case 3: return "Saved";
            default: throw new IllegalArgumentException();
        }
    }
    
    
// -- Saving to / restoring from properties : ----------------------------------
    
    /**
     * Updates information on the specified time window in application properties.
     */
    private void save() {
        Properties prop = app.getUserProperties();
        if (prop != null) {
            int n = data.size();
            ArrayList<String> savedData = null;
            if (n != 0) {
                savedData = new ArrayList<>(n);
                for (int i=0; i<n; i++) {
                    TimeWindow tw = data.get(i);
                    if (tw.isPersistent()) {
                        savedData.add(tw.toNamedCompressedString());
                    }
                    savedData.add(tw.toNamedCompressedString());
                }
            }
            PropertyUtilities.setStringCollection(prop, KEY, savedData);
        }
    }
    
    /** Restore persistent time selections from application properties. */
    private void restore() {
        Properties prop = app.getUserProperties();
        if (prop == null) {
            data = new ArrayList<>(0);
        } else {
            String[] savedData = PropertyUtilities.getStringArray(prop, KEY, new String[0]);
            data = new ArrayList<>(savedData.length);
            Map<String, TimeWindow> windows = new TreeMap<>();
            for (String s : savedData) {
                try {
                    TimeWindow tw = TimeWindow.parseCompressedString(s);
                    windows.put(tw.getName(), tw);
                } catch (IllegalArgumentException x) {
                }
            }
            windows.remove("");
            data = new ArrayList<>(windows.values());
        }
    }
    
}
