package org.lsst.ccs.gconsole.plugins.jgroups;

import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;

/**
 * LSST Command Browser Plugin.
 * The original version of the command dictionary tool was written by Etienne Marin-Matholaz.
 * 
 * @author onoprien
 */
@Plugin(name="LSST JGroups Plugin",
        id="jgroups",
        description="LSST CCS JGroups plugin provides tools for monitoring and directly interacting with the JGroups messaging system.",
        shortDescription="LSST JGroups Monitor")
public class LsstJGroupsPlugin extends ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------
            
    private final ArrayList<ProbePage> probePages = new ArrayList<>(0);
    
   
// -- Life cycle : -------------------------------------------------------------
    
    public LsstJGroupsPlugin() {
    }

    @Override
    public void initialize() {
        Action newDictionaryAction = new AbstractAction("Probe") {
            @Override
            public void actionPerformed(ActionEvent e) {
                LsstJGroupsPlugin.this.createProbePage(null);
            }
        };
        getServices().addMenu(newDictionaryAction, "400: CCS Tools :-1:20", "JGroups:0");
    }

    
// -- Creating browsers : ------------------------------------------------------
    
    /**
     * Creates and displays a Probe interface page customized by the provided descriptor.
     * Called on EDT.
     * 
     * @param descriptor Descriptor of the page to be created.
     */
    private void createProbePage(ProbePage.Descriptor descriptor) {
        if (descriptor == null) descriptor = new ProbePage.Descriptor();
        ProbePage page = new ProbePage();
        page.restore(descriptor);
        Map<Object, Object> par = new HashMap<>();
        DataPanelDescriptor panDesc = descriptor.getPage();
        if (panDesc != null && panDesc.isOpen()) {
            Map<String, Serializable> data = panDesc.getData();
            if (data != null) {
                par.putAll(data);
            }
        }
        par.put(Panel.TITLE, "JGroups");
        Consumer<JComponent> onClose = c -> {
            probePages.remove(c);
            probePages.trimToSize();
        };
        par.put(Panel.ON_CLOSE, onClose);
        getConsole().getPanelManager().open(page, par);
        probePages.add(page);
        probePages.trimToSize();
    }
    
    
// -- Saving/restoring : -------------------------------------------------------

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        if (! (storageBean instanceof Descriptor)) return true;
        
        Descriptor desc = (Descriptor) storageBean;
        if (!probePages.isEmpty()) {
            ArrayList<ProbePage> copy = new ArrayList<>(probePages);
            PanelManager panMan = getConsole().getPanelManager();
            copy.forEach(pp -> panMan.close(pp));
            probePages.clear();
        }
       
        ProbePage.Descriptor[] dd = desc.getProbePages();
        if (dd != null) {
            for (ProbePage.Descriptor d : dd) {
                createProbePage(d);
            }
        }
        return true;
    }

    @Override
    public Descriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        if (!probePages.isEmpty()) {
            ProbePage.Descriptor[] descriptors = new ProbePage.Descriptor[probePages.size()];
            for (int i = 0; i < descriptors.length; i++) {
                descriptors[i] = probePages.get(i).save();
            }
            desc.setProbePages(descriptors);
        }
        return desc;
    }

    static public class Descriptor extends ComponentDescriptor {

        private ProbePage.Descriptor[] probePages;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public ProbePage.Descriptor[] getProbePages() {
            return probePages;
        }

        public void setProbePages(ProbePage.Descriptor[] probePages) {
            this.probePages = probePages;
        }
        
    }




    
}
