package org.lsst.ccs.gconsole.util.swing;

import java.awt.Dimension;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JComboBox;
import javax.swing.JFormattedTextField;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.text.NumberFormatter;
import static org.lsst.ccs.gconsole.base.Const.HSPACE;

/**
 * Time duration chooser component.
 *
 * @author onoprien
 */
public final class DurationSelector extends JPanel {

// -- Fields : -----------------------------------------------------------------
    
    static private final ChronoUnit[] defaultUnits = new ChronoUnit[] {ChronoUnit.MONTHS, ChronoUnit.WEEKS, ChronoUnit.DAYS, ChronoUnit.HOURS, ChronoUnit.MINUTES, ChronoUnit.SECONDS};
    
    private final SpinnerNumberModel amountModel;
    private final JSpinner amountSpinner;
    
    private final JComboBox unitCombo;
    
    private final ChronoUnit[] units;
    
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList<>();
    private boolean notMuted = true;

// -- Life cycle : -------------------------------------------------------------
    
    public DurationSelector(int amount, ChronoUnit unit, ChronoUnit... units) {
        super();
        setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
        
        this.units = units.length == 0 ? defaultUnits : units;
        
        amountModel = new SpinnerNumberModel(amount, 1, Integer.MAX_VALUE, 1);
        amountSpinner = new JSpinner(amountModel);
        try {
            JFormattedTextField tf = ((JSpinner.DefaultEditor)amountSpinner.getEditor()).getTextField();
            tf.setColumns(10);
            NumberFormatter f = (NumberFormatter) tf.getFormatter();
            f.setCommitsOnValidEdit(true);
            f.setAllowsInvalid(false);
        } catch (ClassCastException x) {}
//        amountSpinner.setMaximumSize(amountSpinner.getPreferredSize());
        amountSpinner.addChangeListener(e -> fire());
        add(amountSpinner);
        
        add(Box.createRigidArea(new Dimension(HSPACE, 0)));
        
        unitCombo = new JComboBox(units);
        unitCombo.setMaximumSize(unitCombo.getPreferredSize());
        unitCombo.addActionListener(e -> fire());
        add(unitCombo);
        
        setMaximumSize(getPreferredSize());
        setAlignmentX(LEFT_ALIGNMENT);
    }
    
    public DurationSelector(int amount, ChronoUnit unit) {
        this(amount, unit, defaultUnits);
    }
    
    public DurationSelector() {
        this(1, ChronoUnit.DAYS, defaultUnits);
    }
    
    
// -- Getters and setters : ----------------------------------------------------
    
    public void setAmountLimits(int min, int max) {
        amountModel.setMinimum(min);
        amountModel.setMaximum(max);
    }
    
    /**
     * Returns the currently selected duration in seconds.
     * @return Duration in seconds.
     */
    public long get() {
        return getUnit().getDuration().multipliedBy(getAmount()).getSeconds();
    }
    
    public int getAmount() {
        return (int) amountModel.getNumber();
    }
    
    public ChronoUnit getUnit() {
        return (ChronoUnit) unitCombo.getSelectedItem();
    }
    
    public void set(long seconds) {
        if (seconds < 1) seconds = 1;
        long am = 0L;
        ChronoUnit un = units[0];
        for (ChronoUnit u : units) {
            long s = u.getDuration().getSeconds();
            long a = seconds/s;
            if (seconds % s == 0L) {
                am = a;
                un = u;
                break;
            } else if (a > Integer.MAX_VALUE) {
                break;
            }
        }
        notMuted = false;
        amountModel.setValue((int)am);
        unitCombo.setSelectedItem(un);
        notMuted = true;
    }
    
    public void set(int amount, ChronoUnit unit) {
        if (amount < 1) amount = 1;
        int index = -1;
        for (int i=0; i<units.length; i++) {
            if (units[i].equals(unit)) {
                index = i;
                break;
            }
        }
        if (index == -1) {
            set(unit.getDuration().multipliedBy(amount).getSeconds());
        } else {
            notMuted = false;
            amountModel.setValue(amount);
            unitCombo.setSelectedIndex(index);
            notMuted = true;
        }
    }
    
    public void setRounded(long seconds) {
        for (ChronoUnit u : units) {
            long s = seconds / u.getDuration().getSeconds();
            if (s > 0L) {
                set((int)s, u);
                return;
            }
        }
    }

    @Override
    public void setEnabled(boolean enabled) {
        amountSpinner.setEnabled(enabled);
        unitCombo.setEnabled(enabled);
        super.setEnabled(enabled);
    }

    @Override
    public void setToolTipText(String text) {
        super.setToolTipText(text);
        amountSpinner.setToolTipText(text);
        unitCombo.setToolTipText(text);
    }

// -- Handling listeners : -----------------------------------------------------
    
    public interface Listener {
        void changed(DurationSelector source);
    }
    
    public void addListener(Listener listener) {
        listeners.addIfAbsent(listener);
    }
    
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }
    
    protected void fire() {
        if (notMuted) {
            listeners.forEach(listener -> listener.changed(this));
        }
    }
        
}
