package org.lsst.ccs.gconsole.services.lock;

import java.awt.event.ActionEvent;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.ConsoleService;
import org.lsst.ccs.gconsole.util.ThreadUtil;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentLoginService;
import org.lsst.ccs.services.UnauthorizedLockException;

/**
 * Class that handles interactions between various command browser components and agent services involved in lock management.
 * <p>
 * Getters provided by this class should be used to retrieve lock management information related to
 * multiple remote agents. For information on a specific agent, getters of the corresponding
 * {@link Locker} instance can be used. All getters should be called on EDT. 
 * <p>
 * To be notified of changes, command browser components register listeners
 * through this class. {@link Listener} instances added through {@code addListener(Listener)}
 * method are notified whenever the set of watched agents changes (a remote agent is watched
 * if it is either connected to the buses or is locked by someone). To be notified of changes
 * related to a specific agent, a {@code ChangeListener} should be registered on the corresponding
 * {@link Locker} instance. Listener registration and removal can be requested on any thread,
 * but the callbacks are always executed on EDT. Note that if an unlocked subsystem disconnects
 * from the buses, the corresponding {@link Locker} instance is destroyed. Listeners that wish
 * to resume receiving subsystem-specific events once the subsystem reconnects, should listen to
 * service events and register themselves on the new {@link Locker} when it is created.
 * <p>
 * To initiate lock management related operations, clients can call one of the 
 * {@link #executeOperation(Operation, String, int)}, {@link #executeBulkOperation(Operation, List)}, and {@link #login(String, String)} methods.
 * These methods can be called on any thread. If the requested operation involves 
 * communicating over the buses, it will be offloaded to a worker thread.
 *
 * @author onoprien
 */
@Plugin(name = "Lock Service Plugin",
        id="lock-service",
        description = "Graphical console service that provides interface between GUI components and Agent services dealing with locks, levels, logins, etc.")
public class LockService extends ConsolePlugin implements ConsoleService {
    
    /** Lock management operations. */
    public enum Operation {
        LOCK("Lock"), 
        UNLOCK("Unlock"), 
        ATTACH("Attach"), 
        DETACH("Detach"), 
        DESTROY("Destroy"), 
        LEVEL("Set level");
        
        private final String humanReadable;
    
        Operation(String humanReadable) {
            this.humanReadable = humanReadable;
        }
        
        public Action getAction(String agentName) {
            Action act = new AbstractAction(humanReadable) {
                @Override
                public void actionPerformed(ActionEvent e) {
                    LockService.getService().executeOperation(Operation.this, agentName, -1);
                }
            };
            act.putValue(Action.SHORT_DESCRIPTION, humanReadable + " " + agentName);
            return act;
        }

        @Override
        public String toString() {
            return humanReadable;
        }
    }

    /** States of remote agents with respect to lock management. */
    public enum State {
        /** Unlocked.*/ UNLOCKED,
        /** Locked by someone else.*/ LOCKED,
        /** Locked by me, not attached.*/ DETACHED,
        /** Locked by me, not attached.*/ ATTACHED
    }

// -- Fields : -----------------------------------------------------------------
        
    private final TreeMap<String,Locker> agents = new TreeMap<>();
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList<>();
    
    private final Console console = Console.getConsole();
    private final AgentLockService lockService = console.getAgentService(AgentLockService.class);
    private final AgentLoginService loginService = console.getAgentService(AgentLoginService.class);
    
    private final AgentLockService.AgentLockUpdateListener lockListener;
    private final AgentLoginService.AgentLoginUpdateListener loginListener;
    private final AgentCommandDictionaryService.AgentCommandDictionaryListener dictionaryListener;
    
    private final LinkedList<String> users = new LinkedList<>();

// -- Life cycle : -------------------------------------------------------------
    
    public LockService() {
        
        lockListener = new AgentLockService.AgentLockUpdateListener() {
            
            @Override
            public void onAgentHeldLockUpdate(String agentName, AgentLock lock) {
                lockUpdate(agentName, lock);
            }
            
            @Override
            public void onGlobalLockUpdate(String agentName, String owner, AgentLock lock) {
                lockUpdate(agentName, lock);
            }
            
            private void lockUpdate(String agentName, AgentLock aLock) {
                if (aLock instanceof AgentLockInfo && ((AgentLockInfo)aLock).getStatus() == AgentLockInfo.Status.REQUESTED) return;
                ThreadUtil.invokeLater(() -> {
                    Locker agent = agents.get(agentName);
                    AgentLock lock = aLock;
                    if (lock instanceof AgentLockInfo) {
                        AgentLockInfo.Status lockStatus = ((AgentLockInfo)lock).getStatus();
                        if (AgentLockInfo.Status.RELEASED == lockStatus || AgentLockInfo.Status.REJECTED == lockStatus) {
                            lock = null;
                        }
                    }
                    if (agent == null) {
                        if (lock != null) {
                            agent = new Locker(agentName, lockService);
                            agent.onLock(lock);
                            agents.put(agentName, agent);
                            notifyListenersAdd(agent);
                        }
                    } else {
                        agent.onLock(lock);
                        if (lock == null && !agent.isOnline()) {
                            agents.remove(agentName);
                            agent.removeAllListeners();
                            notifyListenersRemove(agent);
                        } else {
                            notifyListenersUpdate(agent);
                        }
                    }
                });
            }

            @Override
            public void onAgentLevelChange(String agentName, int level) {
                ThreadUtil.invokeLater(() -> {
                    Locker agent = agents.get(agentName);
                    if (agent != null) {
                        agent.onLevel(level);
                        notifyListenersUpdate(agent);
                    }
                });
            }
            
        };
        
        dictionaryListener = e -> {
            ThreadUtil.invokeLater(() -> {
                AgentInfo agentInfo = e.getAgentInfo();
                String name = agentInfo.getName();
                Locker agent = agents.get(name);
                switch (e.getEventType()) {
                    case ADDED: // agent comes online
                        boolean newAgent = (agent == null);
                        if (newAgent) {
                            agent = new Locker(name, lockService);
                            agents.put(name, agent);
                        }
                        agent.onOnline(agentInfo, e.getDictionary());
                        if (newAgent) {
                            notifyListenersAdd(agent);
                        }
                        break;
                    case REMOVED: // agent goes offline
                        if (agent != null) {
                            agent.onOnline(null, null);
                            if (agent.getLock() == null) {
                                agents.remove(name);
                                agent.removeAllListeners();
                                notifyListenersRemove(agent);
                            } else {
                                notifyListenersUpdate(agent);
                            }
                        }
                        break;
                }
            });
        };
        
        loginListener = new AgentLoginService.AgentLoginUpdateListener() {
            @Override
            public void onAgentLoginUpdate(String oldUserId, String newUserId) {
                ThreadUtil.invokeLater(() -> {
                    
                    // Update console title bar
                    
                    try {
                        JFrame top = (JFrame) console.getWindow();
                        top.setTitle("CCS Console : "+ newUserId);
                    } catch (RuntimeException x) {
                    }
                    
                    // Update Lockers
                    
                    agents.values().forEach(agent -> {
                        agent.onLogin(oldUserId, newUserId);
                    });
                    
                    // Notify listeners
                    
                    listeners.forEach(listener -> {
                        try {
                            listener.userChanged(oldUserId, newUserId);
                        } catch (RuntimeException x) {
                            Console.getConsole().getLogger().error("Error notifying LockService listeners of user name change", x);
                        }
                    });
                });
            }
        };
        
    }
    
    static public LockService getService() {
        return Console.getConsole().getSingleton(LockService.class);
    }
    
    @Override
    public void startService() {
        
        lockService.addAgentLockUpdateListener(lockListener);
        loginService.addAgentLoginUpdateListener(loginListener);
        console.getAgentService(AgentCommandDictionaryService.class).addAgentCommandDictionaryListener(dictionaryListener);
        
        Action act = new AbstractAction("Login ...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                JComboBox<String> box = new JComboBox<>(users.toArray(new String[0]));
                box.setEditable(true);
                int out = JOptionPane.showConfirmDialog(getConsole().getWindow(), box, "Login as", JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE);
                if (out == JOptionPane.OK_OPTION) {
                    String osUser = System.getProperty("user.name", "unknown");
                    String currentUser = lockService.getUserId();
                    String newUser = box.getSelectedItem().toString().trim();
                    if (newUser.isEmpty()) {
                        newUser = osUser;
                    }
                    if (!currentUser.equals(newUser)) {
                        users.remove(newUser);
                        if (!users.isEmpty() && users.getFirst().equals(osUser)) {
                            users.add(1, currentUser);
                        } else {
                            users.addFirst(currentUser);
                        }
                        if (users.size() > 10) {
                            users.removeLast();
                        }
                        login(newUser, "");
                    }
                }
            }
        };
        getServices().addMenu(act, "File:-2:1");

    }
    
    @Override
    public void stopService() {
        console.getAgentService(AgentCommandDictionaryService.class).removeAgentCommandDictionaryListener(dictionaryListener);
        lockService.removeAgentLockUpdateListener(lockListener);
    }

    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns {@link Locker} for the specified agent.
     * Like all getters of this class, this method should be called on EDT.
     * 
     * @param agentName Agent name.
     * @return {@code Locker} for the specified agent, or {@code null} if the agent is unlocked and offline.
     */
    public Locker getAgent(String agentName) {
        return agents.get(agentName);
    }
    
    /**
     * Returns {@link Locker}s for agents in one of the specified states.Like all getters of this class, this method should be called on EDT.
     *
     * @param states States to include.
     * @return {@code Locker} for the specified agent, or {@code null} if the agent is unlocked and offline.
     */
    public List<Locker> getAgents(State... states) {
        switch (states.length) {
            case 0:
                return new ArrayList<>(agents.values());
            case 1:
                State state = states[0];
                return agents.values().stream().filter(locker -> locker.getState() == state).collect(Collectors.toList());
            default:
                EnumSet<State> ss = EnumSet.copyOf(Arrays.asList(states));
                return agents.values().stream().filter(locker -> ss.contains(locker.getState())).collect(Collectors.toList());
        }
    }
    
    /**
     * Returns user ID of this console.
     * Like all getters of this class, this method should be called on EDT.
     * 
     * @return User ID.
     */
    public String getUserId() {
        return lockService.getUserId();
    }
    
    /**
     * Lists names of agents in the specified states.
     * Like all getters of this class, this method should be called on EDT.
     * 
     * @param states States to include.
     * @return List of agent names.
     */
    public List<String> getAgentNames(State... states) {
        switch (states.length) {
            case 0:
                return agents.values().stream()
                        .map(locker -> locker.getName()).collect(Collectors.toList());
            case 1:
                State state = states[0];
                return agents.values().stream()
                        .filter(locker -> locker.getState() == state)
                        .map(locker -> locker.getName()).collect(Collectors.toList());
            default:
                EnumSet<State> ss = EnumSet.copyOf(Arrays.asList(states));
                return agents.values().stream()
                    .filter(locker -> ss.contains(locker.getState()))
                    .map(locker -> locker.getName()).collect(Collectors.toList());
        }
    }

    
// -- Lock management operations : ---------------------------------------------
    
    /**
     * Executes the specified operation on multiple agents.
     * This method can be called on any thread. If the requested operation involves 
     * communicating over the buses, it will be offloaded to a worker thread.
     * 
     * @param op Operation.
     * @param agentNames Agent names.
     */
    public void executeBulkOperation(Operation op, List<String> agentNames) {
        ThreadUtil.invokeLater(() -> {
            agentNames.forEach(agentName -> {
                Locker agent = agents.get(agentName);
                if (agent != null) {
                    agent.adjusting = true;
                    agent.notifyListeners();
                }
            });
            new SwingWorker<Map<String, Exception>, Object>() {
                @Override
                protected Map<String, Exception> doInBackground() throws Exception {
                    Map<String, Exception> failures = new LinkedHashMap<>();
                    agentNames.forEach(agentName -> {
                        try {
                            switch (op) {
                                case LOCK:
                                    lockService.lockAgent(agentName);
                                    break;
                                case UNLOCK:
                                    lockService.unlockAgent(agentName);
                                    break;
                                case ATTACH:
                                    lockService.attachLock(agentName);
                                    break;
                                case DETACH:
                                    lockService.detachLock(agentName);
                                    break;
                                default:
                                    throw new UnsupportedOperationException(op + " cannot be performed on multiple subsystems");
                            }
                        } catch (UnauthorizedLockException | RuntimeException x) {
                            failures.put(agentName, x);
                        }
                    });
                    return failures;
                }
                @Override
                protected void done() {
                    try {
                        Map<String, Exception> failures = get();
                        if (!failures.isEmpty()) {
                            StringBuilder sb = new StringBuilder("Failed to execute ").append(op).append(" for these subsystems:").append(System.lineSeparator());
                            failures.forEach((agent,exception) -> {
                                sb.append(agent).append(" -> ").append(exception).append(System.lineSeparator());
                            });
                            Console.getConsole().error(sb.toString());
                        }
                    } catch (InterruptedException x) {
                        Console.getConsole().getLogger().warn("ConsoleLockService: AWT interrupted");
                        Thread.currentThread().interrupt();
                    } catch (ExecutionException x) {
                        try {
                            Console.getConsole().error("Unable to "+ op +" on "+ agentNames, (Exception) x.getCause());
                        } catch (ClassCastException xx) {
                            Console.getConsole().error("Unable to "+ op +" on "+ agentNames, x);
                        }
                    }
                    agentNames.forEach(agentName -> {
                        Locker agent = agents.get(agentName);
                        if (agent != null) {
                            agent.adjusting = false;
                            agent.notifyListeners();
                        }
                    });
                }
            }.execute();
        });
    }    
    
    /**
     * Executes the specified operation on the given agent.
     * This method can be called on any thread. If the requested operation involves 
     * communicating over the buses, it will be offloaded to a worker thread.
     * 
     * @param op Operation.
     * @param agentName Agent name.
     * @param level Desired level (ignored for operations other than {@code LEVEL}.
     */
    public void executeOperation(Operation op, String agentName, int level) {
        ThreadUtil.invokeLater(() -> {
            
            Locker agent = agents.get(agentName);
            
            // Pre-execution communication with the user:
            
            int finalLevel;
            int out;
            switch (op) {
                case ATTACH:
                    out = JOptionPane.showConfirmDialog(
                        Console.getConsole().getWindow(),
                        "<html>You are about to attach an existing lock owned by user \"" + lockService.getUserId() + ".\" This lock may<br> already be in use by another console or a script running under the same user name.<p> <p>Would you like to proceed?",
                        "Attaching existing lock", JOptionPane.OK_CANCEL_OPTION);
                    if (out != JOptionPane.OK_OPTION) return;
                    finalLevel = level;
                    break;
                case DESTROY:
                    out = JOptionPane.showConfirmDialog(
                        Console.getConsole().getWindow(),
                        "<html>Are you sure you want to destroy <b>"+ agentName +"</b> subsystem lock owned by user <b>" + agent.getLock().getOwner() + "</b>?<br> If practical, please contact the lock owner instead.<p> <p>Would you like to proceed?",
                        "Destroying lock", JOptionPane.OK_CANCEL_OPTION);
                    if (out != JOptionPane.OK_OPTION) return;
                    finalLevel = level;
                    break;
                case LEVEL:
                    if (level < 0) {
                        String s = (String) JOptionPane.showInputDialog(Console.getConsole().getWindow(), "Desired level for "+ agentName, "Select level", JOptionPane.QUESTION_MESSAGE, null, null, 0);
                        try {
                            finalLevel = Integer.parseInt(s);
                            if (finalLevel < 0 || finalLevel > 99) throw new NumberFormatException();
                        } catch (NumberFormatException x) {
                            Console.getConsole().error("Illegal lock level. Must be integer between 0 and 99.");
                            return;
                        }
                    } else {
                        finalLevel = level;
                    }
                    break;
                default:
                    finalLevel = level;
            }
            
            // Notify listeners we are staring operation:
            
            if (agent != null) {
                agent.adjusting = true;
                agent.notifyListeners();
            }
            
            // Actual communication with the lock service is outsourced to another thread:
            
            AgentLock lock = agent == null ? null : agent.getLock();
            State state = agent == null ? State.UNLOCKED : agent.getState();
            new SwingWorker<Object, Object>() {
                @Override
                protected Object doInBackground() throws Exception {
                    switch (op) {
                        case LOCK:
                            lockService.lockAgent(agentName);
                            break;
                        case UNLOCK:
                            lockService.unlockAgent(agentName);
                            break;
                        case ATTACH:
                            lockService.attachLock(agentName);
                            break;
                        case DETACH:
                            lockService.detachLock(agentName);
                            break;
                        case DESTROY:
                            if (lock != null) {
                                lockService.destroyLock(agentName, lockService.getUserId());
                            }
                            break;
                        case LEVEL:
                            switch (state) {
                                case UNLOCKED:
                                    lockService.lockAgent(agentName);
                                    break;
                                case DETACHED:
                                    lockService.attachLock(agentName);
                                    break;
                            }
                            lockService.setLevelForAgent(agentName, finalLevel);
                            break;
                    }
                    return null;
                }
                @Override
                protected void done() {
                    try {
                        get();
                    } catch (InterruptedException x) {
                        console.getLogger().warn("ConsoleLockService: AWT interrupted");
                        Thread.currentThread().interrupt();
                    } catch (ExecutionException x) { // FIXME: beautify
                        try {
                            Exception cause = (Exception) x.getCause();
                            console.error("Unable to "+ op +" "+ agentName, cause);
                            AgentLock realLock = lockService.getExistingLockForAgent(agentName);
                            lockListener.onGlobalLockUpdate(agentName, null, realLock);
                        } catch (ClassCastException xx) {
                            console.error("Unable to "+ op +" "+ agentName, x);
                        }
                    }
                    Locker agent = agents.get(agentName);
                    if (agent != null) {
                        agent.adjusting = false;
                        agent.notifyListeners();
                    }
                }
            }.execute();
        });
    }
    
    /**
     * Changes user ID associated with this console.
     * This method can be called on any thread. If the requested operation involves 
     * communicating over the buses, it will be offloaded to a worker thread.
     * 
     * @param userID Desired user ID, or {@code null} if the user ID should be reset to system user name.
     * @param credentials Ignored in the current implementation.
     */
    public void login(String userID, String credentials) { // FIXME: re-implement once login service in toolkit is updated
        ThreadUtil.invokeLater(() -> {
            String oldID = loginService.getUserId();
            boolean change = !Objects.equals(oldID, userID);
            if (change) {
                try {
                    loginService.disconnect();
                } catch (RuntimeException x) {
                }
                if (userID != null) {
                    try {
                        loginService.login(userID, credentials);
                    } catch (RuntimeException x) {
                        console.error("Failed to login as "+ userID, x);
                    }
                }
            }
        });
    }

    
// -- Handling listeners : -----------------------------------------------------
    
    /**
     * Listener interface for receiving notifications of changes in the set of agents watched by this service, and of changes in userID.
     * An agent is watched if it is either connected to the buses or is locked by someone.
     * <p>
     * All notifications are delivered on EDT.
     */
    public interface Listener {
        
        /**
         * Called when new agents are added to the watch list.
         * New listeners are immediately notified of all watched agents.
         * 
         * @param agents Added agents.
         */
        default void agentsAdded(Locker... agents) {}
        
        /**
         * Called when agents are removed from the watch list.
         * 
         * @param agents Removed agents.
         */
        default void agentsRemoved(Locker... agents) {}
        
        /**
         * Called when there are changes in agents lock status.
         * 
         * @param agents Agents with changes.
         */
        default void agentsUpdated(Locker... agents) {}
        
        /**
         * Called when the user name associated with this console changes.
         * There are no guarantees on whether this method is called before or after individual
         * {@link Locker} listeners are notified of changes triggered by the user name change.
         * 
         * @param oldUserID Previous user name.
         * @param newUserID Current user name.
         */
        default void userChanged(String oldUserID, String newUserID) {}
        
    }
    
    /**
     * Adds a listener to be notified of changes in the set of agents watched by this service, and of changes in userID.
     * @param listener Listener to add.
     */
    public void addListener(Listener listener) {
        SwingUtilities.invokeLater(() -> {
            listeners.addIfAbsent(listener);
            Locker[] existingAgents = agents.values().toArray(new Locker[agents.size()]);
            listener.agentsAdded(existingAgents);
        });
    }
    
    /**
     * Removes a listener.
     * @param listener Listener to remove.
     */
    public void removeListener(Listener listener) {
        ThreadUtil.invokeLater(() -> {
            listeners.remove(listener);
        });
    }

    private void notifyListenersAdd(Locker... agentHandles) {
        listeners.forEach(listener -> {
            try {
                listener.agentsAdded(agentHandles);
            } catch (RuntimeException x) {
                Console.getConsole().getLogger().error("Error notifying LockService listeners of " + 
                        String.join(",", Arrays.stream(agentHandles).map(a -> a.getName()).collect(Collectors.toList())) +" addition", x);
            }
        });
    }

    private void notifyListenersRemove(Locker... agentHandles) {
        listeners.forEach(listener -> {
            try {
                listener.agentsRemoved(agentHandles);
            } catch (RuntimeException x) {
                Console.getConsole().getLogger().error("Error notifying LockService listeners of " + 
                        String.join(",", Arrays.stream(agentHandles).map(a -> a.getName()).collect(Collectors.toList())) +" removal", x);
            }
        });
    }

    private void notifyListenersUpdate(Locker... agentHandles) {
        listeners.forEach(listener -> {
            try {
                listener.agentsUpdated(agentHandles);
            } catch (RuntimeException x) {
                Console.getConsole().getLogger().error("Error notifying LockService listeners of " + 
                        String.join(",", Arrays.stream(agentHandles).map(a -> a.getName()).collect(Collectors.toList())) +" update", x);
            }
        });
    }
    
    

}
