package org.lsst.ccs.gconsole.services.aggregator;

import java.util.*;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusMessage;

/**
 * Mutable extension of {@link AgentStatusEvent}.
 * Provides methods to accumulate changes.
 *
 * @author onoprien
 */
public class MutableAgentStatusEvent extends AgentStatusEvent {

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs an event for statusChanged call.
     * This event can only contain the list of modified channels. 
     * An attempt to call {@code addAddedChannel(...)} or {@code addRemovedChannel(...)} method will result in {@code UnsupportedOperationException}.
     * 
     * @param aggregator Status aggregator instance.
     * @param message Message that triggered this event.
     */
    MutableAgentStatusEvent(AgentStatusAggregator aggregator, StatusMessage message) {
        super(aggregator, null, message, new LinkedHashMap<>(), Collections.emptyList(), Collections.emptyList());
    }
    
    /**
     * Constructs an event for connect, disconnect, and configure calls.
     * This event can only contain lists of added or removed channels. 
     * An attempt to call {@code addChange(...)} method will result in {@code UnsupportedOperationException}.
     * The list of affected agents is compiled automatically when {@code validate()} is called.
     * 
     * @param aggregator Status aggregator instance.
     */
    MutableAgentStatusEvent(AgentStatusAggregator aggregator) {
        super(aggregator, new ArrayList<AgentInfo>(1), null, Collections.emptyMap(), new ArrayList<>(), new ArrayList<>());
    }
        
// -- Modification : -----------------------------------------------------------
    
    public void addChange(AgentChannel channel, String attributeKey) {
        if (attributeKey == null) attributeKey = AgentChannel.Key.VALUE;
        List<String> changedAttributes = statusChanges.get(channel);
        if (changedAttributes == null) {
            changedAttributes = new ArrayList<>(1);
            statusChanges.put(channel, changedAttributes);
        }
        changedAttributes.add(attributeKey);
    }
    
    public void addAddedChannel(AgentChannel channel) {
        addedChannels.add(channel);
    }
    
    public void addAddedChannels(Collection<? extends AgentChannel> channels) {
        addedChannels.addAll(channels);
    }
    
    public void addRemovedChannel(AgentChannel channel) {
        removedChannels.add(channel);
    }
    
    public void addRemovedChannels(Collection<? extends AgentChannel> channels) {
        removedChannels.removeAll(channels);
    }

    /**
     * Validates this event before sending it to listeners.
     */
    public void validate() {
        if (agents.isEmpty()) {
            Map<String,AgentInfo> aa = new LinkedHashMap<>();
            for (AgentChannel c : addedChannels) {
                aa.putIfAbsent(c.getAgentName(), c.getAgent());
            }
            for (AgentChannel c : removedChannels) {
                aa.putIfAbsent(c.getAgentName(), c.getAgent());
            }
            agents.addAll(aa.values());
        }
    }
    
}
