package org.lsst.ccs.gconsole.services.lock;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.services.AgentLockService;

/**
 * Lock management service handle for a specific agent.
 * The equality of instances of this class is based on agent names.
 * All access to this class should be on EDT.
 */
public class Locker {

// -- Fields : -----------------------------------------------------------------

    private final AgentLockService lockService;
    private final String agentName;
    private AgentInfo info;
    private Map<String, Dictionary> dictionaries; // null if agent is offline
    private AgentLock lock; // current lock
    private int level; // current lock level
    private int maxLevel; // maximum level for the current user
    private LockService.State state; // current state
    boolean adjusting; // true if a change in lock or level has been requested but not completed
    
    private final CopyOnWriteArrayList<ChangeListener> listeners = new CopyOnWriteArrayList<>();
    private final ChangeEvent changeEvent = new ChangeEvent(this);

// -- Life cycle : -------------------------------------------------------------
    
    /** Constructor for an agent that is currently offline. */
    Locker(String name, AgentLockService lockService) {
        agentName = name;
        this.lockService = lockService;
    }


// -- Updates : ----------------------------------------------------------------
    
    /**
     * Set the lock, updates internal state, and notifies listeners.
     */
    void onLock(AgentLock lock) {
        if (lock instanceof AgentLockInfo) {
            switch (((AgentLockInfo) lock).getStatus()) {
                case RELEASED:
                case REJECTED:
                    lock = null;
                    break;
                case REQUESTED:
                    return;
            }
        }
        this.lock = lock;
        if (lock == null) {
            state = LockService.State.UNLOCKED;
            maxLevel = lockService.getMaxLevel(lockService.getUserId(), agentName);
        } else {
            if (lock.getOwner().equals(lockService.getUserId())) {
                if (lockService.getDetachableAgents().contains(lock.getAgentName())) {
                    state = LockService.State.ATTACHED;
                    maxLevel = lock.getMaxLevel();
                } else {
                    state = LockService.State.DETACHED;
                    maxLevel = lockService.getMaxLevel(lockService.getUserId(), agentName);
                }
            } else {
                state = LockService.State.LOCKED;
                maxLevel = -1;
            }
        }
        level = lockService.getLevelForAgent(agentName);
        notifyListeners();
    }

    /**
     * Set the level and notifies listeners.
     */
    void onLevel(int level) {
        if (this.level != level) {
            this.level = level;
            notifyListeners();
        }
    }

    /**
     * Set the "adjusting" property and notifies listeners if necessary.
     */
    void onAdjust(boolean adjusting) {
        if (this.adjusting != adjusting) {
            this.adjusting = adjusting;
            notifyListeners();
        }
    }

    /**
     * Set the "online" property and notifies listeners if necessary.
     */
    void onOnline(AgentInfo info, Map<String, Dictionary> dictionaries) {
        this.info = info;
        if (info == null) {
            this.dictionaries = null;
        } else {
            this.dictionaries = new HashMap<>(dictionaries);
        }
        onLock(lockService.getExistingLockForAgent(agentName));
    }

    void onLogin(String oldUser, String newUser) {
        onLock(lock);
    }


// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns agent name.
     * @return Agent name (never changes).
     */
    public String getName() {
        return agentName;
    }

    /**
     * Returns agent descriptor, if available.
     * @return Agent descriptor, or {@code null} in the agent has never been online since this handle creation.
     */
    public AgentInfo getInfo() {
        return info;
    }

    /**
     * Returns current lock for the agent.
     * @return Current lock, or {@code null} if the agent is unlocked.
     */
    public AgentLock getLock() {
        return lock;
    }

    /**
     * Returns current level for the agent.
     * @return Current level (always 0 if the agent is not attached to this console).
     */
    public int getLevel() {
        return level;
    }
    
    /**
     * Returns maximum level the agent is authorized to set for the target agent.
     * If the target is locked by someone else, returns -1.
     * @return Maximum possible level.
     */
    public int getMaxLevel() {
        return maxLevel;
    }

    /**
     * Returns current state for the agent.
     * @return Current state.
     */
    public LockService.State getState() {
        return state;
    }

    /**
     * Checks whether the agent is currently connected to the buses and its command dictionary is available.
     * @return True if the agent is online.
     */
    public boolean isOnline() {
        return dictionaries != null;
    }

    /**
     * Checks whether the locking state of the agent is in the process of being adjusted.
     * @return True if this console is performing an operation that might result in the status change.
     */
    public boolean isAdjusting() {
        return adjusting;
    }

    /**
     * Returns command dictionaries for the agent.
     * @return Command dictionaries.
     */
    public Map<String, Dictionary> getDictionaries() {
        return dictionaries == null ? Collections.emptyMap() : Collections.unmodifiableMap(dictionaries);
    }


// -- Handling listeners : -----------------------------------------------------
    
    public boolean addListener(ChangeListener listener) {
        return listeners.addIfAbsent(listener);
    }

    public boolean removeListener(ChangeListener listener) {
        return listeners.remove(listener);
    }

    public boolean removeAllListeners() {
        boolean out = !listeners.isEmpty();
        listeners.clear();
        return out;
    }

    void notifyListeners() {
        listeners.forEach((javax.swing.event.ChangeListener listener) -> {
            try {
                listener.stateChanged(changeEvent);
            } catch (RuntimeException x) {
                Console.getConsole().getLogger().error("Error processing lock change event from " + getName(), x);
            }
        });
    }


// -- Overriding Object : ------------------------------------------------------
    
    /**
     * Computes hash code based on agent name.
     * @return Hash code.
     */
    @Override
    public int hashCode() {
        return getName().hashCode();
    }

    /**
     * Checks equality based on agent name.
     * @param obj The other object.
     * @return True if equal.
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof Locker) {
            return getName().equals(((Locker) obj).getName());
        } else {
            return false;
        }
    }
    
}
