package org.lsst.ccs.gconsole.plugins.tutorial;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.freehep.application.mdi.PageEvent;
import org.freehep.application.mdi.PageListener;
import org.freehep.application.studio.Studio;
import org.freehep.jas.plugin.console.Console;
import org.freehep.jas.plugin.console.ConsoleInputStream;
import org.freehep.jas.plugin.console.ConsoleOutputStream;
import org.freehep.jas.plugin.console.ConsoleService;

/**
 * A class which illustrates how to create input, output and input/output
 * consoles.
 *
 * @author tonyj
 */
class TutorialConsole {

    private static final Logger logger = Logger.getLogger(TutorialPlugin.class.getName());
    private final Studio studio;

    TutorialConsole(Studio studio) {
        this.studio = studio;
    }

    void createOutputConsole() {
        // The basic JAS console functionality is provided by the console
        // plugin. We need to look that up before we can open a console.
        ConsoleService consoleService = (ConsoleService) studio.getLookup().lookup(ConsoleService.class);
        // Although the console plugin is included with JAS we shold always protect
        // against the possibility of it not being installed
        if (consoleService != null) {
            Console console = consoleService.createConsole("Output Console", null);
            sendOutputToConsole(console, consoleService);
        }
    }

    void createInputConsole() {
        // The basic JAS console functionality is provided by the console
        // plugin. We need to look that up before we can open a console.
        ConsoleService consoleService = (ConsoleService) studio.getLookup().lookup(ConsoleService.class);
        // Although the console plugin is included with JAS we shold always protect
        // against the possibility of it not being installed
        if (consoleService != null) {
            // Create a new console, and get an input string for reading the users input.
            final Console console = consoleService.createConsole("Input Console", null);
            readInputFromConsole(console);
        }
    }

    void createInputOutputConsole() {
        // The basic JAS console functionality is provided by the console
        // plugin. We need to look that up before we can open a console.
        ConsoleService consoleService = (ConsoleService) studio.getLookup().lookup(ConsoleService.class);
        // Although the console plugin is included with JAS we shold always protect
        // against the possibility of it not being installed
        if (consoleService != null) {
            final Console console = consoleService.createConsole("Input/Output Console", null);
            sendOutputToConsole(console, consoleService);
            readInputFromConsole(console);
        }
    }

    private void sendOutputToConsole(Console console, ConsoleService consoleService) {
        // Create a PrintWriter that can be used to send output to the console
        // Note that the output stream created by the console is thread safe, so it is OK
        // to print to it from any thread (such as the Timer thread created below).
        final ConsoleOutputStream consoleOutputStream = console.getOutputStream(null);
        final PrintWriter pw = new PrintWriter(consoleOutputStream, true);
        // Just as a demo, we will generate some tick messages to go to the newly created console
        Timer timer = new Timer();
        final TimerTask timerTask = new TimerTask() {
            private int tick = 0;

            @Override
            public void run() {
                tick++;
                pw.printf("tick %d\n", tick);
                // Log it too, so we can verify that the ticks stop when the console
                // is closed.
                logger.info(String.format("tick %d\n", tick));
            }
        };
        timer.scheduleAtFixedRate(timerTask, 1000, 1000);
        // If we don't arrange for the timer to be cancelled when the console is closed, it will 
        // run forever. So we attach a listener to detect when the console is closed, and when it is
        // fired cancel the timer task.
        consoleService.getPageContextForConsole(console).addPageListener(new PageListener() {
            @Override
            public void pageChanged(PageEvent pe) {
                if (pe.getID() == PageEvent.PAGECLOSED) {
                    timerTask.cancel();
                    logger.info("Console closed, timertask canceled");
                }
            }
        });
    }

    private void readInputFromConsole(final Console console) {
        // Get an input stream for reading the users input.
        String initialPrompt = "CCS> ";
        final ConsoleInputStream inputStream = console.getInputStream(initialPrompt);
        // We need to create a background thread that will wait for input
        Thread thread = new Thread() {
            @Override
            public void run() {
                try {
                    BufferedReader in = new BufferedReader(new InputStreamReader(inputStream));
                    for (;;) {
                        String line = in.readLine();
                        if (line == null) {
                            // Indicates the input stream has been closed as the console has been closed
                            break;
                        }
                        if ("exit".equals(line)) {
                            // Close the console progamatically. A side effect will be that the
                            // ConsoleInputStream will be closed, and the thread will exit due to 
                            // the check above.
                            console.close();
                        } else if (line.startsWith("setPrompt")) {
                            String newPrompt = line.substring(10);
                            inputStream.setOneTimePrompt(newPrompt);
                        } else if (line.startsWith("changePrompt")) {
                            String newPrompt = line.substring(13);
                            inputStream.setPrompt(newPrompt);
                        }
                        logger.log(Level.INFO, "Read line: {0}", line);
                    }
                    logger.info("Exiting since console has been closed");
                } catch (IOException ex) {
                    logger.log(Level.SEVERE, "Error reading from console", ex);
                }
            }
        };
        thread.start();
    }
}
