package org.lsst.ccs.gconsole.plugins.trending.dataselection;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JSeparator;
import javax.swing.ListCellRenderer;
import org.freehep.application.studio.Studio;
import org.freehep.jas.services.PlotPage;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.plugins.trending.LsstTrendingPlugin;
import org.lsst.ccs.gconsole.plugins.trending.TrendPlotter;

/**
 * GUI for selecting data type for trending plots.
 * Provides a way to select time window from a list of presets and to open custom time window
 * creation and preset management dialogs, as well as access the currently selected time window.
 *
 * @author onoprien
 */
final public class DataTypeSelector extends JComboBox {
    
// -- Private parts : ----------------------------------------------------------
       
    private final String CHOOSE_BUTTON = "Choose...";
    private final String PRESETS_BUTTON = "From Saved...";
    private final String PLOT_BUTTON = "From Plot";
    private final int MAX_PRESETS = 10; // Maximum number of presets to include in drop-down list
    
    private final PresetList presetList;
    private volatile DataType selection;
    
    private final CopyOnWriteArrayList<Listener> listeners = new CopyOnWriteArrayList<>();
    

// -- Construction and initialization : ----------------------------------------
    
    /**
     * Constructs an instance.
     * @param application Console application.
     * @param seed Initial value.
     */
    public DataTypeSelector(Studio application, DataType seed) {
        
        presetList = new PresetList(application);
        
        setRenderer(new Renderer(getRenderer()));
        setMaximumRowCount(MAX_PRESETS + 6);
        
        Model model = new Model(presetList);
        setModel(model);
        
        setSelectedDataType(seed);
        selection = model.getSelectedItem();
        
        addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                DataType ts = myModel().getSelectedItem();
                Model model;
                
                switch (ts.getName()) {
                    
                    case CHOOSE_BUTTON: // custom type
                        
                        ts = DataTypeDialog.defineDataType(DataTypeSelector.this, selection);
                        if (ts == null) {
                            setSelectedItem(selection);
                        } else {
                            ts = presetList.insert(ts);
                            model = new Model(presetList);
                            model.setSelectedItem(ts);
                            setModel(model);
                            setSelection(ts);
                        }
                        break;
                        
                    case PRESETS_BUTTON: // one of the presets has been selected through presets dialog
                        
                        ts = PresetsDialog.managePresets(DataTypeSelector.this, presetList);
                        if (ts == null) ts = selection;
                        ts.touch();
                        model = new Model(presetList);
                        int index = model.getIndexOf(ts);
                        if (index == -1) index = 1;
                        setSelection(model.getElementAt(index));
                        model.setSelectedItem(selection);
                        setModel(model);
                        break;
                        
                    case PLOT_BUTTON: // fetch type from currently selected plot
                        
                        try {
                            PlotPage page = LsstTrendingPlugin.getSelectedPage();
                            TrendPlotter plot = (TrendPlotter) page.currentRegion().currentPlot();
                            ts = presetList.insert(plot.getDataType());
                            model = new Model(presetList);
                            model.setSelectedItem(ts);
                            setModel(model);
                            setSelection(ts);
                        } catch (Exception x) {
                            Console.getConsole().error("There is no currently selected plot.");
                            setSelectedItem(selection);
                        }
                        break;
                        
                    default: // one of the presets has been selected from drop-down list
                        
                        ts.touch();
                        setSelection(ts);
                }
            }
        });
    }
    
// -- Getters and setters : ----------------------------------------------------
    
    /** Returns currently selected time window. */
    public DataType getSelectedDataType() {
        return selection;
    }
   
    /**
     * Sets selected time window.
     * If the specified window has a name, the window should already be present in the list of presets.
     * 
     * @param type The time window to be selected.
     */
    public void setSelectedDataType(DataType type) {
        if (type == null) {
            myModel().setSelectedItem(DataType.DEFAULT);
            setSelection(DataType.DEFAULT);
        } else {
            type = presetList.insert(type);
            int i = myModel().getIndexOf(type);
            if (i >= 0) {
                myModel().setSelectedItem(type);
            } else {
                type.touch();
                Model model = new Model(presetList);
                model.setSelectedItem(type);
                setModel(model);
            }
            setSelection(type);
        }
    }
    
    /** 
     * Returns {@code PresetList} instance that maintains a list of time 
     * window presets currently known to the application.
     */
    public PresetList getPresetList() {
        return presetList;
    }
    
    
// -- Handling listeners : -----------------------------------------------------
    
    public interface Listener {
        void selectionChanged(Event event);
    }
    
    static public class Event extends EventObject {

        private final DataType timeWindow;

        public Event(Object source, DataType timeWindow) {
            super(source);
            this.timeWindow =  timeWindow;
        }

        public DataType getTimeWindow() {
            return timeWindow;
        }
        
    }
    
    public void addListener(Listener listener) {
        listeners.add(listener);
    }
    
    public void removeListener(Listener listener) {
        listeners.remove(listener);
    }
    
    public void removeAllListeners() {
        listeners.clear();
    }
    
    private void setSelection(DataType type) {
        if (!Objects.equals(selection, type)) {
            selection = type;
            Event event = new Event(this, selection);
            listeners.forEach(listener -> listener.selectionChanged(event));
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private Model myModel() {
        return (Model) super.getModel();
    }


// -- Custom renderer : --------------------------------------------------------
    
    private class Renderer implements ListCellRenderer {

        private final ListCellRenderer horse;
        private final JPanel separatorPanel = new JPanel(new BorderLayout());
        private final JSeparator separator = new JSeparator();

        Renderer(ListCellRenderer renderer) {
            horse = renderer;
        }

        @Override
        public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            Component comp = horse.getListCellRendererComponent(list, ((DataType)value).getName(), index, isSelected, cellHasFocus);
            if (index == 0 || index == 2 || index == DataTypeSelector.this.getItemCount()-3) {
                separatorPanel.removeAll();
                separatorPanel.add(comp, BorderLayout.CENTER);
                separatorPanel.add(separator, BorderLayout.SOUTH);
                comp = separatorPanel;
//            } else if (index > 4 && index == DataTypeSelector.this.getItemCount()-1) {
//                separatorPanel.removeAll();
//                separatorPanel.add(comp, BorderLayout.CENTER);
//                separatorPanel.add(separator, BorderLayout.NORTH);
//                comp = separatorPanel;
            }
            return comp;
        }

    }
    
// -- Custom model : -----------------------------------------------------------
    
    private class Model extends DefaultComboBoxModel<DataType> {
        
        Model(PresetList tsList) {

            // Add element for launching type construction dialog :
            
            addElement(new DataType(CHOOSE_BUTTON, false, 0, false));

            // Add standard ranges :
            
            addElement(DataType.RAW);
            addElement(DataType.DEFAULT);

            // Add custom ranges :
            
            List<DataType> customRanges = tsList.getRecent(MAX_PRESETS);

            for (DataType ts : customRanges) {
                addElement(ts);
            }
            
            // Add element for copying settings from currently selected plot: 
            
            addElement(new DataType(PLOT_BUTTON, false, 0, false));

            // Add element for launching time range presets dialog :
            
            addElement(new DataType(PRESETS_BUTTON, false, 0, false));
        }
    
        @Override
        public DataType getSelectedItem() {
            return (DataType) super.getSelectedItem();
        }
        
    }
    
}
