package org.lsst.ccs.gconsole.plugins.trending;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executor;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.gconsole.plugins.trending.TrendData.MetaValue;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindowSelector;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.services.rest.LsstRestService;
import org.lsst.ccs.localdb.statusdb.server.AlertEvent;
import org.lsst.ccs.localdb.statusdb.server.AlertInfo;
import org.lsst.ccs.localdb.statusdb.server.ChannelMetaData;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.StateChange;
import org.lsst.ccs.localdb.statusdb.server.StateInfo;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.statusdb.server.TrendingData.DataValue;
import org.lsst.ccs.localdb.statusdb.server.TrendingResult;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Source of trending data based on a REST server. Provides getters for retrieving the list
 * of known trending channels and the data for specified channels.
 * Public methods of this class can be called on any thread.
 * <p>
 * This sources watches for external events that can change the list of known channels,
 * and notifies listeners on EDT when this happens.
 *
 * @author onoprien
 */
public class RestSource {
    
// -- Fields : -----------------------------------------------------------------
    
    static public final String VALUE_KEY = "value";
    
    static final String ALERT = "alert";
    static final String STATE = "state";
    static final String STRING = "string";
    static final String BOOL = "boolean";
    static final String DOUBLE = "double";
    static final String ARRAY = "array";
    static final String UNKNOWN = "";
    
    static final String[] BOOL_VALUES = {"false", "true"};
    static final String[] ALERT_VALUES = {"NOMINAL", "WARNING", "ALARM"};
    
    static final int MIN_BEGIN = 3600 * 24 * 7;  // one week in seconds
    
    static final String FLAVOR_RAW = "raw";
    static final String FLAVOR_STAT = "stat";
     
    static private final Pattern STATE_PATH = Pattern.compile("(?<agent>[\\w\\-]+)"+ AgentChannel.MARK_STATE +"(?:(?<component>[\\w\\-/]*)/)?(?<"+ STATE +">[\\w\\-]+)");
    static private final Pattern ALERT_PATH = Pattern.compile("(?<agent>[\\w\\-]+)"+ AgentChannel.MARK_ALERT +"(?<"+ ALERT +">[\\w\\-/]+)");
    static private final Pattern LONG = Pattern.compile("[\\d]+");
    
    private final LsstTrendingPlugin plugin;
    private final TrendingPreferences pref;
    private final LsstRestService restService;
    private final Executor dataFetcher;
    private final Logger logger;
    
    private final Map<String,String> channels = new LinkedHashMap<>(); // path -> channel type
    private volatile int begin;
    static private final Map<String,Map<String,String[]>> stateValues = new ConcurrentHashMap<>(); // agent -> {state -> values}. Only for previously used states.
    
    private final CopyOnWriteArrayList<ChangeListener> listeners = new CopyOnWriteArrayList<>();
    
    private final TimeWindowSelector.Listener timeListener;
    private final ChangeListener restListener;
    

// -- Life cycle : -------------------------------------------------------------
    
//<editor-fold defaultstate="collapsed">
    /** Real constructor. */
    RestSource(LsstTrendingPlugin plugin) {
        
        this.plugin = plugin;
        pref = plugin.getPreferences();
        restService = (LsstRestService) plugin.getConsole().getConsoleLookup().lookup(LsstRestService.class);
        dataFetcher = plugin.getExecutor();
        logger = plugin.getConsole().getLogger();
        
        timeListener = event -> {
            onTimeWindowChange(event.getTimeWindow());
        };
        restListener = event -> {
            onRestServerChange();
        };
    }
    
    /** Testing-only constructor. */
    RestSource(LsstRestService restService) {
        plugin =null;
        pref = null;
        this.restService = restService;
        dataFetcher = r -> r.run();
        logger = Logger.getLogger("test");
        timeListener = null;
        restListener = null;
    }
    
    /**  Initialization. Called on EDT. */
    void start() {
        plugin.getTimeWindowSelector().addListener(timeListener);
        restService.addListener(restListener);
        onTimeWindowChange(plugin.getSelectedTimeWindow());
    }
    
    /** Shutdown. Called on EDT. */
    void stop() {
        TimeWindowSelector ts = plugin.getTimeWindowSelector();
        if (ts != null) {
            ts.removeListener(timeListener);
        }
        restService.removeListener(restListener);
    }
//</editor-fold>

// -- Updating this source and notifying listeners : ---------------------------
    
//<editor-fold defaultstate="collapsed">
    /**
     * Triggers refreshing of the list of channels by retrieving them from the REST server.
     * Returns immediately without waiting for the refreshing to complete.
     */
    public void refresh() {
        dataFetcher.execute(this::refreshChannelList);
    }
    
    private void onRestServerChange() {
        refresh();
    }
    
    private void onTimeWindowChange(TimeWindow timeWindow) {
        int time;
        if (timeWindow == null) {
            time = MIN_BEGIN;
        } else {
            time = (int) Math.max(MIN_BEGIN, (System.currentTimeMillis() - timeWindow.getLowerEdge())/1000L + 3600L);
        }
        if (time != begin) {
            begin = time;
            refresh();
        }
    }
    
    private void refreshChannelList() {
        List<DataChannel> dataChannels;
        try {
            DataChannel.DataChannelList channelList = restService.getTrendingList(begin);
            if (channelList == null || channelList.list == null || channelList.list.isEmpty()) {
                dataChannels = Collections.emptyList();
            } else {
                dataChannels = channelList.list;
            }
        } catch (RuntimeException e) {
            logger.warn("Unable to retrieve the list of channels from the REST server.");
            dataChannels = Collections.emptyList();
        }
        List<DataChannel> stateChannels;
        try {
            DataChannel.DataChannelList channelList = restService.getStateList(begin);
            if (channelList == null || channelList.list == null || channelList.list.isEmpty()) {
                stateChannels = Collections.emptyList();
            } else {
                stateChannels = channelList.list;
            }
        } catch (RuntimeException e) {
            logger.warn("Unable to retrieve the list of states from the REST server.");
            stateChannels = Collections.emptyList();
        }
        List<AlertInfo> alertChannels;
        try {
            AlertInfo.AlertInfoList channelList = restService.getAlertList(begin);
            if (channelList == null || channelList.list == null || channelList.list.isEmpty()) {
                alertChannels = Collections.emptyList();
            } else {
                alertChannels = channelList.list;
            }
        } catch (RuntimeException e) {
            logger.warn("Unable to retrieve the list of alerts from the REST server.");
            alertChannels = Collections.emptyList();
        }
        synchronized (channels) {
            if (!(channels.size() == dataChannels.size() + stateChannels.size() + alertChannels.size())) {
                channels.clear();
                
                Set<String> agents = new HashSet<>(128);
                dataChannels.forEach(c -> {
                    String path = String.join("/", c.getPath());
                    String type = convertType(c.getMetadata().get("type"));
                    channels.put(path, type);
                    agents.add(c.getPath()[0]);
                });
                
                Map<String,String[]> cache = new HashMap<>();
                Map<String,Map<String,String[]>> agent2values = new HashMap<>();
                String[] S0 = new String[0];
                stateChannels.forEach(c -> {
                    String[] ss = c.getPath();
                    String agent = ss[0];
                    String path = agent + AgentChannel.MARK_STATE + String.join("/", Arrays.asList(ss).subList(1, ss.length));
                    String type = c.getMetadata().get("type");
                    if (type != null && type.contains(".")) {
                        String[] values = cache.get(type);
                        if (values == null) {
                            try {
                                Class<?> cl = Class.forName(type);
                                if (cl.isEnum()) {
                                    values = Arrays.stream(cl.getEnumConstants()).map(e -> e.toString()).collect(Collectors.toList()).toArray(S0);
                                }
                            } catch (ClassNotFoundException e) {
                                cache.put(type, S0);
                            }
                        }
                        if (values != null && values != S0) {
                            Map<String,String[]> state2values = agent2values.get(agent);
                            if (state2values == null) {
                                state2values = Collections.synchronizedMap(new HashMap<>());
                                agent2values.put(agent, state2values);
                            }
                            String state = ss[ss.length-1];
                            state2values.putIfAbsent(state, values);
                        }
                    }
                    channels.put(path, STATE);
                });
                agent2values.forEach((agent, state2values) -> {
                    Map<String,String[]> old = stateValues.get(agent);
                    if (old == null) {
                        stateValues.put(agent, state2values);
                    } else {
                        state2values.forEach((state, values) -> {
                            old.putIfAbsent(state, values);
                        });
                    }
                });
                
                alertChannels.forEach(c -> {
                    String agent = c.getSubsystemName();
                    if (agents.contains(agent)) {
                        String alertID = c.getAlertId();
                        long id = c.getId();
                        String path = agent + AgentChannel.MARK_ALERT + alertID;
                        channels.put(path, Long.toString(id));
                    }
                });
                
                fireEvent();
            }
        }
    }
    
    /**
     * Registers a listener to be notified if the list of channels available from this trending source changes.
     * This method can be called on any thread. The listeners are notified on the EDT.
     * 
     * @param listener Listener to be added.
     */
    public void addListener(ChangeListener listener) {
        listeners.add(listener);
    }
    
    /**
     * Removes a listener.
     * @param listener Listener to be removed.
     */
    public void removeListener(ChangeListener listener) {
        listeners.remove(listener);
    }
    
    /**
     * Removes all listeners.
     */
    public void removeAllListeners() {
        listeners.clear();
    }
    
    /**
     * Notifies listeners of changes in this source.
     * Can be called on any thread. Listeners are notified on EDT.
     */
    protected void fireEvent() {
        SwingUtilities.invokeLater(() -> {
            ChangeEvent event = new ChangeEvent(RestSource.this);
            listeners.forEach(listener -> {
                try {
                    listener.stateChanged(event);
                } catch (RuntimeException x) {
                    if (logger != null) {
                        logger.error(listener +" threw an exception while processing RestSource event", x);
                    }
                }
            });
        });
    }
//</editor-fold>"

// -- Retrieving data : --------------------------------------------------------

    /**
     * Returns the list of known data channel paths.
     * Calling this method does not retrieve data from the REST server, the returned
     * list reflects the result of the last refresh.
     * 
     * @return List of channel paths.
     */
    public List<String> getChannels() {
        synchronized (channels) {
            return new ArrayList<>(channels.keySet());
        }
    }
    
    /**
     * Retrieves from the REST server and processes historical data for the specified trending channel.
     * If the supplied trends contain previously retrieved data, some of that data can be re-used, reducing the
     * amount of data that needs to be retrieved from the REST server.
     * This method blocks until all requested data have been received. 
     * 
     * @param trends List of trends for which the data has to be retrieved.
     * @param begin Beginning of time window.
     * @param end End of time window
     * @param raw {@code TRUE} or {@code FALSE} depending on whether raw or statistical data should be retrieved;
     *            if {@code null}, current application preferences will be consulted.
     * @param nBins Approximate desired number of bins in the resulting dataset;
     *             if {@code null}, current application preferences will be consulted.
     * @return List of newly created datasets, in the {@code trends} order.
     *         Note that this list will never be {@code null}, but individual elements
     *         can be {@code null} if data retrieval for corresponding trends fails for any reason.
     */
    public ArrayList<TrendData> get(List<Trend> trends, long begin, long end, Boolean raw, Integer nBins) {
        
        // Check preferences if necessary and set query parameters
        
        nBins = nBins == null ? pref.getNBins() : nBins;
        raw = raw == null ? pref.isUseRawData() : raw;
        boolean separateStrings = ! (raw && nBins == 0);
        
        // Define groups that require separate queries
        
        Map<String,String> filter = new HashMap<>();
        filter.put(ALERT, ALERT);
        filter.put(STATE, STATE);
        if (separateStrings) {
            filter.put(STRING, STRING);
            filter.put(BOOL, STRING);
        }
        
        // Separate trends into groups
        
        LinkedHashMap<String,List<Trend>> groupTrends = new LinkedHashMap<>(2*trends.size());
        List<String> groups = new ArrayList<>(trends.size());
        for (Trend trend : trends) {
            String type = getType(trend);  // this is where Trend stores its type and values[] if seen here for the first time
            String group = filter.getOrDefault(type, DOUBLE);
            List<Trend> tt = groupTrends.get(group);
            if (tt == null) {
                tt = new ArrayList<>(trends.size());
                groupTrends.put(group, tt);
            }
            tt.add(trend);
            groups.add(group);
        }
        
        // Get data for each group
        
        LinkedHashMap<String,List<TrendData>> groupOut = new LinkedHashMap<>(2*trends.size());
        for (Map.Entry<String,List<Trend>> e : groupTrends.entrySet()) {
            groupOut.put(e.getKey(), getGroup(e.getValue(), begin, end, raw, nBins));
        }
        
        // Combine results for all types into a single list
        
        ArrayList<TrendData> out = new ArrayList<>(trends.size());
        LinkedHashMap<String,Iterator<TrendData>> it = new LinkedHashMap<>(2*trends.size());
        for (Map.Entry<String,List<TrendData>> e : groupOut.entrySet()) {
            it.put(e.getKey(), e.getValue().iterator());
        }
        for (String group : groups) {
            out.add(it.get(group).next());
        }
        return out;
    }
    
    List<TrendData> getGroup(List<Trend> trends, long begin, long end, boolean raw, int nBins) {
        
        if (trends.isEmpty()) return Collections.emptyList();
        
        boolean cache = true;
        long beginCache = 0;
        long endCache = Long.MAX_VALUE;
        for (Trend trend : trends) {
            TrendData data = trend.getData();
            if (data == null) {
                cache = false;
            } else {
                switch (data.getType()) {
                    case DOUBLE:
                        cache = cache && data.isRawUnbinned() && raw && nBins == 0;
                        break;
                    case STATE:
                    case ALERT:
                        break;
                    case STRING:
                    case BOOL:
                        cache = cache && ((data.isRawUnbinned() && raw && nBins == 0) || (data.getBins() == nBins));
                        break;
                    default:
                        cache = false;
                }
            }
            if (!cache) break;
            beginCache = Math.max(beginCache, data.getLowT());
            endCache = Math.min(endCache, data.getHighT());
        }
        cache = cache && (beginCache < endCache);
        
        long beginQuery = begin;
        long endQuery = end;
        if (cache) {
            if (beginCache <= begin && endCache > begin) {
                beginQuery = endCache;
            }
            if (beginCache < end && endCache >= end) {
                endQuery = beginCache;
            }
        }
        
        int nTrends = trends.size();
        Result[] r = beginQuery < endQuery ? fetch(trends, beginQuery, endQuery, raw, nBins) : new Result[nTrends];
        ArrayList<TrendData> out = new ArrayList<>(nTrends);
        for (int i = 0; i < nTrends; i++) {
            out.add(merge(trends.get(i), r[i], begin, end, raw, nBins));
        }
        return out;
    }
    
    Result[] fetch(List<Trend> trends, long begin, long end, boolean raw, int nBins) {
        if (trends.isEmpty()) return new Result[0];
        String type = getType(trends.get(0));
        switch (type) {
            case STATE:
                return fetchState(trends, begin, end);
            case ALERT:
                return fetchAlert(trends, begin, end);
            default:
                return fetchTrending(trends, begin, end, raw, nBins);
        }
    }
    
    Result[] fetchTrending(List<Trend> trends, long begin, long end, boolean raw, int nBins) {        
        String[] paths = trends.stream().map(trend -> trend.getDescriptor().getPath()).collect(Collectors.toList()).toArray(new String[0]);
        TrendingResult[] out;
        if (raw && nBins == 0) {
            out = restService.getTrendingData(begin, end, "raw", nBins, paths);
        } else {
            String type = getType(trends.get(0));
            switch (type) {
                case STRING:
                case BOOL:
                    out = restService.getTrendingData(begin, end, "raw", 0, paths);
                    break;
                default:
                    String flavor;
                    if (nBins > 0) {
                        flavor = raw ? null : "stat";
                    } else {
                        flavor = raw ? "raw" : "stat";
                    }
                    out = restService.getTrendingData(begin, end, flavor, nBins, paths);
            }
        }
        return Result.make(paths, begin, end, raw, nBins, out);
    }
    
    @SuppressWarnings("unchecked")
    Result[] fetchState(List<Trend> trends, long begin, long end) {
        String[] paths = trends.stream().map(trend -> trend.getDescriptor().getPath()).collect(Collectors.toList()).toArray(new String[0]);

        Map<String,String> path2agent = new LinkedHashMap<>(trends.size()*2);
        Map<String,StateQueryInfo> agent2info = new LinkedHashMap<>(trends.size()*2);
        for (Trend trend : trends) {
            String path = trend.getDescriptor().getPath();
            StateID id = StateID.parse(path);
            if (id != null) {
                path2agent.put(path, id.agent);
                StateQueryInfo info = agent2info.get(id.agent);
                if (info == null) {
                    info = new StateQueryInfo(id.agent);
                    agent2info.put(id.agent, info);
                }
                info.components.add(id.component);
                info.states.add(id.state);
            }
        }
        Map<String,List<StateChange>> agent2result = new HashMap<>(agent2info.size()*2); 
        for (StateQueryInfo info : agent2info.values()) {
            agent2result.put(info.agent, restService.getStateTransitions(begin, end, info.agent, info.components, info.states));
        }
        List<StateChange>[] out = (List<StateChange>[]) new List[paths.length];
        for (int i=0; i<paths.length; i++) {
            out[i] = agent2result.get(path2agent.get(paths[i]));
        }
        return Result.make(paths, begin, end, out);
    }
    
    Result[] fetchAlert(List<Trend> trends, long begin, long end) {
        int n = trends.size();
        String[] paths = new String[n];
        String[] ids = new String[n];
        Iterator<Trend> it = trends.iterator();
        for (int i=0; i<n; i++) {
            Trend trend = it.next();
            paths[i] = trend.getDescriptor().getPath();
            ids[i] = channels.getOrDefault(paths[i], "NO_SUCH_ALERT");
        }
        List<AlertEvent> out = restService.geAlerts(begin, end, ids);
        return Result.make(paths, begin, end, out);
    }
    
    /**
     * Produces new data set by merging data from cache and new query.
     * 
     * @param trend Trend with previously fetched data.
     * @param result Result of new query (may be {@code null}).
     * @param begin Start of time range for new dataset.
     * @param end End of time range for new dataset.
     * @param raw Requested kind of data for the new dataset.
     * @param nBins Requested binning for the new dataset.
     * @return New dataset.
     */
    TrendData merge(Trend trend, Result result, long begin, long end, boolean raw, int nBins) {
        
        String type = getType(trend);
        if (result != null) {
            String resultType = result.getType();
            if (resultType != null && !UNKNOWN.equals(resultType) && !resultType.equals(type)) {
                trend.setType(resultType);
                type = resultType;
            }
        }
        
        long beginQuery = result == null ? Long.MAX_VALUE : result.begin;
        long endQuery = result == null ? 0L : result.end;
        
        Data cache = null;
        Data query = null;
        if (beginQuery <= begin) {
            if (endQuery >= end) { // query covers entire range
                if (DOUBLE.equals(type)) {
                    return mergeDouble(null, result, begin, end, raw, nBins);
                } else {
                    query = Data.make(result, begin, end, raw, nBins);
                    return stitch(query, null, begin, end, raw, nBins, trend);
                }
            } else if (endQuery >= begin) { // query covers beginning of range; no machinery do do this for DOUBLE
                query = Data.make(result, begin, endQuery, raw, nBins);
                cache = Data.make(trend, endQuery, end, raw, nBins);
                return stitch(query, cache, begin, end, raw, nBins, trend);
            } else { // query before range, then cache should cover entire range; this should not happen
                if (DOUBLE.equals(type)) {
                    return cut(trend.getData(), begin, end);
                } else {
                    cache = Data.make(trend, begin, end, raw, nBins);
                    return stitch(cache, null, begin, end, raw, nBins, trend);
                }
            }
        } else if (beginQuery < end && endQuery >= end) { // query covers end of range
            if (DOUBLE.equals(type)) {
                return mergeDouble(trend, result, begin, end, raw, nBins);
            } else {
                cache = Data.make(trend, begin, beginQuery, raw, nBins);
                query = Data.make(result, beginQuery, end, raw, nBins);
                return stitch(cache, query, begin, end, raw, nBins, trend);
            }
        } else { // cache covers entire range
            if (DOUBLE.equals(type)) {
                return cut(trend.getData(), begin, end);
            } else {
                cache = Data.make(trend, begin, end, raw, nBins);
                return stitch(cache, null, begin, end, raw, nBins, trend);
            }
        }
    }
    
    /**
     * Creates a new dataset for the specified time window using data from an existing dataset and a result of a new query.
     * Assumptions:<ul>
     * <li>If {@code cachedData} is not {@code null}, its time range contains {@code begin} but does not contain {@code end}.
     * <li>If {@code newData} is not {@code null}, it does not contain points before the end of {@code cachedData} range.
     * </ul>
     * 
     * @param cachedData Existing dataset.
     * @param newData Result of a new query.
     * @param begin Start of the time range of the new dataset.
     * @param end End of the time range of the new dataset.
     * @return New dataset.
     */
    TrendData mergeDouble(Trend trend, Result queryData, long begin, long end, boolean raw, int nBins) {
        
        TrendData cachedData = trend == null ? null : trend.getData();
        
        TrendingResult newData; // FIXME: unnecessary if
        boolean rawUnbinned;    // queryData == null
        if (queryData == null) {
            newData = null;
            rawUnbinned = cachedData.isRawUnbinned();
        } else {
            newData = (TrendingResult) queryData.data;
            rawUnbinned = raw && nBins == 0;
        }

        long[] times;
        Map<String, double[]> values = new HashMap<>();
        Map<String, ArrayList<MetaValue>> meta = new HashMap<>();
        
        // inspect cached data, set index to the first data point to be re-used, "oldN" to number of points to re-use
        
        int oldN = 0;
        int index = 0;
        long[] oldTime = null;
        if (cachedData != null) {
            oldTime = cachedData.getTimes();
            index = Arrays.binarySearch(oldTime, begin);
            if (index < 0) {
                index = -index -1;
            }
            oldN = oldTime.length - index;
        }
        
        // set "newN" to the number of points in the new data, "n" to the total number of points
        
        int newN = 0;
        TrendingData[] dataArray = null;
        if (newData != null) {
            dataArray = newData.getTrendingDataArray();
            if (dataArray != null) {
                newN = dataArray.length;
            }
        }       
        int n = oldN + newN;
        
        // copy cached on-point data to new arrays, set "index" to next index
        
        if (oldN == 0) {
            times = new long[n];
        } else {
            times = Arrays.copyOfRange(oldTime, index, index + n);
            for (String key : cachedData.getOnPointKeys()) {
                values.put(key, Arrays.copyOfRange(cachedData.getValues(key), index, index + n));
            }
        }
        index = oldN;
        
        // add points from new query  // FIXME: if this was used for raw data only, I could stop looking up by key
        
        if (newN > 0) {
            for (TrendingData td : dataArray) {
                for (DataValue dv : td.getDatavalue()) {
                    String key = dv.getName();
                    double[] vv = values.get(key);
                    if (vv == null) {
                        vv = new double[n];
                        values.put(key, vv);
                    }
                    vv[index] = dv.getValue();
                }
                times[index++] = td.getAxisvalue().getValue();
            }
        }
        
        // fill metadata lists from cached data
        
        if (cachedData != null) {
            Map<String, ArrayList<MetaValue>> cachedMeta = cachedData.getOffPointMetadata();
            for (Map.Entry<String, ArrayList<MetaValue>> e : cachedMeta.entrySet()) {
                String key = e.getKey();
                ArrayList<MetaValue> newList = new ArrayList<>();
                for (MetaValue mp : e.getValue()) {
                    MetaValue mv = MetaValue.trim(mp, begin, end);
                    if (mv != null) newList.add(mv);
                }
                if (!newList.isEmpty()) {
                    meta.put(key, newList);
                }
            }
        }
        
        // add metadata from new query
        
        if (newData != null) {
            List<ChannelMetaData> metaList = newData.getChannelMetadata();
            if (metaList != null) {
                for (ChannelMetaData md : metaList) {
                    String key = md.getName();
                    ArrayList<MetaValue> mvList = meta.get(key);
                    if (mvList == null) {
                        MetaValue mv = MetaValue.trim(md, begin, end);
                        if (mv != null) {
                            mvList = new ArrayList<>();
                            mvList.add(mv);
                            meta.put(key, mvList);
                        }
                    } else {
                        MetaValue last = mvList.get(mvList.size() - 1);
                        MetaValue mv = MetaValue.trim(md, last.getStop(), end);
                        if (mv != null) {
                            MetaValue merge = MetaValue.merge(last, mv);
                            if (merge == null) {
                                mvList.add(mv);
                            } else {
                                mvList.set(mvList.size() - 1, merge);
                            }
                        }
                        

                    }
                }
            }
        }
        
        // create and return new dataset
        
        if (values.isEmpty()) {
            values = Collections.emptyMap();
        } else if (values.size() == 1) {
            Map.Entry<String, double[]> e = values.entrySet().iterator().next();
            values = Collections.singletonMap(e.getKey(), e.getValue());
        }
        if (meta.isEmpty()) {
            meta = Collections.emptyMap();
        } else {
            meta.values().forEach(list -> list.trimToSize());
        }
        TrendData trd =  new TrendData(times, values, meta, new long[] {begin, end}, raw, nBins);
//
//        Diagnostics for LSSTCCS-2253:
//
//        printTR(newData, begin, end);
//        printTD(trd);
//        
        return trd;
    }
    
    /**
     * Creates a new dataset by combining two {@code Data}.
     * 
     * @param d1 Chronologically first {@code Data}.
     * @param d2 Chronologically second {@code Data}, may be {@code null}.
     * @param begin Beginning of desired time range for the new dataset.
     * @param end End of desired time range for the new dataset.
     * @param raw Data kind specified in client request.
     * @param nBins Binning specified in client request.
     * @param trend Trend.
     * @return  Newly created dataset.
     */
    TrendData stitch(Data d1, Data d2, long begin, long end, boolean raw, int nBins, Trend trend) {
        
        if (d1 == null) return null;
        
        String[] names = null;
        String type = getType(trend);
        switch (type) { // FIXME
            case BOOL:
            case STRING:
            case STATE:
            case ALERT:
                names = Data.mergeNames(trend.getValues(), d1, d2);
                break;
            default:
                throw new IllegalArgumentException("cannot stitch data of type "+ type);
        }
        
        boolean endpointsAllowed = !(raw && nBins == 0) || STATE.equals(type) || ALERT.equals(type);
        int n = 0;
        boolean needsBefore = endpointsAllowed && valid(d1.beforeValue()) && (!valid(d1.firstTime()) || (d1.firstTime() > begin + Math.max((end-begin)/50, 1)));
        needsBefore = needsBefore || (d1.size() == 0 && d2 != null && valid(d2.beforeValue()) && (!valid(d2.firstTime()) || (d2.firstTime() > begin + Math.max((end-begin)/50, 1))));
        if (needsBefore) n++;
        n += d1.size();
        Data lastData = d1;
        boolean needsBetween = false;
        if (d2 != null) {
            needsBetween = ((STATE.equals(type) || ALERT.equals(type)) && d2.size()>0 && valid(d1.afterValue()) && !equal(d1.afterValue(), d2.firstValue()));
            if (needsBetween) n++;
            lastData = d2;
            n += d2.size();
        }
        boolean needsAfter = endpointsAllowed && valid(lastData.afterValue()) && (!valid(lastData.lastTime()) || (lastData.lastTime() < end - Math.max((end-begin)/50, 1)));
        needsAfter = needsAfter || (endpointsAllowed && d2 != null && d2.size() == 0 && valid(d1.afterValue()) && (!valid(d1.lastTime()) || (d1.lastTime() < end - Math.max((end-begin)/50, 1))));
        if (needsAfter) n++;
        
        long[] times = new long[n];
        double[] values = new double[n];
        int index = 0;
        if (needsBefore) {
            times[index] = begin;
            values[index++] = valid(d1.beforeValue()) ? d1.beforeValue() : d2.beforeValue();
        }
        while (d1.hasNext()) {
            d1.next();
            times[index] = d1.time();
            values[index++] = d1.value();
        }
        if (d2 != null) {
            if (needsBetween) {
                times[index] = d2.firstTime();
                values[index++] = d1.afterValue();
            }
            while (d2.hasNext()) {
                d2.next();
                times[index] = d2.time();
                values[index++] = d2.value();
            }
        }
        if (needsAfter) {
            times[index] = end;
            values[index++] = valid(lastData.afterValue()) ? lastData.afterValue() : d1.afterValue();
        }
        
        return new TrendData(times, values, new long[] {begin, end}, names, getType(trend), raw, nBins);
    }
    
    /**
     * Creates a dataset for a portion of the time range covered by the original raw trending numerical data dataset.
     * 
     * @param data Original dataset.
     * @param begin Start of the time range of the new dataset.
     * @param end End of the time range of the new dataset.
     * @return New dataset. {@code null} if the original dataset is {@code null}, or if the
     *         original dataset does not cover the entire range from {@code begin} to {@code end}.
     */
    static TrendData cut(TrendData data, long begin, long end) {
        
        if (data == null || end <= begin) return null;
        long[] dataTimeRange = data.getTimeRange();
        if (dataTimeRange[0] > begin || dataTimeRange[1] < end) return null;

        long[] times;
        Map<String, double[]> values = new HashMap<>();
        Map<String, ArrayList<MetaValue>> meta = new HashMap<>();
        
        // data and on-point metadata
        
        long[] oldTime = data.getTimes();
        int start = Arrays.binarySearch(oldTime, begin);
        if (start < 0) {
            start = -start - 1;
        }
        int stop = Arrays.binarySearch(oldTime, end);
        if (stop < 0) {
            stop = -stop - 1;
        } else {
            stop++;
        }
        if (stop > start) {
            times = Arrays.copyOfRange(oldTime, start, stop);
            for (String key : data.getOnPointKeys()) {
                values.put(key, Arrays.copyOfRange(data.getValues(key), start, stop));
            }
        } else {
            times = new long[0];
            for (String key : data.getOnPointKeys()) {
                values.put(key, new double[0]);
            }
        }
        
        // off-point metadata
        
        Map<String, ArrayList<MetaValue>> cachedMeta = data.getOffPointMetadata();
        for (Map.Entry<String, ArrayList<MetaValue>> e : cachedMeta.entrySet()) {
            String key = e.getKey();
            ArrayList<MetaValue> newList = new ArrayList<>();
            for (MetaValue mp : e.getValue()) {
                MetaValue mv = MetaValue.trim(mp, begin, end);
                if (mv != null) {
                    newList.add(mv);
                }
            }
            if (!newList.isEmpty()) {
                meta.put(key, newList);
            }
        }
        
        // create and return new dataset
        
        if (values.isEmpty()) {
            values = Collections.emptyMap();
        } else if (values.size() == 1) {
            Map.Entry<String, double[]> e = values.entrySet().iterator().next();
            values = Collections.singletonMap(e.getKey(), e.getValue());
        }
        if (meta.isEmpty()) {
            meta = Collections.emptyMap();
        } else {
            meta.values().forEach(list -> list.trimToSize());
        }
        return new TrendData(times, values, meta, new long[] {begin, end}, true, 0);
    }
    
    
// -- Local classes : ----------------------------------------------------------

    /**
     * Result of a REST server query for a single channel.
     * Contains query parameters and retrieved data.
     */
    static class Result {
        
        // Query parameters:
        
        final String path;
        final long begin, end;
        final boolean raw;
        final int bins;
        
        // Query result:
        
        final Object data; // either TrendingResult or List<StateChange>
        
        // Life cycle:

        Result(String path, long begin, long end, boolean raw, int bins, Object data) {
            this.path = path;
            this.begin = begin;
            this.end = end;
            this.raw = raw;
            this.bins = bins;
            this.data = data;
        }
        
        static Result[] make(String[] paths, long begin, long end, boolean raw, int bins, TrendingResult[] results) {
            int n = paths.length;
            Result[] out = new Result[n];
            for (int i = 0; i < n; i++) {
                out[i] = results == null ? null : new Result(paths[i], begin, end, raw, bins, results[i]);
            }
            return out;
        }
        
        static Result[] make(String[] paths, long begin, long end, List<StateChange>[] results) {
            int n = paths.length;
            Result[] out = new Result[n];
            for (int i = 0; i < n; i++) {
                out[i] = results == null ? null : new Result(paths[i], begin, end, true, 0, results[i]);
            }
            return out;
        }
        
        static Result[] make(String[] paths, long begin, long end, List<AlertEvent> result) {
            int n = paths.length;
            Result[] out = new Result[n];
            if (result != null) {
                for (int i = 0; i < n; i++) {
                    out[i] = new Result(paths[i], begin, end, true, 0, result);
                }
            }
            return out;
        }
        
        // Getters:
        
        String getType() {
            if (data instanceof TrendingResult) {
                TrendingResult tr = (TrendingResult) data;
                List<ChannelMetaData> mdl = tr.getChannelMetadata();
                if (mdl == null) {
                    return UNKNOWN;
                } else {
                    for (ChannelMetaData md : mdl) {
                        if ("type".equals(md.getName())) {
                            return convertType(md.getValue());
                        }
                    }
                    return UNKNOWN;
                }
            } else if (STATE_PATH.matcher(path).matches()) {
                return STATE;
            } else if (ALERT_PATH.matcher(path).matches()) {
                return ALERT;
            } else {
                return UNKNOWN;
            }
        }
        
        TrendingResult getTrendingResult() {
            return (TrendingResult) data;
        }
        
        List<StateChange> getStateChanges() {
            return (List<StateChange>) data;
        }
        
        List<AlertEvent> getAlerts() {
            return (List<AlertEvent>) data;
        }
        
        boolean isRaw() {
            return raw && bins == 0;
        }
        
    }
    
    /** Parses state channel path and holds agent, component, and state parts. */
    static private class StateID {
        
        final String agent;
        final String component;
        final String state;

        public StateID(String agent, String component, String state) {
            this.agent = agent;
            this.component = component;
            this.state = state;
        }
                
        static StateID parse(String path) {
            Matcher m = STATE_PATH.matcher(path);
            if (m.matches()) {
                String comp = m.group("component");
                return new StateID(m.group("agent"), comp == null ? "" : comp, m.group(STATE));
            } else {
                return null;
            }
        }
    }
    
    /** Parses alert channel path and holds agent, alertID parts. */
    static private class AlertID {
        
        final String agent;
        final String alertID;

        public AlertID(String agent, String alertID) {
            this.agent = agent;
            this.alertID = alertID;
        }
                
        static AlertID parse(String path) {
            Matcher m = ALERT_PATH.matcher(path);
            if (m.matches()) {
                return new AlertID(m.group("agent"), m.group(ALERT));
            } else {
                return null;
            }
        }
    }
    
    static private class StateQueryInfo {
        StateQueryInfo(String agent) {
            this.agent = agent;
        }
        final String agent;
        final Set<String> states = new LinkedHashSet<>();
        final Set<String> components = new LinkedHashSet<>();
    }
    
    static private interface Data {
        
        static Data make(Trend trend, long begin, long end, boolean raw, int nBins) {
            return new DataStringCache(trend, begin, end, raw, nBins);
        }
        
        static Data make(Result result, long begin, long end, boolean raw, int nBins) {
            if (result == null) return null;
            switch (result.getType()) {
                case STRING:
                case BOOL:
                case STATE:
                case ALERT:
                    return new DataStringQuery(result, begin, end, raw, nBins);
               default:
                    return null;
            }
        }
        
        static String[] mergeNames(String[] names, Data d1, Data d2) {
            if (names == null) {
                TreeSet<String> all = new TreeSet<>();
                all.addAll(d1.valueSet());
                if (d2 != null) all.addAll(d2.valueSet());
                names = all.toArray(new String[0]);
            }
            d1.setMap(names);
            if (d2 != null) d2.setMap(names);
            return names;
        }
        
        /* Number of points in [begin, end[ */
        int size();
        
        /** Subsequent call to {@code next()} will move the cursor to the first point */
        void rewind();
        
        /** Has more points after the cursor */
        boolean hasNext();
        
        /** Move cursor to the next point */
        void next();
        
        /** Time of current point */
        long time();
        
        /** Value of current point */
        double value();
        
        long begin();
        
        long end();
        
        /** Time of the first point, -1 if this dataset has no points */
        long firstTime();
        
        /** Value of the first point, -1 if this dataset has no points */
        double firstValue();
        
        /** Time of the last point, -1 if this dataset has no points */
        long lastTime();
        
        /** Value of the last point, -1 if this dataset has no points */
        double lastValue();
        
        /** Last value before {@code begin}, if known and either equal to value of first point,
         * or dataset is empty and before == after; -1 otherwise */
        double beforeValue();
        
        /** First value after {@code end}, if known and either equal to value of last point,
         * or dataset is empty and before == after; -1 otherwise */
        double afterValue();
        
        /** Sets map for converting from string to int */
        void setMap(String[] names);
        
        /** Set of string values in this dataset, including before and after */
        Set<String> valueSet();
        
    }
    
    /* @see Data */
    static private final class DataStringCache implements Data {
        
        private long begin, end;
        private int from, to;
        private long[] times;
        private double[] values;
        private double before, after;
        private boolean fixedNames;
        private String[] names;
        
        private int index;
        
        DataStringCache(Trend trend, long begin, long end, boolean raw, int nBins) {
            this.begin = begin;
            this.end = end;
            this.fixedNames = trend.getValues() != null;
            TrendData data = trend.getData();
            if (data != null && data.getTimes().length > 0) {
                times = data.getTimes();
                values = data.getValues();
                names = data.getNames();
                from = Arrays.binarySearch(times, begin);
                if (from < 0) {
                    from = -from - 1;
                } else if (from > 0 && times[from-1] == times[from]) {
                    from--;
                }
                to = Arrays.binarySearch(times, end);
                if (to < 0) {
                    to = -to - 1;
                } else if (to > 0 && times[to-1] == times[to]) {
                    to--;
                }
            }
            index = from - 1;
            String type = trend.getType();
            if (STATE.equals(type) || ALERT.equals(type)) {
                if (from > 0) {
                    before = values[from-1];
                } else if (times.length > 1 && times[0] == times[1]) {
                    before = values[0];
                } else {
                    before = -1;
                }
                after = (to - from) > 0 ? values[to-1] : before;
            } else {
                boolean rawUnbinned = raw && nBins == 0;
                if (rawUnbinned) {
                    before = after = -1;
                } else {
                    before = from > 0 && from < values.length && equal(values[from-1], values[from]) ? values[from] : -1;
                    after = to < values.length && (to-1) >= 0 && equal(values[to-1], values[to]) ? values[to] : -1;
                }
            }
        }

        @Override
        public int size() {
            return to - from;
        }

        @Override
        public void rewind() {
            index = from - 1;
        }

        @Override
        public boolean hasNext() {
            return index < (to - 1);
        }

        @Override
        public void next() {
            index++;
        }

        @Override
        public long time() {
            return times[index];
        }

        @Override
        public double value() {
            return values[index];
        }

        @Override
        public long begin() {
            return begin;
        }

        @Override
        public long end() {
            return end;
        }

        @Override
        public long firstTime() {
            return size() == 0 ? -1 : times[from];
        }

        @Override
        public double firstValue() {
            return size() == 0 ? -1 : values[from];
        }

        @Override
        public long lastTime() {
            return size() == 0 ? -1 : times[to-1];
        }

        @Override
        public double lastValue() {
            return size() == 0 ? -1 : values[to-1];
        }

        @Override
        public double beforeValue() {
            return before;
        }

        @Override
        public double afterValue() {
            return after;
        }

        @Override
        public void setMap(String[] names) {
            if (!fixedNames && !Objects.deepEquals(this.names, names)) {
                HashMap<String,Integer> string2new = new HashMap<>();
                for (int i=0; i<names.length; i++) {
                    string2new.put(names[i], i);
                }
                for (int i=from; i<to; i++) {
                    values[i] = string2new.get(this.names[toInt(values[i])]);
                }
                if (valid(before)) before = string2new.get(this.names[toInt(before)]);
                if (valid(after)) after = string2new.get(this.names[toInt(after)]);
                this.names = names;
            }
        }

        @Override
        public Set<String> valueSet() {
            if (fixedNames) {
                return new LinkedHashSet<>(Arrays.asList(names));
            } else {
                Set<String> out = new LinkedHashSet<>();
                for (int i = from; i<to; i++) {
                    out.add(names[toInt(values[i])]);
                }
                if (valid(before)) out.add(names[toInt(before)]);
                if (valid(after)) out.add(names[toInt(after)]);
                return out;
            }
        }
    }
    
    /* @see Data */
    static private final class DataStringQuery implements Data {
        
        private long begin, end;
        private long[] times; // {0, [0] ..... [size], 0}
        private String[] values; // {before, [0] ..... [size], after}
        private int size;
        private Map<String,Double> str2value;
        
        int index;
        
        DataStringQuery(Result result, long begin, long end, boolean raw, int nBins) {
            this.begin = begin;
            this.end = end;
            str2value = new HashMap<>();
            switch (result.getType()) {
                case STATE:
                    initState(result); break;
                case ALERT:
                    initAlert(result); break;
                default:
                    initString(result, raw, nBins);
            }
            index = 0;
        }
        
        void initString(Result result, boolean raw, int nBins) {
            if (result != null) {
                TrendingData[] tds = result.getTrendingResult().getTrendingDataArray();
                if (tds != null && tds.length > 0) {
                    long gap;
                    if (nBins == 0) {
                        gap = raw ? 0 : Long.MAX_VALUE;
                    } else {
                        gap = (end-begin)/nBins;
                    }
                    times = new long[tds.length +2];
                    values = new String[tds.length +2];
                    String prev = null;
                    for (int i=0; i<tds.length; i++) {
                        TrendingData td = tds[i];
                        long t = td.getAxisvalue().getValue();
                        String s = tds[i].getStrValue(VALUE_KEY);
                        if (s != null) {
                            if (t < begin) {
                                values[0] = s;
                            } else {
                                if (t < end) {
                                    str2value.put(s, null);
                                    if ( gap == 0 || index < 2 || !(s.equals(values[index]) && s.equals(values[index - 1])) || (t-times[index - 1]) > gap ) {
                                        index++;
                                    }
                                    times[index] = t;
                                    values[index] = s;
                                } else {
                                    values[index+1] = s;
                                    break;
                                }
                            }
                        }
                    }
                    size = index;
                    if (!Objects.equals(values[0], values[1])) {
                        values[0] = null;
                    }
                    if (!Objects.equals(values[size], values[size + 1])) {
                        values[size + 1] = null;
                    }
                }
            }
        }
        
        void initState(Result result) {
            List<StateChange> data = result.getStateChanges();
            if (data.isEmpty()) return;
            StateID sid = StateID.parse(result.path);
            Set<String> validValues = getValidStateValues(sid);
            times = new long[2*data.size()+2];
            values = new String[2*data.size()+2];
            index = 0;
            String prev = null;
            boolean skip = true;
            ListIterator<StateChange> it = data.listIterator();
            while (it.hasNext()) {
                StateChange sc = it.next();
                String value = sc.getState(sid.component, sid.state);
                if (value != null && validValues.contains(value)) {
                    long t = sc.getTime();
                    if (!skip || t >= begin) {
                        if (skip) {
                            values[0] = prev;
                            skip = false;
                        }
                        if (t < end) {
                            if (prev != null && !prev.equals(value)) {
                                times[++index] = t;
                                values[index] = prev;
                            }
                            times[++index] = t;
                            values[index] = value;
                        } else {
                            break;
                        }
                    }
                    prev = value;
                }
            }
            size = index++;
            values[index] = prev;
        }
        
        void initAlert(Result result) {
            List<AlertEvent> data = result.getAlerts();
            if (data.isEmpty()) return;
            AlertID sid = AlertID.parse(result.path);
            if (sid == null) return;
            Set<String> validValues = new HashSet<>(Arrays.asList(ALERT_VALUES));
            times = new long[2*data.size()+2];
            values = new String[2*data.size()+2];
            index = 0;
            String prev = null;
            boolean skip = true;
            ListIterator<AlertEvent> it = data.listIterator();
            while (it.hasNext()) {
                AlertEvent sc = it.next();
                String value = sc.getSeverity().toString();
                if (value != null && validValues.contains(value) && sid.alertID.equals(sc.getAlertId())) {
                    long t = sc.getTime();
                    if (!skip || t >= begin) {
                        if (skip) {
                            values[0] = prev;
                            skip = false;
                        }
                        if (t < end) {
                            if (prev != null && !prev.equals(value)) {
                                times[++index] = t;
                                values[index] = prev;
                            }
                            times[++index] = t;
                            values[index] = value;
                        } else {
                            break;
                        }
                    }
                    prev = value;
                }
            }
            size = index++;
            if (size == 0) values[0] = prev;
            values[index] = prev;
        }

        @Override
        public int size() {
            return size;
        }

        @Override
        public void rewind() {
            index = 0;
        }

        @Override
        public boolean hasNext() {
            return index < size;
        }

        @Override
        public void next() {
            index++;
        }

        @Override
        public long time() {
            return times[index];
        }

        @Override
        public double value() {
            return str2value.get(values[index]);
        }

        @Override
        public long begin() {
            return begin;
        }

        @Override
        public long end() {
            return end;
        }

        @Override
        public long firstTime() {
            return size > 0 ? times[1] : -1;
        }

        @Override
        public double firstValue() {
            return size > 0 ? str2value.get(values[1]) : -1;
        }

        @Override
        public long lastTime() {
            return size > 0 ? times[size] : -1;
        }

        @Override
        public double lastValue() {
            return size > 0 ? str2value.get(values[size]) : -1;
        }

        @Override
        public double beforeValue() {
            Double d = values == null || values[0] == null ? -1. : str2value.get(values[0]);
            return d == null ? -1 : d;
        }

        @Override
        public double afterValue() {
            Double d = values == null || values[size+1] == null ? -1. : str2value.get(values[size+1]);
            return d == null ? -1 : d;
        }

        @Override
        public void setMap(String[] names) {
            str2value = new HashMap<>();
            for (int i=0; i<names.length; i++) {
                str2value.put(names[i], (double)i);
            }
        }

        @Override
        public Set<String> valueSet() {
            return str2value.keySet();
        }
        
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    /** Converts channel type returned by REST server into type used by this class. */
    static private String convertType(String rawType) {
        if (rawType == null) return UNKNOWN;
        switch (rawType) {
            case "java.lang.String":
                return STRING;
            case "trending":
                return UNKNOWN;
            case "boolean":
                return BOOL;
            case "double":
            case "int":
            case "long":
                return DOUBLE;
            case "state":
                return STATE;
            default:
                if (rawType.contains("[]")) {
                    return ARRAY;
                }
        }
        return rawType;
    }
    
    /** Stores type and values[] into Trend if not already there. */
    private String getType(Trend trend) {
        String type = trend.getType();
        if (type == null) {
            String path = trend.getDescriptor().getPath();
            synchronized (channels) {
                type = channels.get(path);
            }
            if (type == null) {
                if (STATE_PATH.matcher(path).matches()) {
                    type = STATE;
                }
            } else {
                if (LONG.matcher(type).matches()) {
                    type = ALERT;
                }
            }
            if (type != null) {
                trend.setType(type);
                String[] values = null;
                switch (type) {
                    case BOOL:
                        values = BOOL_VALUES;
                        break;
                    case STATE:
                        StateID sid = StateID.parse(trend.getDescriptor().getPath());
                        Map<String, String[]> state2values = stateValues.get(sid.agent);
                        if (state2values == null) {
                            StateInfo.StateInfoList infoList = restService.getStateInfo(sid.agent);
                            if (infoList != null) {
                                state2values = new HashMap<>();
                                for (StateInfo info : infoList.stateInfoList) {
                                    String[] vv = new ArrayList<>(info.getStateValues()).toArray(new String[0]);
                                    state2values.put(info.getStateClass(), vv);
                                }
                                values = state2values.get(sid.state);
                                state2values = Collections.synchronizedMap(state2values);
                                stateValues.put(sid.agent, state2values);
                            }
                        } else {
                            values = state2values.get(sid.state);
                            if (values == null) {
                                StateInfo.StateInfoList infoList = restService.getStateInfo(sid.agent);
                                if (infoList != null) {
                                    for (StateInfo info : infoList.stateInfoList) {
                                        String[] vv = new ArrayList<>(info.getStateValues()).toArray(new String[0]);
                                        state2values.put(info.getStateClass(), vv);
                                    }
                                    values = state2values.get(sid.state);
                                }
                            }
                        }
                        break;
                    case ALERT:
                        values = ALERT_VALUES;
                        break;
                    default:
                        if (type.contains(".")) {
                            try {
                                Class<?> c = Class.forName(type);
                                if (c.isEnum()) {
                                    values = Arrays.stream(c.getEnumConstants()).map(e -> e.toString()).collect(Collectors.toList()).toArray(new String[0]);
                                }
                            } catch (ClassNotFoundException e) {
                            }
                        }
                }
                trend.setValues(values);
            }
        }
        return type == null ? UNKNOWN : type;
    }
    
    static private boolean equal(double d1, double d2) {
        return Math.round(d1) == Math.round(d2);
    }
    
    static private boolean valid(double d) {
        return d > -.5;
    }
    
    static private boolean valid(long l) {
        return l > 0L;
    }
    
    static private int toInt(double d) {
        return (int) Math.round(d);
    }
    
    static private Set<String> getValidStateValues(StateID sid) {
        Map<String, String[]> state2values = stateValues.get(sid.agent);
        if (state2values == null) {
            return Collections.emptySet();
        } else {
            String[] values = state2values.get(sid.state);
            return values == null ? Collections.emptySet() : new HashSet<>(Arrays.asList(values));
        }
    }
    
}