package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.Component;
import javax.swing.JComponent;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.gconsole.base.OutputPane;
import org.lsst.ccs.gconsole.base.panel.DataPage;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.gconsole.services.persist.Persistable;
import org.lsst.ccs.gconsole.util.ThreadUtil;

/**
 * Configurable message viewer that manages a single page displaying filtered bus messages.
 * <p>
 * <i>Implementation notes.<br>
 * Persistency. Two kinds of Tracer instances exist: those created by {@code Creator}, and those created by {@code TracerEditor}.
 * For the first kind, the descriptor will always have valid {@code creator} property and null {@code filter} property.
 * For the second kind, {@code creator} property is {@code null}. These instances are created by the default constructor and a
 * call to {@code restore(...)}.
 * </i>
 *
 * @author onoprien
 */
public class Tracer implements Persistable {

// -- Fields : -----------------------------------------------------------------
    
    static public final String CATEGORY = "Tracer";
    
    private Panel panel;
    private volatile MessageFilter filter;
    private volatile Descriptor descriptor;

// -- Life cycle : -------------------------------------------------------------
    
    public Tracer() {
        descriptor = new Descriptor();
        OutputPane.Descriptor d = new OutputPane.Descriptor();
        d.setSkipLine(true);
        descriptor.setOut(d);
    }
    
    /**
     * Returns the graphical component that displays messages, creating it if it does not already exist.
     * This method should be called on EDT.
     * 
     * @return Graphical component of this viewer.
     */
    public JComponent getPanel() {
        if (panel == null) {
            panel = new Panel(descriptor.getOut());
        }
        return panel;
    }
    
    public MessageFilter getFilter() {
        return filter;
    }
    
    public void setFilter(MessageFilter filter) {
        this.filter = filter;
        if (filter instanceof MultistepFilter) {
            descriptor.setCreator(null);
        }
    }

    
// -- Provessing messages : ----------------------------------------------------
    
    public void onMessage(BusMessage message) {
        FilteredMessage before = new FilteredMessage(message);
        FilteredMessage after = filter.apply(before);
        if (after != null) {
            ThreadUtil.invokeLater(() -> {
                if (panel != null) {
                    panel.println(after.getText(), after.getColor());
                }
            });
        }
    }
    
    
// -- Graphical component class : ----------------------------------------------
    
    static private final class Panel extends OutputPane implements DataPage {
        Panel() {
            super();
        }
        Panel(OutputPane.Descriptor desc) {
            super(desc);
        }
    }

    
// -- Saving/restoring : -------------------------------------------------------
    
    static public class Descriptor extends Persistable.Descriptor {

        private DataPanelDescriptor panel;
        private OutputPane.Descriptor out;
        private MultistepFilter.Descriptor filter;

        @Override
        public void setCategory(String category) {
        }

        @Override
        public String getCategory() {
            return Tracer.CATEGORY;
        }

        public MultistepFilter.Descriptor getFilter() {
            return filter;
        }

        public void setFilter(MultistepFilter.Descriptor filter) {
            this.filter = filter;
        }

        public OutputPane.Descriptor getOut() {
            return out;
        }

        public void setOut(OutputPane.Descriptor out) {
            this.out = out;
        }

        public DataPanelDescriptor getPanel() {
            return panel;
        }

        public void setPanel(DataPanelDescriptor panel) {
            this.panel = panel;
        }

        @Override
        public Descriptor clone() {
            Descriptor desc = (Descriptor) super.clone();
            if (panel != null) desc.panel = panel.clone();
            if (out != null) desc.out = out.clone();
            if (filter != null) desc.filter = filter.clone();
            return desc;
        }
        
    }

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }

    @Override
    public Descriptor save() {
        Descriptor desc = descriptor.clone();
        if (panel == null) {
            desc.setPanel(null);
        } else {
            desc.setPanel(DataPanelDescriptor.get(panel));
            desc.setOut(panel.save());
        }
        MessageFilter f = filter;
        if (f instanceof MultistepFilter) {
            desc.setFilter(((MultistepFilter) f).save());
        }
        return desc;
    }

    @Override
    public void restore(Persistable.Descriptor d) {
        if (d instanceof Descriptor) {
            descriptor = ((Descriptor)d).clone();
            if (descriptor.getCreator() == null) {
                MultistepFilter.Descriptor fd = descriptor.getFilter();
                if (fd != null) {
                    descriptor.setFilter(null);
                    filter = new MultistepFilter(fd);
                }
            }
        }
    }

    @Override
    public Tracer edit(String title, Component parent) {
        return TracerEditor.edit(this, parent);
    }
    
}
