package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.RowFilter;
import javax.swing.RowSorter;
import javax.swing.SortOrder;
import javax.swing.event.ChangeListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableModel;
import javax.swing.table.TableRowSorter;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.bus.data.AgentGroup;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.services.lock.Locker;
import org.lsst.ccs.gconsole.services.lock.LockService;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;

/**
 * A {@link Browser} that works with multiple remote subsystems.
 *
 * @author onoprien
 */
public final class BrowserFull extends Browser {

// -- Fields : -----------------------------------------------------------------

    private AgentTableModel agentTableModel;
    private AgentTable agentTable;    
    private AgentPanel agentPanel;
    private JLabel filterLabel;
   
    private final ArrayList<Locker> agents = new ArrayList<>();
    private final Filter filter = new Filter();
    private Descriptor descriptor;
    
    private ChangeListener agentProxyListener;
    private LockService.Listener lockServiceListener;
    
    private final AbstractAction lockAllAction, unlockAllAction, attachAllAction, detachAllAction;
    

// -- Life cycle : -------------------------------------------------------------
    
    public BrowserFull() {
        descriptor = new Descriptor();
        
        // Actions on all locks :
        
        lockAllAction = new AbstractAction("Lock all") {
            @Override
            public void actionPerformed(ActionEvent e) {
                List<String> toAttach = new ArrayList<>(agents.size());
                List<String> toLock = new ArrayList<>(agents.size());
                for (Locker a : agents) {
                    if (filter.filter(a)) {
                        switch (a.getState()) {
                            case DETACHED:
                                toAttach.add(a.getName()); break;
                            case UNLOCKED:
                                toLock.add(a.getName()); break;
                        }
                    }
                }
                if (!toAttach.isEmpty()) {
                    service.executeBulkOperation(LockService.Operation.ATTACH, toAttach);
                }
                if (!toLock.isEmpty()) {
                    service.executeBulkOperation(LockService.Operation.LOCK, toLock);
                }
            }            
        };
        lockAllAction.putValue(Action.SHORT_DESCRIPTION, "Lock or attach all available subsystems");
        
        unlockAllAction = new AbstractAction("Unlock all") {
            @Override
            public void actionPerformed(ActionEvent e) {
                List<String> toUnlock = new ArrayList<>(agents.size());
                for (Locker a : agents) {
                    if (filter.filter(a)) {
                        switch (a.getState()) {
                            case DETACHED:
                            case ATTACHED:
                                toUnlock.add(a.getName()); break;
                        }
                    }
                }
                if (!toUnlock.isEmpty()) {
                    service.executeBulkOperation(LockService.Operation.UNLOCK, toUnlock);
                }
            }            
        };
        unlockAllAction.putValue(Action.SHORT_DESCRIPTION, "Unlock all subsystems locked by "+ service.getUserId());
        
        attachAllAction = new AbstractAction("Attach all") {
            @Override
            public void actionPerformed(ActionEvent e) {
                List<String> toAttach = new ArrayList<>(agents.size());
                for (Locker a : agents) {
                    if (filter.filter(a)) {
                        switch (a.getState()) {
                            case DETACHED:
                                toAttach.add(a.getName()); break;
                        }
                    }
                }
                if (!toAttach.isEmpty()) {
                    service.executeBulkOperation(LockService.Operation.ATTACH, toAttach);
                }
            }            
        };
        attachAllAction.putValue(Action.SHORT_DESCRIPTION, "Attach all subsystems locked by "+ service.getUserId());
        
        detachAllAction = new AbstractAction("Detach all") {
            @Override
            public void actionPerformed(ActionEvent e) {
                List<String> toDetach = new ArrayList<>(agents.size());
                for (Locker a : agents) {
                    if (filter.filter(a)) {
                        switch (a.getState()) {
                            case ATTACHED:
                                toDetach.add(a.getName()); break;
                        }
                    }
                }
                if (!toDetach.isEmpty()) {
                    service.executeBulkOperation(LockService.Operation.DETACH, toDetach);
                }
            }            
        };
        detachAllAction.putValue(Action.SHORT_DESCRIPTION, "Detach all subsystems without unlocking them");
    }
    
    @Override
    public JComponent getPanel() {
        if (browserPanel == null) {
            super.getPanel();
            
            // Agent panel and state label (right):
            
            JPanel rightPane = new JPanel(new BorderLayout());
            rightPane.add(stateLabel, BorderLayout.NORTH);
            agentPanel = new AgentPanel();
            rightPane.add(agentPanel, BorderLayout.CENTER);
            browserPanel.add(rightPane, BorderLayout.CENTER);

            // Subsystem selection panel (left):
            
            JPanel leftPane = new JPanel(new BorderLayout());
            leftPane.setPreferredSize(new Dimension(300, 800));
            browserPanel.add(leftPane, BorderLayout.WEST);

            agentTableModel = new AgentTableModel();
            agentTable = new AgentTable(agentTableModel);
            agentTable.init();
            
            JScrollPane agentsView = new JScrollPane(agentTable);
            leftPane.add(agentsView, BorderLayout.CENTER);

            // Filter panel
            
            Box filterPanel = Box.createHorizontalBox();
            leftPane.add(filterPanel, BorderLayout.SOUTH);
            JButton filterButton = new JButton("Filter...");
            filterButton.addActionListener(e -> FilterDialog.show(filter, browserPanel));
            filterPanel.add(filterButton);
            filterPanel.add(Box.createRigidArea(Const.HDIM));
            filterLabel = new JLabel(filter.getDescription());
            filterPanel.add(filterLabel);
            
            // Set state based on descriptor to the extent possible without seeing connected agents:
            
            restore(descriptor);
            
            // Listener for agent-specific:
            
            agentProxyListener = e -> {
                Locker agent = (Locker) e.getSource();
                for (int i=0; i<agents.size(); i++) {
                    if (agent.equals(agents.get(i))) {
                        agentTableModel.fireTableRowsUpdated(i, i);
                    }
                }
            };
            
            // Start listening for new agents:
            
            lockServiceListener = new LockService.Listener() {
                @Override
                public void agentsAdded(Locker... agentHandles) {
                    int selectIndex = -1;
                    int prevSize = agents.size();
                    for (Locker agent : agentHandles) {
                        agents.add(agent);
                        agent.addListener(agentProxyListener);
                        if (agent.getName().equals(descriptor.getAgent())) {
                            int i = agentTable.convertRowIndexToView(agents.size() - 1);
                            agentTable.getSelectionModel().setSelectionInterval(i, i);
                        }
                    }
                    if (agents.size() > prevSize) {
                        agentTableModel.fireTableRowsInserted(prevSize, agents.size() - 1);
                        if (selectIndex != -1) {
                            agentTable.getSelectionModel().setSelectionInterval(selectIndex, selectIndex);
                        }
                    }
                }
                @Override
                public void agentsRemoved(Locker... agentHandles) {
                    for (Locker agentHandle : agentHandles) {
                        for (int i = 0; i < agents.size(); i++) {
                            if (agentHandle.equals(agents.get(i))) {
                                agents.remove(i);
                                agentTableModel.fireTableRowsDeleted(i, i);
                                break;
                            }
                        }
                    }
                }
            };
            LockService.getService().addListener(lockServiceListener);
            
        }
        return browserPanel;
    }

    @Override
    public void shutdown() {
        LockService.getService().removeListener(lockServiceListener);
        agents.forEach(a -> a.removeListener(agentProxyListener));
        browserPanel = null;
    }
    
    
// -- Getters : ----------------------------------------------------------------

    @Override
    public String getName() {
        return descriptor.getName();
    }

    @Override
    String getAgentName() {
        Locker agent = getSelectedAgent();
        return agent == null ? null : agent.getName();
    }

// -- Saving/restoring : -------------------------------------------------------

    @Override
    public void restore(Serializable descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
            filter.restore();
            if (browserPanel != null) {
                String selectedAgentName = this.descriptor.getAgent();
                if (selectedAgentName == null) {
                    agentTable.getSelectionModel().clearSelection();
                } else {
                    for (int i = agents.size()-1; i>=0; i--) {
                        if (selectedAgentName.equals(agents.get(i).getName())) {
                            i = agentTable.convertColumnIndexToView(i);
                            agentTable.getSelectionModel().setSelectionInterval(i, i);
                            break;
                        }
                    }

                }
                AgentPanel.Descriptor apDesc = this.descriptor.getAgentPanel();
                if (apDesc != null) agentPanel.restore(apDesc);
            }
        } else {
            logger.debug("Incorrect descriptor type for BrowserFull command browser: "+ descriptor);
        }
    }

    @Override
    public Descriptor save() {
        if (agentPanel == null) {
            descriptor.setAgentPanel(null);
        } else {
            descriptor.setAgentPanel(agentPanel.save());
        }
        if (browserPanel != null) {
            descriptor.setPage(DataPanelDescriptor.get(browserPanel));
        }
        return descriptor;
    }
    
    public static class Descriptor extends Browser.Descriptor {

        private String[] filterTypes;
        private ArrayList<String> filterNames;
        private ArrayList<String> filterGroups;
        private HashMap<String, String> filterProperties;

        public String[] getFilterTypes() {
            return filterTypes;
        }

        public void setFilterTypes(String[] filterTypes) {
            this.filterTypes = filterTypes;
        }

        public ArrayList<String> getFilterNames() {
            return filterNames;
        }

        public void setFilterNames(ArrayList<String> filterNames) {
            this.filterNames = filterNames;
        }

        public ArrayList<String> getFilterGroups() {
            return filterGroups;
        }

        public void setFilterGroups(ArrayList<String> filterGroups) {
            this.filterGroups = filterGroups;
        }

        public HashMap<String, String> getFilterProperties() {
            return filterProperties;
        }

        public void setFilterProperties(HashMap<String, String> filterProperties) {
            this.filterProperties = filterProperties;
        }
        
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private Locker getSelectedAgent() {
        if (agentTable == null) return null;
        int i = agentTable.getSelectedRow();
        if (i != -1) i = agentTable.convertRowIndexToModel(i);
        return i < 0 ? null : agents.get(i);
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private class AgentTableModel extends AbstractTableModel {

        @Override
        public int getRowCount() {
            return agents.size();
        }

        @Override
        public int getColumnCount() {
            return 3;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            Locker a = agents.get(rowIndex);
            switch (columnIndex) {
                case 0:
                    return " "+ a.getName();
                case 1:
                    return renderLock(a);
                case 2:
                    return a.getLevel();
            }
            throw new IllegalArgumentException();
        }

        @Override
        public boolean isCellEditable(int rowIndex, int columnIndex) {
            return columnIndex == 2 && !agents.get(rowIndex).isAdjusting();
        }

        @Override
        public Class<?> getColumnClass(int columnIndex) {
            switch (columnIndex) {
                case 0:
                    return String.class;
                case 1:
                    return RenderedLock.class;
                case 2:
                    return Integer.class;
            }
            throw new IllegalArgumentException();
        }

        @Override
        public String getColumnName(int columnIndex) {
            switch (columnIndex) {
                case 0:
                    return "Subsystem";
                case 1:
                    return "Lock";
                case 2:
                    return "Level";
            }
            throw new IllegalArgumentException();
        }

        @Override
        public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
            if (columnIndex == 2) {
                Locker agent = agents.get(rowIndex);
                try {
                    int level = Integer.parseInt(aValue.toString());
                    service.executeOperation(LockService.Operation.LEVEL, agent.getName(), level);
                } catch (NumberFormatException x) {
                    Console.getConsole().error("Illegal level format", x);
                }
            }
        }
        
    }
    
    private class AgentTable extends JTable implements HasPopupItems {
        
        AgentTable(AgentTableModel model) {
            super(model);
        }
        
        void init() {
            setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
            setFillsViewportHeight(true);
            setDefaultRenderer(String.class, new AgentNameRenderer());
            setDefaultRenderer(RenderedLock.class, new LockRenderer());
            TableCellRenderer headerRenderer = agentTable.getTableHeader().getDefaultRenderer();
            for (int i = 0; i < 3; i++) {
                TableColumn column = getColumnModel().getColumn(i);
                int w = headerRenderer.getTableCellRendererComponent(null, column.getHeaderValue(), false, false, 0, 0).getPreferredSize().width;
                column.setMinWidth(Math.round(w*1.5f));
                column.setPreferredWidth(i == 0 ? 10*w : Math.round(w*1.5f));
            }
            setRowHeight(Math.max(getRowHeight(), 27));
            
            TableRowSorter<TableModel> sorter = new TableRowSorter<>(getModel());
            List<RowSorter.SortKey> sortKeys = new ArrayList<>();
            sortKeys.add(new RowSorter.SortKey(0, SortOrder.ASCENDING));
            sortKeys.add(new RowSorter.SortKey(1, SortOrder.ASCENDING));
            sorter.setSortKeys(sortKeys);
            sorter.setRowFilter(filter);
            setRowSorter(sorter);
            
            getSelectionModel().addListSelectionListener(e -> {
                if (!e.getValueIsAdjusting()) {
                    Locker agent = getSelectedAgent();
                    agentPanel.setAgent(agent);
                    updateStateDisplay();
                }
            });
            
            addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    if (e.getClickCount() == 2) {
                        Point point = e.getPoint();
                        int column = agentTable.columnAtPoint(point);
                        if (column == 1) {
                            int row = agentTable.convertRowIndexToModel(agentTable.rowAtPoint(point));
                            if (row >= 0) {
                                RenderedLock renderedLock = (RenderedLock)  agentTableModel.getValueAt(row, 1);
                                if (renderedLock != null && renderedLock.defaultAction != null) {
                                    renderedLock.defaultAction.actionPerformed(null);
                                }
                            }
                        }
                    }
                }
            });
            
        }

        @Override
        public JPopupMenu modifyPopupMenu(JPopupMenu menu, Component component, Point point) {
            
            // Agent-specific actions
            
            int row = agentTable.rowAtPoint(point);
            if (row >= 0) row = agentTable.convertRowIndexToModel(row);
            if (row >= 0) {
                RenderedLock renderedLock = (RenderedLock)  agentTableModel.getValueAt(row, 1);
                if (renderedLock != null && renderedLock.availableActions != null) {
                    renderedLock.availableActions.forEach(a -> menu.add(new JMenuItem(a)));
                }
                menu.addSeparator();
            }
            
            // Bulk actions
            
            menu.add(new JMenuItem(lockAllAction));
            menu.add(new JMenuItem(unlockAllAction));
            menu.add(new JMenuItem(attachAllAction));
            menu.add(new JMenuItem(detachAllAction));
            
            // Return modified menu
            
            return menu;
        }
        
    }

    private class AgentNameRenderer extends DefaultTableCellRenderer {
        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
            setToolTipText(value.toString());
            Locker agent = agents.get(table.convertRowIndexToModel(row));
            setEnabled(agent.isOnline());
            return this;
        }
    }
    
    class LockRenderer extends DefaultTableCellRenderer {
        @Override
        protected void setValue(Object value) {
            super.setValue(value);
            setText("");
            RenderedLock renderedLock = (RenderedLock) value;
            setIcon(renderedLock.icon);
            setToolTipText(renderedLock.tooltip);
        }
    }
    
    class Filter extends RowFilter<Object,Integer> {
        
        EnumSet<AgentInfo.AgentType> types;
        ArrayList<String> names;
        ArrayList<String> groups;
        HashMap<String, String> properties;

        @Override
        public boolean include(RowFilter.Entry<? extends Object, ? extends Integer> entry) {
            int i = entry.getIdentifier();
            Locker agent = agents.get(i);
            return filter(agent);
        }
        
        boolean filter(Locker agent) {
            
            AgentInfo info = agent.getInfo();
            String name = agent.getName();

            if (types != null && info != null) {
                if (!types.contains(info.getType())) {
                    return false;
                }
            }

            if (names != null) {
                boolean fail = true;
                for (String fragment : names) {
                    if (name.contains(fragment)) {
                        fail = false;
                        break;
                    }
                }
                if (fail) {
                    return false;
                }
            }

            if (groups != null && info != null) {
                String group = info.getAgentProperty(AgentGroup.AGENT_GROUP_PROPERTY, "");
                if (!groups.contains(group)) {
                    return false;
                }
            }

            if (properties != null && info != null) {
                boolean fail = true;
                for (Map.Entry<String, String> e : properties.entrySet()) {
                    String value = info.getAgentProperty(e.getKey());
                    if (Objects.equals(value, e.getValue()) || (value != null && "".equals(e.getValue()))) {
                        fail = false;
                        break;
                    }
                }
                if (fail) {
                    return false;
                }
            }

            return true;
        }
        
        void apply() {
            agentTableModel.fireTableDataChanged();
            filterLabel.setText(getDescription());
        }
    
        void save() {
            if (types == null) {
                descriptor.setFilterTypes(null);
            } else {
                String[] tt = new String[this.types.size()];
                int i = 0;
                for (AgentInfo.AgentType type : this.types) {
                    tt[i++] = type.name();
                }
                descriptor.setFilterTypes(tt);
            }

            descriptor.setFilterNames(names == null ? null : new ArrayList<>(names));
            descriptor.setFilterGroups(groups == null ? null : new ArrayList<>(groups));
            descriptor.setFilterProperties(properties == null ? null : new HashMap<>(properties));
        }

        void restore() {
            if (descriptor.getFilterTypes() == null) {
                types = null;
            } else {
                types = EnumSet.noneOf(AgentInfo.AgentType.class);
                for (String s : descriptor.getFilterTypes()) {
                    try {
                        types.add(AgentInfo.AgentType.valueOf(s));
                    } catch (IllegalArgumentException x) {
                    }
                }
            }
            names = descriptor.getFilterNames() == null ? null : new ArrayList<>(descriptor.getFilterNames());
            groups = descriptor.getFilterGroups() == null ? null : new ArrayList<>(descriptor.getFilterGroups());
            properties = descriptor.getFilterProperties() == null ? null : new HashMap<>(descriptor.getFilterProperties());
        }
        
        String getDescription() {
            if (types == null) {
                if (names == null && groups == null && properties == null) {
                    return "All";
                }
            } else {
                if (names == null && groups == null && properties == null && types.size() < 3) {
                    return String.join(" & ", types.stream().map(t -> t.displayName()).collect(Collectors.toList()));
                }
            }
            return "Custom";
        }
    }
    
}
