package org.lsst.ccs.gconsole.util.tree;

import java.io.Serializable;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;
import org.lsst.ccs.gconsole.services.persist.Savable;

/**
 * Model for {@link STree}.
 *
 * @author onoprien
 * @param <T> Type of objects that can be associated with nodes of this tree.
 */
public class SModel<T> extends DefaultTreeModel implements Savable {

// -- Fields : -----------------------------------------------------------------
    
    protected String rootName = "";
    protected Map<String,T> pathToPayloadMap;
    
    protected Descriptor descriptor = new Descriptor();

// -- Life cycle : -------------------------------------------------------------
    
    public SModel() {
        super(null);
    }
    
    public SModel(Map<String, ? extends T> data) {
        super(null);
        update(data);
    }
    
    
// -- Override DefaultTreeModel : ----------------------------------------------

    @Override
    public SNode<T> getChild(Object parent, int index) {
        return (SNode<T>) super.getChild(parent, index);
    }

    @Override
    public SNode<T> getRoot() {
        return (SNode<T>) super.getRoot();
    }


// -- Setters/getters : --------------------------------------------------------
    
    public void setRootName(String rootName) {
        this.rootName = rootName;
    }
    
    public String getRootName() {    
        return rootName;
    }
    
    public TreePath getPath(String path) {
        String[] pp = path.split("/+");
        return getPath(pp);
    }
    
    public TreePath getPath(String[] path) {
        Object[] treePath = new Object[path.length+1];
        SNode<T> node = getRoot();
        treePath[0] = node;
        for (int i=0; i<path.length; i++) {
            node = node.getChild(path[i]);
            if (node == null) return null;
            treePath[i+1] = node;
        }
        return new TreePath(treePath);
    }
    
    public LinkedHashMap<String,T> getData() {
        return new LinkedHashMap<>(pathToPayloadMap);
    }

    
// -- Updating : ---------------------------------------------------------------
    
    public void sort(Sort sort) {
        if (sort == null) sort = Sort.NONE;
        if (descriptor.getSort() != sort) {
            descriptor.setSort(sort);
            if (pathToPayloadMap != null && !pathToPayloadMap.isEmpty()) {
                update(pathToPayloadMap);
            }
        }
    }
    
    public void update(Map<String, ? extends T> data) {
        pathToPayloadMap = new LinkedHashMap<>(data);
        
        Map<String,T> sortedData;
        if (descriptor.getSort() == Sort.NONE) {
            sortedData = pathToPayloadMap;
        } else {
            sortedData = new TreeMap<>(descriptor.getSort());
            sortedData.putAll(pathToPayloadMap);
        }
        
        SNode<T> root = new SNode<>(rootName);
        for (Map.Entry<String,T> e : sortedData.entrySet()) {
            String[] pp = e.getKey().split("/+");
            SNode<T> parent = root;
            for (String p : pp) {
                int n = parent.getChildCount();
                SNode<T> node = null;
                for (int i=0; i<n; i++) {
                    SNode<T> child = parent.getChildAt(i);
                    if (p.equals(child.getName())) {
                        node = child;
                        break;
                    }
                }
                if (node == null) {
                    node = new SNode(p);
                    parent.add(node);
                }
                parent = node;
            }
            parent.setUserObject(e.getValue());
        }
        
        setRoot(root);
    }
    

// -- Saving/restoring : -------------------------------------------------------
    
    static public class Descriptor implements Serializable, Cloneable {

        private Sort sort = Sort.ALPHABETIC_LEAVES_LAST;

        public Sort getSort() {
            return sort;
        }

        public void setSort(Sort sort) {
            this.sort = sort;
        }

        @Override
        public Descriptor clone() {
            try {
                return (Descriptor) super.clone();
            } catch (CloneNotSupportedException x) {
                throw new RuntimeException(x); // NEVER
            }
        }
        
    }

    @Override
    public void restore(Serializable descriptor) {
        if (descriptor == null) descriptor = new Descriptor();
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
            if (pathToPayloadMap != null && !pathToPayloadMap.isEmpty()) {
                update(pathToPayloadMap);
            }
        }
    }

    @Override
    public Descriptor save() {
        return descriptor;
    }

}
