package org.lsst.ccs.gconsole.plugins.trending;

import java.util.*;
import org.lsst.ccs.localdb.statusdb.server.ChannelMetaData;

/**
 * Time history dataset for a specific channel and a specific time window.
 * Includes all metadata. Immutable.
 * <p>
 * The data is intended to be directly displayable by the plotter, possibly with scaling (for strings)
 * but without adding or removing points. Correspondence between points received from the REST server
 * and those contained in this dataset:
 * <ul>
 * <li>Raw unbinned trending (numeric or string) datasets: one-to-one, nothing is added or removed.
 * <li>Raw binned or stat numeric trending: one-to-one. Do not use as cache.
 * <li>Raw binned or stat string trending: remove some or all intermediate points in series of same values.
 *     Add endpoints if known. Can be used as cache if new request is not for raw data.
 * <li>Each state change is turned into a pair of points if the previous state is known. 
 *     Endpoints are added if known.
 * </ul>
 * 
 * In case of raw trending (numeric or string) datasets, each point
 * is received from the REST server, nothing is added or removed.
 *
 * @author onoprien
 */
public class TrendData {
    
// -- Fields : -----------------------------------------------------------------
    
    private final long[] times; // time stamps for data and on-point metadata
    private final Map<String, double[]> values; // key to array of values for data and on-point metadata
    private final long begin, end;
    private final boolean raw;
    private final int bins;
    private final String type; // data type
    
    private final String[] names; // Allowed string values, indexed by integer in {@code values}; {@code null} for numeric dataset.
    
    private final Map<String, ArrayList<MetaValue>> meta; // key to off-point metadata
    
// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs dataset with double values.
     */
    TrendData(long[] times, Map<String, double[]> onPointValues, Map<String, ArrayList<MetaValue>> offPointMetadata, long[] timeRange, boolean raw, int bins) {
        this.times = times;
        this.values = onPointValues;
        this.begin = timeRange[0];
        this.end = timeRange[1];
        this.meta = offPointMetadata;
        this.names = null;
        this.type = RestSource.DOUBLE;
        this.raw = raw;
        this.bins = bins;
    }
    
    /**
     * Constructs dataset with String values.
     */
    TrendData(long[] times, double[] values, long[] timeRange, String[] names, String type, boolean raw, int bins) {
        this.times = times;
        this.values = Collections.singletonMap(RestSource.VALUE_KEY, values);
        this.begin = timeRange[0];
        this.end = timeRange[1];
        this.meta = Collections.emptyMap();
        this.names = names;
        this.type = type;
        this.raw = raw;
        this.bins = bins;
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns an array of timestamps for points in this dataset.
     * Shared by data and on-point metadata.
     * 
     * @return An array of times, in milliseconds.
     */
    public long[] getTimes() {
        return times;
    }
    
    /**
     * Returns an array of values for data in this dataset.
     * @return Data values.
     */
    public double[] getValues() {
        return values.get(RestSource.VALUE_KEY);
    }

    /**
     * Returns an array of timestamps available in this dataset for the specified data or metadata.
     * 
     * @param key String that identifies the data or metadata.
     * @return An array of times, in milliseconds, or {@code null} if there is no such data or metadata.
     */
    public long[] getTimes(String key) {
        if (values.containsKey(key)) {
            return times;
        } else {
            List<MetaValue> metaList = meta.get(key);
            if (metaList == null) return null;
            int n = metaList.size();
            long[] out = new long[n * 2];
            for (int i = 0; i < n; i++) {
                int k = 2 * i;
                MetaValue mp = metaList.get(i);
                out[k] = mp.getStart();
                out[k + 1] = mp.getStop();
            }
            return out;
        }
    }
    
    /**
     * Returns an array of values contained in this dataset for the specified metadata type.
     * 
     * @param key String that identifies the metadata.
     * @return An array of values for the specified metadata, or {@code null} if there is no
     *         such data or metadata that could be converted to double values.
     */
    public double[] getValues(String key) {
        double[] out = values.get(key);
        if (out == null) {
            List<MetaValue> metaList = meta.get(key);
            if (metaList == null) return null;
            int n = metaList.size();
            out = new double[n * 2];
            try {
                for (int i = 0; i < n; i++) {
                    int k = 2 * i;
                    double v = metaList.get(i).getDoubleValue();
                    out[k] = v;
                    out[k + 1] = v;
                }
            } catch (NumberFormatException x) {
                return null;
            }
        }
        return out;
    }
    
    /**
     * Returns an array of string values contained in this dataset for the specified off-point metadata type.
     * 
     * @param key String that identifies the metadata.
     * @return An array of metadata values, or {@code null} if there is no such metadata in this dataset.
     */
    public String[] getMetadata(String key) {
        ArrayList<MetaValue> metaList = meta.get(key);
        if (metaList == null) return null;
        int n = metaList.size();
        String[] out = new String[n * 2];
        for (int i=0; i<n; i++) {
            int k = 2*i;
            MetaValue mp = metaList.get(i);
            out[k] = mp.getValue();
            out[k+1] = mp.getValue();
        }
        return out;
    }
    
    /**
     * Returns off-point metadata value for the specified key if it is valid for the entire time window covered by this dataset.
     * Note that the current implementation does not require validity throughout time window, it returns a non-null value
     * for any single value metadata. This is to work around LSSTCCS-2253.
     * 
     * @param key String that identifies the metadata.
     * @return Value, or {@code null} if there is no such single-value metadata.
     */
    public String getSingleValueMetadata(String key) {
        ArrayList<MetaValue> metaList = meta.get(key);
        if (metaList == null || metaList.isEmpty()) return null;
        String out = metaList.get(0).getValue();
        for (int i=1; i<metaList.size(); i++) {
            if (!Objects.equals(out, metaList.get(i).getValue())) {
                return null;
            }
        }
        return out;
    }
    
    /**
     * Returns the set of keys for on-point data and metadata present in this dataset.
     * The set includes {@code VALUE_KEY}.
     * 
     * @return Set of keys.
     */
    public Set<String> getOnPointKeys() {
        HashSet<String> out = new HashSet<>(values.keySet());
        return out;
    }
    
    /**
     * Returns earliest time for data points contained in this dataset (excluding added endpoints).
     * @return Earliest time for data contained in this dataset, or {@code -1} if empty.
     */
    public long getLowT() {
        if (times == null || times.length == 0) {
            return -1;
        } else if (RestSource.STATE.equals(type) || (!raw && (RestSource.BOOL.equals(type) || RestSource.STRING.equals(type)))) {
            if (times.length == 1) {
                return times[0];
            } else {
                if (times[0] == begin + 1) {
                    if (times.length == 2 && times[1] == end) {
                        double[] v = getValues();
                        if (Math.round(v[0]) == Math.round(v[1])) {
                            return -1;
                        } else {
                            return times[0];
                        }
                    } else {
                        return times[0] == times[1] ? times[1] : times[0];
                    }
                } else {
                    return times[0];
                }
            }
        } else {
            return times[0];
        }
    }
    
    /**
     * Returns latest time for data points contained in this dataset (excluding added endpoints).
     * @return Latest time for data contained in this dataset, or {@code -1} if empty. 
     */
    public long getHighT() {
        return (times == null || times.length == 0) ? -1 : times[times.length - 1]; // FIXME
    }
    
    /**
     * Returns the time range of this data set.
     * @return two-element array containing the bounds of the time range of this data set.
     */
    public long[] getTimeRange() {
        return new long[] {begin, end};
    }
    
    /**
     * Returns the beginning of the time range of this data set.
     * @return Beginning of time window covered by this dataset.
     */
    public long getBegin() {
        return begin;
    }
    
    /**
     * Returns the end of the time range of this data set.
     * @return End of time window covered by this dataset.
     */
    public long getEnd() {
        return end;
    }

    /**
     * Returns {@code true} if this dataset contains unbinned raw data.
     * @return {@code true} if unbinned raw data.
     */
    public boolean isRawUnbinned() {
        return raw && bins == 0;
    }

    /**
     * True if this dataset was created in response to request for raw (binned or unbinned) data.
     * @return Requested data kind.
     */
    public boolean isRaw() {
        return raw;
    }

    /**
     * Returns the number of bins specified in the request in response to which this dataset was created.
     * @return Requested number of bins.
     */
    public int getBins() {
        return bins;
    }
        
    /**
     * Returns names of String constants that can be included in this dataset, indexed by value.
     * @return Possible String values.
     * @throws UnsupportedOperationException if this is a numeric dataset.
     */
    public String[] getNames() {
        if (names == null) {
            throw new UnsupportedOperationException("This is a numeric dataset, no names.");
        } else {
            return names;
        }
    }
    
    /**
     * Returns {@code true} if this is a numeric dataset (as opposed to string or state).
     * @return True if this is a numeric dataset.
     */
    public boolean isNumeric() {
        return names == null;
    }

    /**
     * Returns data type in this dataset (see {@code RestSource}).
     * @return Data type.
     */
    public String getType() {
        return type;
    }
    
    
// -- Package-private access to off-point metadata : ---------------------------
    
    static class MetaValue {
        
        private final long start;
        private final long stop;
        private final String value;
        
        MetaValue(long start, long stop, String value) {
            this.start = start;
            this.stop = stop;
            this.value = value;
        }
        
        long getStart() {
            return start;
        }
        
        long getStop() {
            return stop;
        }
        
        String getValue() {
            return value;
        }
        
        double getDoubleValue() {
            return Double.parseDouble(value);
        }
        
        static MetaValue trim(MetaValue in, long begin, long end) {
            if (in.start >= end || in.stop <= begin) {
                return null;
            } else if (in.start < begin || in.stop > end) {
                return new MetaValue(Math.max(in.start, begin), Math.min(in.stop, end), in.value);
            } else {
                return in;
            }
        }
        
        static MetaValue trim(ChannelMetaData in, long begin, long end) {
            long start = in.getTstart();
            long stop = in.getTstop();
            if (stop == -1L) stop = Long.MAX_VALUE;
            if (start >= end || stop <= begin) {
                return null;
            } else {
                String value = in.getValue();
                if (value == null) {
//                    Console.getConsole().getLogger().warn("Null metadata value: name "+ in.getName() +", start "+ start +", stop "+ stop); // FIXME: diagnostics for LSSTCCS-2792
                    return null;
                }
                return new MetaValue(Math.max(start, begin), Math.min(stop, end), value);
            }
        }
        
        static MetaValue merge(MetaValue prev, MetaValue next) {
            if (!prev.value.equals(next.value)) return null;
            if (prev.stop >= next.start || prev.stop + 1L == next.start) {
                return new MetaValue(prev.start, next.stop, prev.value);
            } else {
                return null;
            }
        }
        
    }
    
    Map<String, ArrayList<MetaValue>> getOffPointMetadata() {
        return meta;
    }
    
}
