package org.lsst.ccs.gconsole.util.swing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.event.ActionListener;
import java.util.*;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import org.lsst.ccs.gconsole.base.Const;

/**
 * Widget for searching through a list of strings.
 * <p>
 * To use a {@code SearchField} embedded in an arbitrary component, register an {@code ActionListener} on it.
 * The listener will be notified when the user makes a selection, with the action command indicating the chosen
 * string (an empty string means the search was canceled).
 * <p>
 * To display a pre-configured search dialog, use {@link #search} method.
 *
 * @author onoprien
 */
public final class SearchField extends JTextField {

// -- Fields : -----------------------------------------------------------------
    
    private final ArrayList<String> all;
    private ArrayList<String> candidates;
    private String prevText;
    private String selected;
    private int maxItems = 20;
    private JPopupMenu popup;
    private final ActionListener itemsListener;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs a search field with a pre-defined list of choices.
     * @param choices Strings to select from.
     */
    public SearchField(List<String> choices) {
        super();
        all = new ArrayList<>(choices);
        addCaretListener(e -> updatePopup());
        itemsListener = e -> {
            selected = e.getActionCommand();
            fireActionPerformed();
        };
        updatePopup();
    }

    
// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the maximum number of choices to display in the popup.
     * The default is 20.
     * 
     * @param nItems Maximum number of strings to display in the popup.
     */
    public void setMaxItemsInPopup(int nItems) {
        maxItems = nItems;
    }
    
    
// -- Notifying listeners : ----------------------------------------------------

    @Override
    protected void fireActionPerformed() {
        if (selected == null) {
            String text = getText();
            if (text.trim().isEmpty()) {
                setActionCommand("");
            } else {
                String out = "";
                for (String s : all) {
                    if (s.contains(text)) {
                        out = s;
                        break;
                    }
                }
                setActionCommand(out);
            }
        } else {
            setActionCommand(selected);
        }
        super.fireActionPerformed();
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void updatePopup() { // FIXME: PoP, should be done efficiently
        selected = null;
        String text = getText();
        if (text.isEmpty()) {
            hidePopup();
            candidates = all;
        } else {
            if (!(text.contains(prevText))) {
                candidates = all;
            }
            ArrayList<String> items = new ArrayList<>(maxItems);
            int n = 0;
            for (String candidate : candidates) {
                if (candidate.contains(text)) {
                    if (++n > maxItems) {
                        hidePopup();
                        setForeground(Color.BLACK);
                        return;
                    } else {
                        items.add(candidate);
                    }
                }
            }
            if (items.isEmpty()) {
                hidePopup();
                setForeground(Color.RED);
            } else {
                showPopup(items);
                setForeground(Color.BLACK);
            }
            candidates = items;
        }
        prevText = text;
        requestFocusInWindow();
    }
    
    private void hidePopup() {
        if (popup != null) {
            popup.setVisible(false);
            popup = null;
        }
    }
    
    private void showPopup(List<String> items) {
        popup = new JPopupMenu();
        for (String item : items) {
            JMenuItem menuItem = new JMenuItem(item);
            menuItem.addActionListener(itemsListener);
            popup.add(menuItem);
        }
        popup.show(this, 0, this.getHeight());
    }
    
    
// -- Search dialog : ----------------------------------------------------------
    
    /**
     * Displays a dialog that allows searching for a string among the provided list.
     * 
     * @param choices Strings to select from.
     * @param title Dialog title.
     * @param parent Parent component.
     * @return Selected string, or {@code null} if the search is canceled.
     */
    static public String search(List<String> choices, String title, Component parent) {
        Dialog dialog = new Dialog(choices, title, parent);
        dialog.setSize(dialog.getPreferredSize());
        dialog.pack();
        dialog.setLocationRelativeTo(parent);
        dialog.field.requestFocusInWindow();
        dialog.setVisible(true);
        return dialog.select;
        
    }
    
    static private final class Dialog extends JDialog {
        
        String select;
        final SearchField field;
        
        Dialog(List<String> items, String title, Component parent) {
            super(parent == null ? null : SwingUtilities.getWindowAncestor(parent), null, ModalityType.APPLICATION_MODAL);
            setLayout(new BorderLayout());
            setResizable(false);
            
            Box box = Box.createHorizontalBox();
            add(box, BorderLayout.NORTH);
            box.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE));
            box.add(new JLabel(title));
            box.add(Box.createRigidArea(Const.HDIM));
            box.add(Box.createHorizontalGlue());
            JButton cancel = new JButton("Cancel");
            box.add(cancel);
            cancel.addActionListener(e -> {
                select = null;
                dispose();
            });
            
            field = new SearchField(items);
            add(field, BorderLayout.SOUTH);
            field.addActionListener(e -> {
                String item = e.getActionCommand();
                select = "".equals(item) ? null : item;
                dispose();
            });
        }
        
    }
    
    
// -- Test : -------------------------------------------------------------------
    
    static public void main(String... args) {
        SwingUtilities.invokeLater(() -> {
            List<String> items = Arrays.asList("uno", "dos", "tres", "treinta", "sorok/chetyre/s/polovinoy");
            String selected = search(items, "Search for a channel", null);
            System.out.println(selected);
        });
    }
    
    
    

}
