package org.lsst.ccs.gconsole.services.aggregator;

import java.util.*;
import java.util.function.Predicate;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Collectors;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.util.MatchPredicate;

/**
 * Checks whether an {@code AgentChannel} satisfies certain criteria.
 * <p>
 * Selectors are created by calling one of the static {@code compile(...)} methods that convert
 * string representation of selectors into instances of {@code ChannelSelector}.
 * See {@link AgentChannelsFilter} for details on how string representations are interpreted.
 *
 * @author onoprien
 */
public interface ChannelSelector {
    
// -- Static constants and utilities : -----------------------------------------
    
    /** Predefined selector that accepts all channels. */
    static ChannelSelector ALL = new ChannelSelector() {
        @Override
        public String getAgent() {
            return null;
        }
        @Override
        public boolean match(String originPath) {
            return true;
        }
    };
    
    static final Pattern pRegEx = Pattern.compile("[/+.*?(){}|\\\\]");

    /**
     * Constructs a selector based on a string in path, template, or selector format.
     * See {@link AgentStatusAggregator#addListener(AgentStatusListener, Collection, Collection) AgentStatusAggregator.addListener(...)} for details.
     * 
     * @param rawSelector String representation of selector.
     * @return New instance of {@code ChannelSelector}, or {@code null} if the provided string is not a valid representation of a channel selector.
     */
    static ChannelSelector compile(String rawSelector) {
        try {
            if (pRegEx.matcher(rawSelector).find()) {
                return new PathPattern(rawSelector);
            } else {
                return new AttributeSelector(rawSelector);
            }
        } catch (RuntimeException x) {
            return null;
        }
    }
    
    /**
     * Constructs selectors from their string representations (in path, template, or selector format).
     * See {@link AgentStatusAggregator#addListener(AgentStatusListener, Collection, Collection) AgentStatusAggregator.addListener(...)} for details.
     * 
     * @param rawSelectors String representations of selectors.
     * @return New instances of {@code ChannelSelector}.
     */
    static List<ChannelSelector> compile(Collection<String> rawSelectors) {
        return rawSelectors == null ? null : rawSelectors.stream()
                                                         .map(cs -> ChannelSelector.compile(cs))
                                                         .filter(s -> s != null)
                                                         .collect(Collectors.toList());
    }
    
    /**
     * Checks whether the specified channel is accepted by at least one of the provided selectors.
     * 
     * @param agents Acceptable agents ({@code null} means all) for selectors that do not
     *               explicitly specify agent ({@code getAgent()} method returns {@code null}).
     * @param channelSelectors Selectors to try. If {@code null}, any channel that belongs to one of
     *                         the specified agents is accepted.
     * @param channel Channel to check.
     * @return True if the specified channel is accepted by at least one of the provided selectors.
     */
    static boolean accept(List<String> agents, List<ChannelSelector> channelSelectors, AgentChannel channel) {
        String agentName = channel.getAgentName();
        boolean watchAgent = agents == null || agents.contains(agentName);
        if (channelSelectors == null) {
            return watchAgent;
        } else {
            for (ChannelSelector s : channelSelectors) {
                if (watchAgent || agentName.equals(s.getAgent())) {
                    return s.match(channel);
                }
            }
            return false;
        }
    }
    
    /**
     * Checks whether the specified channel is accepted by at least one of the provided selectors.
     * 
     * @param agents Acceptable agents ({@code null} means all) for selectors that do not
     *               explicitly specify agent ({@code getAgent()} method returns {@code null}).
     * @param channelSelectors Selectors to try. If {@code null}, any channel that belongs to one of
     *                         the specified agents is accepted.
     * @param originPath Original channel path to check.
     * @return True if the specified channel is accepted by at least one of the provided selectors.
     */
    static boolean accept(List<String> agents, List<ChannelSelector> channelSelectors, String originPath) {
        String agentName = originPath.substring(0, originPath.indexOf('/'));
        boolean watchAgent = agents == null || agents.contains(agentName);
        if (channelSelectors == null) {
            return watchAgent;
        } else {
            for (ChannelSelector s : channelSelectors) {
                if (watchAgent || agentName.equals(s.getAgent())) {
                    return s.match(originPath);
                }
            }
            return false;
        }
    }
    
    /**
     * Filters list of channels, selecting those that match at least one of the selectors.
     * 
     * @param <T> {@code AgentChannel} type.
     * @param agents Acceptable agents ({@code null} means all) for selectors that do not
     *               explicitly specify agent ({@code getAgent()} method returns {@code null}).
     * @param channelSelectors Selectors to try. If {@code null}, any channel that belongs to one of
     *                         the specified agents is accepted.
     * @param channels List of candidate channels.
     * @return List of accepted channels.
     */
    static <T extends AgentChannel> List<T> filter(List<String> agents, List<ChannelSelector> channelSelectors, Collection<T> channels) {
        return channels.stream().filter(ch -> accept(agents, channelSelectors, ch)).collect(Collectors.toList());
    }
    
    /**
     * Filters a list of channels, selecting those that might satisfy the provided filter, taking
     * into account its {@code List<String> getOriginChannels()} and {@code getAgents()} methods.
     * 
     * @param <T> {@code AgentChannel} type.
     * @param filter Filter.
     * @param channels List of candidate channels.
     * @return List of accepted channels.
     */
    static <T extends AgentChannel> List<T> filter(AgentChannelsFilter filter, Collection<T> channels) {
        List<String> agents = filter.getAgents();
        List<ChannelSelector> channelSelectors = ChannelSelector.compile(filter.getOriginChannels());
        return filter(agents, channelSelectors, channels);
    }
    
    /**
     * Filters list of original channel paths, selecting those that match at least one of the selectors.
     * 
     * @param agents Acceptable agents ({@code null} means all) for selectors that do not
     *               explicitly specify agent ({@code getAgent()} method returns {@code null}).
     * @param channelSelectors Selectors to try. If {@code null}, any channel that belongs to one of
     *                         the specified agents is accepted.
     * @param paths List of candidate paths.
     * @return List of accepted channels.
     */
    static List<String> filterPaths(List<String> agents, List<ChannelSelector> channelSelectors, Collection<String> paths) {
        return paths.stream().filter(ch -> accept(agents, channelSelectors, ch)).collect(Collectors.toList());
    }
    
    /**
     * Filters a list of original channel paths, selecting those that might satisfy the provided filter, taking
     * into account its {@code List<String> getOriginChannels()} and {@code getAgents()} methods.
     * Attribute selectors are ignored by this method.
     * 
     * @param filter Filter.
     * @param paths List of candidate paths.
     * @return List of accepted channels.
     */
    static List<String> filterPaths(AgentChannelsFilter filter, Collection<String> paths) {
        List<String> agents = filter.getAgents();
        List<ChannelSelector> channelSelectors = ChannelSelector.compile(filter.getOriginChannels());
        return filterPaths(agents, channelSelectors, paths);
    }
    
    /**
     * Appends channels matching at least one of the selectors to the provided list.
     * The input collection is assumed to contain channels from one agent.
     * 
     * @param <T> Type of input and output lists, subclass of {@code AgentChannel}.
     * @param agents 
     * @param channelSelectors
     * @param channels
     * @param out
     * @return 
     */
    static <T extends AgentChannel> List<T> filter(List<String> agents, List<ChannelSelector> channelSelectors, Collection<T> channels, List<T> out) {
        if (out == null) out = new ArrayList<>();
        if (!channels.isEmpty()) {
            String agentName = channels.iterator().next().getAgentName();
            boolean watchedAgent = agents == null || agents.contains(agentName);
            if (channelSelectors == null) {
                if (watchedAgent) {
                    out.addAll(channels);
                }
            } else {
                for (T ch : channels) {
                    for (ChannelSelector s : channelSelectors) {
                        if (watchedAgent || agentName.equals(s.getAgent())) {
                            if (s.match(ch)) {
                                out.add(ch);
                                break;
                            }
                        }
                    }
                }
            }
        }
        return out;
    }    


// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns the name of the agent this selector is applicable to.
     * @return Agent name, or {@code null} if this selector might be applicable to multiple agents.
     */
    String getAgent();

    
// -- Apply selector : ---------------------------------------------------------
    
    /**
     * Checks whether the specified channel is accepted by this selector.
     * 
     * @param channel Channel to check.
     * @return {@code true} if the channel satisfies the criteria of this selector.
     */
    default boolean match(AgentChannel channel) {
        return match(channel.getPath());
    }

    /**
     * Checks whether the specified channel path is accepted by this selector.
     * 
     * @param originPath Original channel path to check.
     * @return {@code true} if the channel satisfies the criteria of this selector.
     */
    boolean match(String originPath);
    
}


// -- Implementation for selecting based on agent properties and channel attributes : --

final class AttributeSelector implements ChannelSelector {

    private final String agent;
    private final String[] data;
    
    AttributeSelector(String rawTemplate) {
        String[] ss = rawTemplate.split("&");
        ArrayList<String> out = new ArrayList<>(ss.length * 2);
        String agentName = null;
        for (String s : ss) {
            if (!s.isEmpty()) {
                String[] tt = s.split("=");
                String key = tt[0];
                String value = tt.length == 1 ? null : tt[1];
                if ("agent.name".equals(key)) {
                    agentName = value;
                }
                out.add(key);
                out.add(value);
            }
        }
        if (out.isEmpty()) throw new RuntimeException();
        agent = agentName;
        data = out.toArray(new String[0]);
    }

    public String get(String key) {
        for (int i=0; i<data.length; i += 2) {
            if (data[i].equals(key)) {
                return data[i+1];
            }
        }
        return null;
    }

    @Override
    public String getAgent() {
        return agent;
    }

    @Override
    public boolean match(AgentChannel channel) {
        for (int i=0; i<data.length; i += 2) {
            String key = data[i];
            Object v = key.startsWith("agent.") ? channel.getAgent().getAgentProperty(key.substring("agent.".length())) : channel.get(key);
            if (v == null) return false;
            String requiredValue = data[i+1];
            if (!(requiredValue == null || requiredValue.equals(v.toString()))) {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean match(String originPath) {
        return false;
    }
}


// -- Implementation for selecting based on a template : -----------------------

final class PathPattern implements ChannelSelector {
    
    static final Pattern pAgent = Pattern.compile("([\\w\\-]+)/.*");

    private final String agent;
    private final Predicate<String> pattern;
    
    PathPattern(String rawTemplate) {
        if (rawTemplate.startsWith("/")) {
            rawTemplate = "[\\w\\-]+" + rawTemplate;
            agent = null;
        } else {
            Matcher m = pAgent.matcher(rawTemplate);
            agent = m.matches() ? m.group(1) : null;
        }
        if (rawTemplate.endsWith("/")) {
            rawTemplate = rawTemplate + ".+";
        }
        pattern = new MatchPredicate(rawTemplate);
    }

    @Override
    public String getAgent() {
        return agent;
    }

    @Override
    public boolean match(String originPath) {
        return pattern.test(originPath);
    }

}
