package org.lsst.ccs.gconsole.services.aggregator;

import java.time.Instant;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.bus.data.AgentLockInfo;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.AgentLockService.AgentLockUpdateListener;
import org.lsst.ccs.services.AgentLoginService;
import org.lsst.ccs.services.AgentLoginService.AgentLoginUpdateListener;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Collects status data from local and remote agents and notifies clients of relevant changes.
 * A singleton instance of this class is created by the graphical console.
 * <p>
 * For each Agent present on the buses, this aggregator obtains data dictionary and compiles the
 * complete list of {@link AgentChannel}. Published data not present in the dictionary is ignored.
 * If there are any listeners interested in a specific Agent, all channels for that Agent are kept
 * up to date. If there are no such listeners, only "infrequently updated channels" are updated.
 * <p>
 * Listeners registered with this aggregator are notified of any changes in the channels they
 * requested. See 
 * {@link #addListener(AgentStatusListener listener, AgentChannelsFilter filter)} and
 * {@link #addListener(AgentStatusListener listener, Collection agents, Collection channels)}
 * method descriptions for explanation of how filters, as well as explicit lists of agents and
 * channels, are used to determine what events are delivered to each listener.
 * <p>
 * The aggregator modifies its state (in response to status messages received on the buses)
 * and notifies listeners on a single dedicated thread. No new messages will be processed,
 * and the aggregator state will not change, until all listeners process the current event.
 * Listeners that need to perform any time consuming or potentially blocking operations
 * should do that on other threads to avoid clogging the aggregator.
 * <p>
 * Methods that retrieve current status of agents and channels can be called on any thread - they
 * are guaranteed to report consistent information that reflects the aggregator state that existed
 * at some point between processing two consecutive status messages. There is no guarantee, however,
 * that the aggregator will remain in that state by the time these methods return. There is also no
 * guarantee that the state of the aggregator reflects the actual current state of an Agent,
 * due to the asynchronous nature of message delivery and delays in processing.
 *
 * @author onoprien
 */
public class AgentStatusAggregator {
    
// -- Fields : -----------------------------------------------------------------
    
    // Accessible to AgentHandle:
    
    final ThreadPoolExecutor exec; // single-threaded executor for processing status change events and notifying listeners
    final ArrayList<GlobalListenerHandle> listeners = new ArrayList<>(0); // all registered listeners
    volatile Logger logger;
    
    // Private:

    private final AgentPresenceListener agentPresenceListener;
    private final StatusMessageListener statusMessageListener;
    private final DataProviderDictionaryService.DataProviderDictionaryListener dictionaryListener;
    private final AgentLockUpdateListener lockListener;
    private final AgentLoginUpdateListener loginListener;

    private final ConcurrentHashMap<String,Instant> times = new ConcurrentHashMap<>(); // Agent name to local time of last message
    private final ConcurrentHashMap<String,AgentHandle> agentHandles = new ConcurrentHashMap<>(); // Agent name to handle (all agents present on buses are in this map)
    
    
// -- Life cycle : -------------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">

    public AgentStatusAggregator() {
        
        // Single-threaded executor that updates data structures and notifies listeners.
        
        ThreadFactory threadFactory = (run) -> {
            Thread thread = new Thread(run, "AgentStatusAggregator");
            thread.setDaemon(true);
            return thread;
        };
        exec = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.SECONDS, new LinkedBlockingQueue(), threadFactory){
            @Override
            public void execute(Runnable command) {
                try {
                    super.execute(command);
                } catch (RejectedExecutionException x) {
                    // Ignore tasks submitted when the executor is shutting down
                }
            }
            
        };
        
        // AgentPresenceListener for responding to agents disconnect notifications
        
        agentPresenceListener = new AgentPresenceListener() {
            @Override
            public void disconnected(AgentInfo... agents) {
                exec.execute(() -> {
                    for (AgentInfo agent : agents) {
                        if (!ignoreAgent(agent)) {
                            processDisconnect(agent.getName());
                        }
                    }
                });
            }
        };
        
        // StatusMessageListener for receiving status updates
        
        statusMessageListener = message -> {
            AgentInfo info = message.getOriginAgentInfo();
            if (!ignoreAgent(info)) {
                String agentName = message.getOriginAgentInfo().getName();
                times.put(agentName, Instant.now());
                exec.execute(() -> processStatusMessage(message));
            }
        };
        
        // Dictonary listener for getting data dictionaries
        
        dictionaryListener = e -> {
            if (e.getEventType() == DataProviderDictionaryService.DataProviderDictionaryEvent.EventType.ADDED && !ignoreAgent(e.getAgentInfo())) {
                exec.execute(() -> {
                    processDictionaryEvent(e);
                });
            }
        };
        
        // Lock and login service listeners
        
        lockListener = new AgentLockUpdateListener() {
            @Override
            public void onAgentHeldLockUpdate(String agentName, AgentLock lock) {
                exec.execute(() -> {
                    AgentHandle handle = agentHandles.get(agentName);
                    if (handle != null) {
                        handle.onLockEvent(lock);
                    }
                });
            }
            @Override
            public void onGlobalLockUpdate(String agentName, String owner, AgentLock lock) {
                if (lock instanceof AgentLockInfo) {
                    switch (((AgentLockInfo) lock).getStatus()) {
                        case RELEASED:
                        case REJECTED:
                            lock = null;
                            break;
                        case REQUESTED:
                            return;
                    }
                }
                AgentLock theLock = lock;
                exec.execute(() -> {
                    AgentHandle handle = agentHandles.get(agentName);
                    if (handle != null) {
                        handle.onLockEvent(theLock);
                    }
                });
            }
        };
        
        loginListener = new AgentLoginUpdateListener() {
            @Override
            public void onAgentLoginUpdate(String oldUserId, String newUserId) {
                if (!Objects.equals(oldUserId, newUserId)) {
                    exec.execute(() -> {
                        agentHandles.values().forEach(handle -> {
                            handle.onLoginEvent(newUserId);
                        });
                    });
                }
            }
        };
        
    }
    
    /**
     * Initializes this status aggregator.
     */
    public void initialize() {
        Console agent = Console.getConsole();
        logger = agent.getLogger();
        AgentMessagingLayer messagingAccess = agent.getMessagingAccess();
        messagingAccess.addStatusMessageListener(statusMessageListener);
        messagingAccess.getAgentPresenceManager().addAgentPresenceListener(agentPresenceListener);
        agent.getAgentService(DataProviderDictionaryService.class).addDataProviderDictionaryListener(dictionaryListener);
        agent.getAgentService(AgentLockService.class).addAgentLockUpdateListener(lockListener);
        agent.getAgentService(AgentLoginService.class).addAgentLoginUpdateListener(loginListener);
    }
    
    /**
     * Shuts down this status aggregator.
     * Once shut down, the aggregator cannot be restarted (this restriction can be removed in future versions if necessary).
     */
    public void shutdown() {
        exec.shutdownNow();
    }

// </editor-fold>    
    
// -- Public getters : ---------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    
    /**
     * Lists currently connected agents.
     * @return List of connected agents.
     */
    public List<AgentInfo> getAgents() {
        return agentHandles.values().stream().map(a -> a.getAgent()).collect(Collectors.toList());
    }
    
    /**
     * Return {@code AgentInfo} instance for the specified {@link Agent}.
     * 
     * @param agentName Name of the localAgent.
     * @return {@code AgentInfo} instance, or {@code null} is no localAgent with the specified name is connected to the buses.
     */
    public AgentInfo getAgent(String agentName) {
        AgentHandle data = agentHandles.get(agentName);
        return  data == null ? null : data.getAgent();
    }

    /**
     * Returns {@code true} if the specified localAgent is present on the buses.
     * 
     * @param agentName Agent name.
     * @return {@code true} if the localAgent is connected.
     */
    public boolean hasAgent(String agentName) {
        return agentHandles.get(agentName) != null;
    }
    
    /**
     * Returns a list of channels identified by the specified selectors.
     * <p>
     * This method returns all
     * matching channels found in data dictionaries of remote agents, regardless of whether there
     * are any listeners watching those channels. Unwatched channels can only be used to retrieve
     * their paths and static attributes. Current values and metadata may be not set or outdated.
     * 
     * @param agents Subsystem names.
     * @param channelSelectors Channel selectors.
     * @return List of matching channels.
     */
    public List<AgentChannel> getDictionaryChannels(List<String> agents, List<String> channelSelectors) {
        List<ChannelSelector> selectors = ChannelSelector.compile(channelSelectors);
        List<AgentChannel> out = new ArrayList<>();
        for (AgentHandle agentHandle : agentHandles.values()) {
            List<AgentChannel> channels = agentHandle.getDictionaryChannels();
            ChannelSelector.filter(agents, selectors, channels, out);
        }
        return out;
    }
    
    /**
     * Returns the channel specified by the given path.
     * 
     * @param path Data channel path.
     * @return Requested channel, or {@code null} if the channel is not being watched by this aggregator.
     */
    public AgentChannel getChannel(String path) {
        int i = path.indexOf("/");
        if (i == -1) return null;
        String agentName = path.substring(0, i);
        AgentHandle agentHandle = agentHandles.get(agentName);
        return agentHandle == null ? null : agentHandle.getChannel(path);
    }
    
    /**
     * Returns the current state of the specified {@link Agent}.
     * 
     * @param agentName Name of the localAgent.
     * @return State of the specified localAgent.
     */
    public StateBundle getAgentState(String agentName) {
        AgentHandle agentHandle = agentHandles.get(agentName);
        return agentHandle == null ? null : agentHandle.getState();
    }
    
// </editor-fold>    

// -- Handling watched channels and listeners : --------------------------------
    
// <editor-fold defaultstate="collapsed">

    /**
     * Adds a listener to be notified of changes in agents and channels specified by the filter.
     * <p>
     * The listener is added with the lists of agents and channels returned by the filter's
     * {@code getAgents()} and {@code getOriginChannels()} methods, respectively. No reference
     * to the filter is kept, no filter methods are called when processing messages.
     * 
     * @param listener Listener to be added.
     * @param filter Channels filter.
     * @throws NullPointerException if either argument is {@code null}.
     */
    public void addListener(AgentStatusListener listener, AgentChannelsFilter filter) {
        addListener(listener, filter.getAgents(), filter.getOriginChannels());
    }

    /**
     * Adds a listener to be notified of changes in specified agents and data
     * channels. Note that calling this method with both {@code agents} and
     * {@code channels} arguments equal to {@code null} forces this aggregator
     * to watch all channels from all agents.
     * <p>
     * The list of channels passed to this method may contain:<ul>
     * <li>Complete channel paths.
     * <li>Path templates in the "{@code [Agent]/[partial path][/]}"
     * format. Templates that omit the Agent name are expanded against all
     * agents whose names are in the {@code agents} list (or all agents present
     * on the buses if that list is {@code null}). Templates that end with "/"
     * match all channels that start with "{@code Agent/partial path/}".
     * <li>Selector strings in the
     * "{@code [agent.name=value&][agent.type=value&][agent.key[=value]&…&agent.key[=value]&][key[=value]&…&key[=value]]}"
     * format, where {@code key} and {@code value} are names and values of
     * Agent properties or static channel attributes. If {@code value} is
     * omitted, the existence of the attribute is checked. Note that
     * templates and selectors are only applied to channels and their attributes as
     * listed in the agent channels dictionary. Any channels not appearing
     * in the dictionary are ignored, as are attribute changes.</ul>
     * <p>
     * The listener is notified of agent connection, configuration, and
     * disconnection events if the agent name is in the supplied list of
     * channels, or the list is {@code null}, or the Agent name is
     * explicitly contained in any of the templates and selectors.
     * <p>
     * The listener is notified of value changes in data channels if:<ul>
     * <li>The channel path is contained in the supplied list of channels.
     * <li>The channel path matches any of the supplied templates.
     * <li>The channel satisfies any of the supplied selectors.</ul>
     * <p>
     * Unless the agent list is {@code null}, templates and selectors that do not explicitly
     * specify agent names are only applied to agents whose names are on the Agent list.
     *
     * @param listener Listener to be added.
     * @param agents Names of agents the listener is interested in.
     *               If {@code null}, the listener will be notified of changes in all agents.
     * @param channels Paths, templates, and selectors specifying channels the listener is interested in.
     *                 If {@code null}, the listener is notified of changes in all channels from the
     *                 specified agents. Supplying an empty list ensures that the listener is only
     *                 notified of agent connections and disconnections.
     * @throws NullPointerException if the {@code listener} argument is {@code null}.
     */
    public void addListener(AgentStatusListener listener, Collection<String> agents, Collection<String> channels) {
        List<String> finalAgents = agents == null ? null : new ArrayList<>(agents);
        List<String> finalChannels = channels == null ? null : new ArrayList<>(channels);
        exec.submit(() -> executeAddListener(listener, finalAgents, finalChannels));
    }

    /**
     * Removes the specified listener.
     * Also stops watching channels that were only watched because of this listener.
     * 
     * @param listener Listener to be removed.
     */
    public void removeListener(AgentStatusListener listener) {
       exec.submit(() -> executeRemoveListener(listener));
    }
    
    /** Runs on aggregator thread. */
    private void executeAddListener(AgentStatusListener listener, List<String> agents, List<String> channels) {
        GlobalListenerHandle globalHandle = new GlobalListenerHandle(listener, agents, channels);
        listeners.add(globalHandle);
        MutableAgentStatusEvent event = new MutableAgentStatusEvent(this);
        for (AgentHandle agentHandle : agentHandles.values()) {
            agentHandle.addListener(globalHandle, event); // creates local listener handle and adds channels to the event
        }
        if (!event.isEmpty()) {
            event.validate();
            try {
                listener.configure(event);
            } catch (RuntimeException x) {
                Console.getConsole().getLogger().warn("Error processing status aggregator event by " + listener.getClass(), x);
            }
        }
    }
    
    /** Runs on aggregator thread. */
    private void executeRemoveListener(AgentStatusListener listener) {
        Set<GlobalListenerHandle> removed = Collections.newSetFromMap(new IdentityHashMap<>());
        Iterator<GlobalListenerHandle> it = listeners.iterator();
        while (it.hasNext()) {
            GlobalListenerHandle global = it.next();
            if (global.listener == listener) {
                it.remove();
                if (!removed.contains(global)) {
                    removed.add(global);
                    for (AgentHandle agentHandle : agentHandles.values()) {
                        agentHandle.removeListener(global);
                    }
                }
            }
        }
    }

// </editor-fold>
    
// -- Processing status messages, disconnects, and dictionary events : ---------
    
// <editor-fold defaultstate="collapsed">
    
    /** Runs on aggregator thread. */
    private void processStatusMessage(StatusMessage message) {
        
        AgentInfo agentInfo = message.getOriginAgentInfo();
        String agentName = agentInfo.getName();
        
        // Agent went offline - destroy handle and notify listeners, no other changes:
        
        if (message.getState().isInState(PhaseState.OFF_LINE)) {
            processDisconnect(agentName);
            return;
        }
        
        // Create agent handle if it does not already exist
        
        AgentHandle agentHandle = agentHandles.get(agentName);
        if (agentHandle == null) {
            agentHandle = new AgentHandle(agentInfo, this);
            agentHandles.put(agentName, agentHandle);
        }
        
        // Let AgentHandle process message
        
        agentHandle.onMessage(message);
    }
    
    /** Runs on aggregator thread. */
    private void processDisconnect(String agentName) {
        AgentHandle agentHandle = agentHandles.remove(agentName);
        if (agentHandle != null) agentHandle.onDisconnect();
    }

    /** Runs on aggregator thread. */
    private void processDictionaryEvent(DataProviderDictionaryService.DataProviderDictionaryEvent e) {
        AgentInfo agent = e.getAgentInfo();
        AgentHandle agentHandle = agentHandles.get(agent.getName());
        if (agentHandle == null) {
            agentHandle = new AgentHandle(agent, this);
            agentHandles.put(agent.getName(), agentHandle);
        }
        agentHandle.onDictionary(e.getDictionary());
    }

// </editor-fold>    
    
// -- Local methods : ----------------------------------------------------------
    
    private boolean ignoreAgent(AgentInfo agent) {
        return agent == null || agent.getName() == null || agent.getType() == AgentType.CONSOLE || agent.getType() == AgentType.LISTENER;
    }
    
// -- Auxiliary classes : ------------------------------------------------------
    
    /** Listener and associated data for use in the list of listeners maintained by aggregator. */
    final class GlobalListenerHandle {
        
        final AgentStatusListener listener;
        final List<String> agents;
        final List<ChannelSelector> channelSelectors;
        
        GlobalListenerHandle(AgentStatusListener listener, List<String> agents, List<String> channels) {
            this.listener = listener;
            this.agents = agents;
            this.channelSelectors = ChannelSelector.compile(channels);
        }
    }
    
}
