package org.lsst.ccs.gconsole.plugins.trending.timeselection;

import java.awt.Component;
import java.awt.Window;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

/**
 * Dialog for managing time window presets.
 * 
 * @author onoprien
 */
public class PresetsDialog {

// -- Private parts : ----------------------------------------------------------
    
    private final JScrollPane root;
    private final JButton editButton, deleteButton, newButton, okButton;
    private final JTable table;
    
    private final PresetList timeWindowList;
    private TimeWindow selectedTimeWindow;


// -- Construction and initialization : ----------------------------------------
    
    private PresetsDialog(PresetList data) {
        
        timeWindowList = data;
        
        editButton = new JButton("Edit...");
        editButton.addActionListener(e -> {
            TimeWindow ts = getSelectedPreset();
            ts = TimeWindowDialog.editTimeWindow(editButton, ts);
            if (ts != null) {
                timeWindowList.insert(ts);
                setSelectedPreset(ts);
            }
        });
        
        deleteButton = new JButton("Delete");
        deleteButton.addActionListener(e -> {
            TimeWindow ts = getSelectedPreset();
            if (ts != null) timeWindowList.delete(ts);
        });
        
        newButton = new JButton("New...");
        newButton.addActionListener(e -> {
            TimeWindow ts = TimeWindowDialog.editTimeWindow(newButton, null);
            if (ts != null) {
                timeWindowList.insert(ts);
                setSelectedPreset(ts);
            }
        });
        
        okButton = new JButton(" Close ");
        okButton.addActionListener(e -> {
            selectedTimeWindow = getSelectedPreset();
            Window w = SwingUtilities.getWindowAncestor(okButton);
            if (w != null) w.setVisible(false);
        });
        
        table = new JTable(timeWindowList);
        table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        table.getSelectionModel().addListSelectionListener(e -> {
            TimeWindow ts = getSelectedPreset();
            editButton.setEnabled(ts != null);
            deleteButton.setEnabled(ts != null);
        });
        if (timeWindowList.getRowCount() > 0) {
            table.getSelectionModel().setSelectionInterval(0, 0);
        }
        table.setMinimumSize(table.getPreferredSize());
        
        root = new JScrollPane(table);
    }
    
// -- Showing the dialog : -----------------------------------------------------
    
    /**
     * Displays the dialog for managing time window presets.
     * 
     * @param parent Parent component.
     * @param data List of presets known to the application.
     * @return Preset selected by the user, or {@code null} if no preset was selected or if the dialog was canceled.
     */
    static public TimeWindow managePresets(Component parent, PresetList data) {
        PresetsDialog dialog = new PresetsDialog(data);
        JOptionPane.showOptionDialog(parent, dialog.root, "Trending Time Window Presets", 
                JOptionPane.DEFAULT_OPTION, JOptionPane.PLAIN_MESSAGE, null, 
                new JButton[] {dialog.editButton, dialog.deleteButton, dialog.newButton, dialog.okButton}, dialog.okButton);
        return dialog.selectedTimeWindow;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private TimeWindow getSelectedPreset() {
        int index = table.getSelectedRow();
        return index == -1 ? null : timeWindowList.get(index);
    }
    
    private void setSelectedPreset(TimeWindow ts) {
        int index = timeWindowList.indexOf(ts);
        if (index == -1) {
            table.getSelectionModel().clearSelection();
        } else {
            table.getSelectionModel().setSelectionInterval(index, index);
        }
    }
}
