package org.lsst.ccs.gconsole.plugins.trending;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.stream.Collectors;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.gconsole.plugins.trending.TrendData.MetaValue;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindowSelector;
import org.lsst.ccs.gconsole.services.rest.LsstRestService;
import org.lsst.ccs.localdb.statusdb.server.ChannelMetaData;
import org.lsst.ccs.localdb.statusdb.server.Data;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.Datas;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.statusdb.server.TrendingData.DataValue;
import org.lsst.ccs.localdb.statusdb.server.TrendingResult;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Source of trending data based on a REST server. Provides getters for retrieving the list
 * of known trending channels and the data for specified channels.
 * <p>
 * This sources watches for external events that can change the list of known channels,
 * and notifies listeners on EDT when this happens.
 * <p>
 * <i>Current implementation note. This class maintains a map of paths to {@code DataChannel},
 * but only the list of paths is externally available. Metadata provided by {@code DataChannel}
 * is unused. Accordingly, whenever the map is updated (due to time window or REST server
 * change), listeners are not notified unless the set of available paths changes.</i>
 *
 * @author onoprien
 */
public class RestSource {

// -- Fields : -----------------------------------------------------------------
    
    private final LsstTrendingPlugin plugin;
    private final TrendingPreferences pref;
    private final LsstRestService restService;
    private final ThreadPoolExecutor dataFetcher;
    private final Logger logger;
    
    private final Map<String,DataChannel> channels = new LinkedHashMap<>();
    private volatile int begin;
    
    private final CopyOnWriteArrayList<ChangeListener> listeners = new CopyOnWriteArrayList<>();
    
    private final TimeWindowSelector.Listener timeListener;
    private final ChangeListener restListener;
    

// -- Life cycle : -------------------------------------------------------------
    
//<editor-fold defaultstate="collapsed">
    RestSource(LsstTrendingPlugin plugin) {
        
        this.plugin = plugin;
        pref = plugin.getPreferences();
        restService = (LsstRestService) plugin.getConsole().getConsoleLookup().lookup(LsstRestService.class);
        dataFetcher = plugin.getExecutor();
        logger = plugin.getConsole().getLogger();
        
        timeListener = event -> {
            onTimeWindowChange(event.getTimeWindow());
        };
        restListener = event -> {
            onRestServerChange();
        };
    }
    
    /**  Initialization. Called on EDT. */
    void start() {
        plugin.getTimeWindowSelector().addListener(timeListener);
        restService.addListener(restListener);
        onTimeWindowChange(plugin.getSelectedTimeWindow());
    }
    
    /** Shutdown. Called on EDT. */
    void stop() {
        TimeWindowSelector ts = plugin.getTimeWindowSelector();
        if (ts != null) {
            ts.removeListener(timeListener);
        }
        restService.removeListener(restListener);
    }
//</editor-fold>

// -- Updating this source and notifying listeners : ---------------------------
    
//<editor-fold defaultstate="collapsed">
    /**
     * Refreshes the list of channels by retrieving them from the REST server.
     */
    public void refresh() {
        dataFetcher.execute(this::refreshChannelList);
    }
    
    private void onRestServerChange() {
        refresh();
    }
    
    private void onTimeWindowChange(TimeWindow timeWindow) {
        int time;
        if (timeWindow == null) {
            time = 86400;
        } else {
            time = (int) Math.max(86400, (System.currentTimeMillis() - timeWindow.getLowerEdge())/1000L + 3600L);
        }
        if (time != begin) {
            begin = time;
            refresh();
        }
    }
    
    private void refreshChannelList() {
        List<DataChannel> cc;
        try {
            DataChannel.DataChannelList channelList = restService.getChannelList(begin);
            if (channelList == null || channelList.list == null || channelList.list.isEmpty()) {
                cc = Collections.emptyList();
            } else {
                cc = channelList.list;
            }
        } catch (RuntimeException e) {
            logger.warn("Unable to retrieve the list of channels from the REST server.");
            cc = Collections.emptyList();
        }
        synchronized (channels) {
            if (!(channels.size() == cc.size())) {
                channels.clear();
                cc.forEach(c -> channels.put(String.join("/", c.getPath()), c));
                fireEvent();
            }
        }
    }
    
    /**
     * Registers a listener to be notified if the list of channels available from this trending source changes.
     * This method can be called on any thread. The listeners are notified on the EDT.
     * 
     * @param listener Listener to be added.
     */
    public void addListener(ChangeListener listener) {
        listeners.add(listener);
    }
    
    /**
     * Removes a listener.
     * This method can be called on any thread.
     * 
     * @param listener Listener to be removed.
     */
    public void removeListener(ChangeListener listener) {
        listeners.remove(listener);
    }
    
    /**
     * Removes all listeners.
     * This method can be called on any thread.
     */
    public void removeAllListeners() {
        listeners.clear();
    }
    
    /**
     * Notifies listeners of changes in this source.
     * Can be called on any thread. Listeners are notified on EDT.
     */
    protected void fireEvent() {
        SwingUtilities.invokeLater(() -> {
            ChangeEvent event = new ChangeEvent(RestSource.this);
            listeners.forEach(listener -> {
                try {
                    listener.stateChanged(event);
                } catch (RuntimeException x) {
                    if (logger != null) {
                        logger.error(listener +" threw an exception while processing RestSource event", x);
                    }
                }
            });
        });
    }
//</editor-fold>"

// -- Retrieving data : --------------------------------------------------------

    /**
     * Returns the list of known data channel paths.
     * Calling this method does not retrieve data from the REST server.
     * 
     * @return List of channel paths.
     */
    public List<String> getChannels() {
        synchronized (channels) {
            return new ArrayList<>(channels.keySet());
        }
    }
    
    /**
     * Retrieves from the REST server and processes historical data for the specified trending channel. 
     * 
     * @param path Channel path.
     * @param begin Beginning of time window.
     * @param end End of time window.
     * @param metadata Set of metadata types to be included.
     * @param history Previously known data.
     * @return Newly created data container, or {@code null} if retrieval fails for any reason.
     */
    
    /**
     * Retrieves from the REST server and processes historical data for the specified trending channel.
     * If the supplied trends contain previously retrieved data, some of that data can be re-used, reducing the
     * amount of data that needs to be retrieved from the REST server.
     * 
     * @param trends List of trends for which the data has to be retrieved.
     * @param begin Beginning of time window.
     * @param end End of time window
     * @param raw {@code TRUE} or {@code FALSE} depending on whether raw or statistical data should be retrieved;
     *            if {@code null}, current application preferences will be consulted.
     * @param bins Approximate desired number of bins in the resulting dataset;
     *             if {@code null}, current application preferences will be consulted.
     * @return List of newly created datasets, in the {@code trends} order.
     *         Note that this list will never be {@code null}, but individual elements
     *         can be {@code null} if data retrieval for corresponding trends fails for any reason.
     */
    public ArrayList<TrendData> get(List<Trend> trends, long begin, long end, Boolean raw, Integer bins) {
        
        // Check preferences if necessary and set query parameters
        
        int nBins = bins == null ? pref.getNBins() : bins;
        boolean useRawData = raw == null ? pref.isUseRawData() : raw;
        String flavor;
        boolean cache;
        if (nBins > 0) {
            flavor = useRawData ? null : "stat";
            cache = false;
        } else {
            flavor = useRawData ? "raw" : "stat";
            cache = useRawData;
        }
        
        // See if we can use existing data
        
        ArrayList<TrendData> out = new ArrayList<>(trends.size());
        if (cache) {
            long cachedBegin = 0;
            long cachedEnd = Long.MAX_VALUE;
            for (Trend trend : trends) {
                TrendData data = trend.getData();
                if (data == null || !data.isRaw()) {
                    cache = false;
                    break;
                }
                long[] dataRange = data.getTimeRange();
                cachedBegin = Math.max(cachedBegin, dataRange[0]);
                cachedEnd = Math.min(cachedEnd, data.getHighT());
            }
            if (cache && cachedBegin <= begin && cachedEnd > begin) {
                if (cachedEnd >= end) {
                    for (Trend trend : trends) {
                        out.add(cut(trend.getData(), begin, end));
                    }
                } else {
                    TrendingResult[] tr = fetch(cachedEnd, end, flavor, nBins, trends);
                    for (int i=0; i<trends.size(); i++) {
                        out.add(merge(trends.get(i).getData(), (tr == null ? null : tr[i]), begin, end, true));
                    }
                }
            }
        }
        
        // If the out list has not been filled yet, do a complete query
        
        if (out.isEmpty()) {
            TrendingResult[] tr = fetch(begin, end, flavor, nBins, trends);
            for (int i = 0; i < trends.size(); i++) {
                out.add(merge(null, (tr == null ? null : tr[i]), begin, end, nBins <= 0 && useRawData));
            }
        }
        
        return out;
    }
    
    
// Local methods and classes : -------------------------------------------------
    
    /**
     * Creates a dataset for a portion of the time range covered by the original raw data dataset.
     * This method returns {@code null} if the original dataset is {@code null}, or if the
     * original dataset does not cover the entire range from {@code begin} to {@code end}.
     * 
     * @param data Original dataset.
     * @param begin Start of the time range of the new dataset.
     * @param end End of the time range of the new dataset.
     * @return New dataset.
     */
    static TrendData cut(TrendData data, long begin, long end) {
        
        if (data == null || end <= begin) return null;
        long[] dataTimeRange = data.getTimeRange();
        if (dataTimeRange[0] > begin || dataTimeRange[1] < end) return null;

        long[] times;
        Map<String, double[]> values = new HashMap<>();
        Map<String, ArrayList<MetaValue>> meta = new HashMap<>();
        
        // data and on-point metadata
        
        long[] oldTime = data.getTimes();
        int start = Arrays.binarySearch(oldTime, begin);
        if (start < 0) {
            start = -start - 1;
        }
        int stop = Arrays.binarySearch(oldTime, end);
        if (stop < 0) {
            stop = -stop - 1;
        } else {
            stop++;
        }
        if (stop > start) {
            times = Arrays.copyOfRange(oldTime, start, stop);
            for (String key : data.getOnPointKeys()) {
                values.put(key, Arrays.copyOfRange(data.getValues(key), start, stop));
            }
        } else {
            times = new long[0];
            for (String key : data.getOnPointKeys()) {
                values.put(key, new double[0]);
            }
        }
        
        // off-point metadata
        
        Map<String, ArrayList<MetaValue>> cachedMeta = data.getOffPointMetadata();
        for (Map.Entry<String, ArrayList<MetaValue>> e : cachedMeta.entrySet()) {
            String key = e.getKey();
            ArrayList<MetaValue> newList = new ArrayList<>();
            for (MetaValue mp : e.getValue()) {
                MetaValue mv = MetaValue.trim(mp, begin, end);
                if (mv != null) {
                    newList.add(mv);
                }
            }
            if (!newList.isEmpty()) {
                meta.put(key, newList);
            }
        }
        
        // create and return new dataset
        
        if (values.isEmpty()) {
            values = Collections.emptyMap();
        } else if (values.size() == 1) {
            Map.Entry<String, double[]> e = values.entrySet().iterator().next();
            values = Collections.singletonMap(e.getKey(), e.getValue());
        }
        if (meta.isEmpty()) {
            meta = Collections.emptyMap();
        } else {
            meta.values().forEach(list -> list.trimToSize());
        }
        return new TrendData(times, values, meta, new long[] {begin, end}, true);
    }
    
    private TrendingResult[] fetch(long begin, long end, String flavor, int nBins, List<Trend> trends) {        
        String[] paths = new String[trends.size()];
        for (int i = 0; i < paths.length; i++) {
            paths[i] = trends.get(i).getDescriptor().getPath();
        }
        return restService.getData(begin, end, flavor, nBins, paths);
    }
    
    /**
     * Creates a new dataset for the specified time window using data from an existing dataset and a result of a new query.
     * Assumptions:<ul>
     * <li>If {@code cachedData} is not {@code null}, its time range contains {@code begin} but does not contain {@code end}.
     * <li>If {@code newData} is not {@code null}, it does not contain points before the end of {@code cachedData} range.
     * </ul>
     * 
     * @param cachedData Existing dataset.
     * @param newData Result of a new query.
     * @param begin Start of the time range of the new dataset.
     * @param end End of the time range of the new dataset.
     * @return New dataset.
     */
    static TrendData merge(TrendData cachedData, TrendingResult newData, long begin, long end, boolean rawUnbinned) {

        long[] times;
        Map<String, double[]> values = new HashMap<>();
        Map<String, ArrayList<MetaValue>> meta = new HashMap<>();
        
        // inspect cached data, set index to the first data point to be re-used, "oldN" to number of points to re-use
        
        int oldN = 0;
        int index = 0;
        long[] oldTime = null;
        if (cachedData != null) {
            oldTime = cachedData.getTimes();
            index = Arrays.binarySearch(oldTime, begin);
            if (index < 0) {
                index = -index -1;
            }
            oldN = oldTime.length - index;
        }
        
        // set "newN" to the number of points in the new data, "n" to the total number of points
        
        int newN = 0;
        TrendingData[] dataArray = null;
        if (newData != null) {
            dataArray = newData.getTrendingDataArray();
            if (dataArray != null) {
                newN = dataArray.length;
            }
        }       
        int n = oldN + newN;
        
        // copy cached on-point data to new arrays, set "index" to next index
        
        if (oldN == 0) {
            times = new long[n];
        } else {
            times = Arrays.copyOfRange(oldTime, index, index + n);
            for (String key : cachedData.getOnPointKeys()) {
                values.put(key, Arrays.copyOfRange(cachedData.getValues(key), index, index + n));
            }
        }
        index = oldN;
        
        // add points from new query  // FIXME: if this was used for raw data only, I could stop looking up by key
        
        if (newN > 0) {
            for (TrendingData td : dataArray) {
                for (DataValue dv : td.getDatavalue()) {
                    String key = dv.getName();
                    double[] vv = values.get(key);
                    if (vv == null) {
                        vv = new double[n];
                        values.put(key, vv);
                    }
                    vv[index] = dv.getValue();
                }
                times[index++] = td.getAxisvalue().getValue();
            }
        }
        
        // fill metadata lists from cached data
        
        if (cachedData != null) {
            Map<String, ArrayList<MetaValue>> cachedMeta = cachedData.getOffPointMetadata();
            for (Map.Entry<String, ArrayList<MetaValue>> e : cachedMeta.entrySet()) {
                String key = e.getKey();
                ArrayList<MetaValue> newList = new ArrayList<>();
                for (MetaValue mp : e.getValue()) {
                    MetaValue mv = MetaValue.trim(mp, begin, end);
                    if (mv != null) newList.add(mv);
                }
                if (!newList.isEmpty()) {
                    meta.put(key, newList);
                }
            }
        }
        
        // add metadata from new query
        
        if (newData != null) {
            List<ChannelMetaData> metaList = newData.getChannelMetadata();
            if (metaList != null) {
                for (ChannelMetaData md : metaList) {
                    String key = md.getName();
                    ArrayList<MetaValue> mvList = meta.get(key);
                    if (mvList == null) {
                        MetaValue mv = MetaValue.trim(md, begin, end);
                        if (mv != null) {
                            mvList = new ArrayList<>();
                            mvList.add(mv);
                            meta.put(key, mvList);
                        }
                    } else {
                        MetaValue last = mvList.get(mvList.size() - 1);
                        MetaValue mv = MetaValue.trim(md, last.getStop(), end);
                        if (mv != null) {
                            MetaValue merge = MetaValue.merge(last, mv);
                            if (merge == null) {
                                mvList.add(mv);
                            } else {
                                mvList.set(mvList.size() - 1, merge);
                            }
                        }
                        

                    }
                }
            }
        }
        
        // create and return new dataset
        
        if (values.isEmpty()) {
            values = Collections.emptyMap();
        } else if (values.size() == 1) {
            Map.Entry<String, double[]> e = values.entrySet().iterator().next();
            values = Collections.singletonMap(e.getKey(), e.getValue());
        }
        if (meta.isEmpty()) {
            meta = Collections.emptyMap();
        } else {
            meta.values().forEach(list -> list.trimToSize());
        }
        TrendData trd =  new TrendData(times, values, meta, new long[] {begin, end}, rawUnbinned);
//
//        Diagnostics for LSSTCCS-2253:
//
//        printTR(newData, begin, end);
//        printTD(trd);
//        
        return trd;
    }
    
    
// -- Diagnostics : ------------------------------------------------------------
    
//    static void printTR(TrendingResult tr, long begin, long end) {
//        System.out.println("Result "+ begin +" "+ end);
//        List<ChannelMetaData> mm = tr.getChannelMetadata();
//        long prev = begin;
//        for (ChannelMetaData m : mm) {
//            if (m.getName().equals("alarmLow")) {
//                System.out.println((m.getTstart() - prev) +" "+ (m.getTstop() - m.getTstart()) +" "+ m.getValue());
//                prev = m.getTstop();
//            }
//        }
//        System.out.println(end - prev);
//    }
//    
//    static void printTD(TrendData d) {
//        System.out.println("Data " + d.getTimeRange()[0] + " " + d.getTimeRange()[1]);
//        long[] tt = d.getTimes("alarmLow");
//        String[] ss = d.getMetadata("alarmLow");
//        if (tt != null) {
//            long prev = d.getTimeRange()[0];
//            for (int i = 0; i < tt.length; i += 2) {
//                System.out.println((tt[i] - prev) + " " + (tt[i + 1] - tt[i]) + " " + ss[i] + " " + ss[i + 1]);
//                prev = tt[i + 1];
//            }
//            System.out.println(d.getTimeRange()[1] - prev);
//        }
//    }
        
}
