package org.lsst.ccs.gconsole.plugins.tracer.filters;

import java.awt.Color;
import java.util.Properties;
import java.util.logging.Level;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusAgentInfo;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusCommandDictionary;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusLock;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.plugins.tracer.FilteredMessage;
import org.lsst.ccs.gconsole.plugins.tracer.MessageFilter;
import org.lsst.ccs.gconsole.plugins.tracer.Tracer;

/**
 * Standard formatter for message viewers.
 *
 * @author onoprien
 */
public class StandardFormatter implements MessageFilter {

// -- Fields : -----------------------------------------------------------------
    
    static public final Color COLOR_ERROR = new Color(200, 0, 0);
    static public final Color COLOR_WARNING = new Color(0, 0, 255);
    static public final Color COLOR_GOOD = new Color(0, 200, 0);
    static public final Color COLOR_FINE = new Color(100, 100, 100);
    
    private final boolean brief;
    private final boolean includeAgent;

// -- Life cycle : -------------------------------------------------------------
    
    public StandardFormatter(boolean brief, boolean includeSubsystemName) {
        this.brief = brief;
        includeAgent = includeSubsystemName;
    }
    
// -- Filtering : --------------------------------------------------------------

    @Override
    public FilteredMessage apply(FilteredMessage filteredMessage) {
        
        BusMessage<?,?> bm = filteredMessage.getMessage();
        StringBuilder sb = new StringBuilder();
        
        sb.append(Const.DEFAULT_T_FORMAT.format(bm.getCCSTimeStamp().getUTCInstant())).append(" ");
        
        if (includeAgent) {
            sb.append("[").append(bm.getOriginAgentInfo().getName()).append("] ");
        }

        if (bm instanceof LogMessage) { // LOG bus
            
            LogMessage lm = (LogMessage)bm;
            String s = lm.getFormattedDetails().trim();
            if (s.isEmpty() || s.contains("java.util.logging.LogRecord@")) return null;  // FIXME (ask Max to fix)
            if (brief) {
                appendShortenned(s, sb, 200);
            } else {
                sb.append(s);
            }
            
            int level = StandardConfigurableFilter.parseLevel(lm.getLevel()).intValue();
            if (level >= Level.SEVERE.intValue()) {
                filteredMessage.setColor(COLOR_ERROR);
            } else if (level >= Level.WARNING.intValue()) {
                filteredMessage.setColor(COLOR_WARNING);
            } else if (level <= Level.FINE.intValue()) {
                filteredMessage.setColor(COLOR_FINE);
            }
            
        } else if (bm instanceof StatusMessage) { // STATUS bus
            
            StatusMessage sm = (StatusMessage)bm;
            
            if (sm instanceof StatusHeartBeat) {
                
                sb.append("Heart Beat.");
                
            } else if (sm instanceof StatusSubsystemData) {
                
                String s;
                try {
                    s = sm.getEncodedData().toString();
                } catch (Exception x) {
                    s = sm.toString();
                }
                if (brief) {
                    appendShortenned(s, sb, 200);
                } else {
                    sb.append(s);
                }
                                
                
            } else if (sm instanceof StatusStateChangeNotification) {
                
                StatusStateChangeNotification m = (StatusStateChangeNotification) sm;
                StateBundle change = m.getNewState().diffState(m.getOldState());
                sb.append("State change: ").append(change);
                
            } else if (sm instanceof StatusRaisedAlert) {
                
                StatusRaisedAlert m = (StatusRaisedAlert) sm;
                AlertState level = m.getAlertState();
                switch (level) {
                    case ALARM: filteredMessage.setColor(COLOR_ERROR); break;
                    case WARNING: filteredMessage.setColor(COLOR_WARNING); break;
                    case NOMINAL: filteredMessage.setColor(COLOR_GOOD); break;
                }
                sb.append(level).append(" ");
                Alert alarm = m.getRaisedAlert();
                sb.append(alarm.getAlertId()).append(" ").append(alarm.getDescription()).append("\n");
                sb.append("Cause: ").append(m.getCause());
                
            } else if (sm instanceof StatusClearedAlert) {
                
                StatusClearedAlert m = (StatusClearedAlert) sm;
                sb.append("Cleared alerts: ");
                for (String id : m.getClearAlertIds()) {
                    sb.append(id).append(", ");
                }
                sb.delete(sb.length()-2, sb.length());
                filteredMessage.setColor(COLOR_GOOD);
                
            } else if (sm instanceof StatusDataProviderDictionary) {
                
                sb.append("Data dictionary received");
                if (brief) {
                    sb.append(".");
                } else {
                    StatusDataProviderDictionary m = (StatusDataProviderDictionary) bm;
                    sb.append(", ").append(m.getDataProviderDictionary().getDataProviderInfos().size()).append(" channels.");
                }
                filteredMessage.setColor(COLOR_FINE);
                
            } else if (sm instanceof StatusCommandDictionary) {
                
                sb.append("Command dictionary received.");
                
            } else if (sm instanceof StatusConfigurationInfo) {
                
                ConfigurationInfo ci = ((StatusConfigurationInfo)sm).getConfigurationInfo();
                sb.append("Configuration: ").append(ci.getConfigurationDescription());
                filteredMessage.setColor(COLOR_FINE);
                
            } else if (sm instanceof StatusLock) {
                
                StatusLock m = (StatusLock) sm;
                sb.append("Agent lock, level ").append(m.getLevel());
                
            } else if (sm instanceof StatusAgentInfo) {
                
                sb.append("AgentInfo message");
                if (brief) {
                    sb.append(".");
                } else {
                    AgentInfo info = ((StatusAgentInfo)sm).getAgentInfo();
                    sb.append(", type ").append(info.getType()).append(", properties:\n");
                    Properties pp = info.getAgentProperties();
                    for (String key : pp.stringPropertyNames()) {
                        sb.append("  ").append(key).append("=").append(pp.getProperty(key)).append("\n");
                    }
                    sb.delete(sb.length()-1, sb.length());
                }
                
            } else {
                
                String s;
                try {
                    s = sm.getEncodedData().toString();
                } catch (Exception x) {
                    s = sm.toString();
                }
                if (brief) {
                    appendShortenned(s, sb, 200);
                } else {
                    sb.append(s);
                }
            }
            
        } else { // COMMAND bus

            if (bm instanceof CommandRequest) {
                CommandRequest cr = (CommandRequest) bm;
                sb.append("Command ").append(cr.getBasicCommand().getCommand()).append(" to ").append(cr.getDestination()).append(".");
            } else if (bm instanceof CommandNack) {
                CommandNack cn = (CommandNack) bm;
                sb.append("Command not accepted. Reason: ").append(cn.getReason());
                filteredMessage.setColor(COLOR_WARNING);
            } else if (bm instanceof CommandAck) {
                sb.append("Command accepted.");
            } else if (bm instanceof CommandResult) {
                CommandResult cr = (CommandResult) bm;
                if (cr.wasSuccessful()) {
                    sb.append("Command executed");
                    Object out = cr.getResult();
                    if (out == null) {
                        sb.append(".");
                    } else {
                        sb.append(": ").append(out);
                    }
                } else {
                    sb.append("Command failed: ").append(cr.getResult());
                    filteredMessage.setColor(COLOR_ERROR);
                }
            } else {
                sb.append(bm);
            }

        }

        filteredMessage.setText(sb.toString());
        return filteredMessage;
    }
    
// -- Static utility methods : -------------------------------------------------
    
    static private void appendShortenned(String s, StringBuilder sb, int limit) {
        if (s.length() < limit) {
            sb.append(s);
        } else {
            sb.append(s, 0, limit).append("...");
        }
    }
    
// -- Creators : ---------------------------------------------------------------
    
    @Create(category = Tracer.CATEGORY,
            name = "Brief Format",
            path = "Built-In/Formatters/Brief",
            description = "Accepts all messages, applies brief version of default formatting.")
    static public Tracer brief(
            @Par(def = "false", desc = "Include the source subsystem name into the displayed message.") boolean includeSubsystemName)
    {
        Tracer tracer = new Tracer();
        tracer.setFilter(new StandardFormatter(true, includeSubsystemName));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "Long Format",
            path = "Built-In/Formatters/Long",
            description = "Accepts all messages, applies long version of default formatting.")
    static public Tracer longf(
            @Par(def = "false", desc = "Include the source subsystem name into the displayed message.") boolean includeSubsystemName)
    {
        Tracer tracer = new Tracer();
        tracer.setFilter(new StandardFormatter(false, includeSubsystemName));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "Default Format",
            path = "Built-In/Formatters/Default With Subsystem",
            description = "Accepts all messages, applies default formatting, includes the source subsystem name.")
    static public Tracer defWithName() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new StandardFormatter(true, true));
        return tracer;
    }
    
    @Create(category = Tracer.CATEGORY,
            name = "Default Format",
            path = "Built-In/Formatters/Default No Subsystem",
            description = "Accepts all messages, applies default formatting, no source subsystem name.")
    static public Tracer defNoName() {
        Tracer tracer = new Tracer();
        tracer.setFilter(new StandardFormatter(true, false));
        return tracer;
    }
    
}
